/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * vector.C
 *
 */

/* $Id: vector.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include "header.h"

//
// The init() function is the main initialization routine
// that allocates memory and does other setup for the vector class.
// All constructors call init() in reality.
//
void
vector::init(int len,Basis *b)
{
  n = len;
  basis = b;
  // Null constructor has len==0, so make the data point to nothing
  if (len == 0)
    c = NULL;
  // Otherwise, alloate memory!
  else
    c = (scalar *)mymalloc(sizeof(scalar)*len,"c","vector.init");
}

// Frees up memory used by vector
void
vector::freemem(void)
{
  myfree(c);
  c = NULL;
}

// Null constructor:  the basic
vector::vector()
{
  init(0,NULL);
}

// Constructor with given size
vector::vector(int len)
{
  init(len,NULL);
}

// Constructor with given vector length and basis pointer
vector::vector(int len,Basis *b)
{
  init(len,b);
}

// Copy constructor
vector::vector(const vector &v1)
{
  // make the object the same size as v1 and get memory for it
  init(v1.n,v1.basis);

  // Copy v1's contents over
  int i;
  for (i=0; i < v1.n; i++)
    c[i] = v1.c[i];
}

// Destructor
vector::~vector()
{
  freemem();
}

/* Assignment:  nonstandard in that it returns void.  To make it standard,
 * replace void -> vector and uncomment the return *this; */
void vector::operator=(const vector &v1)
{
  int i;

  /* The sizes must agree */
  if (n != v1.n)
    die("in vector::operator=, n != v1.n\n");
  for (i=0; i < v1.n; i++)
    c[i] = v1.c[i];
  basis = v1.basis;
  /* return *this; */
}

/* Adding vectors */
vector vector::operator+(const vector &v1)
{
  int i;
  vector tv(v1);

  if (v1.n != n)
    die("In vector::operator+, n != v1.n\n");
  for (i=0; i < v1.n; i++)
    tv.c[i] += c[i];

  return tv;
}

/* Accumulated sum of vectors */
void vector::operator+=(const vector &v1)
{
  int i;

  if (v1.n != n)
    die("In vector::operator+=, n != v1.n\n");
  for (i=0; i < v1.n; i++)
    c[i] += v1.c[i];
}

/* Subtracting vectors */
vector vector::operator-(const vector &v1)
{
  int i;
  vector tv(v1);

  if (v1.n != n)
    die("In vector::operator-, n != v1.n\n");
  for (i=0; i < v1.n; i++)
    tv.c[i] = c[i]-v1.c[i];

  return tv;
}

/* Accumulated substraction of vectors */
void vector::operator-=(const vector &v1)
{
  int i;

  if (v1.n != n)
    die("In vector::operator-=, n != v1.n\n");
  for (i=0; i < v1.n; i++)
    c[i] -= v1.c[i];
}

/* Scaling vector :  v*s  */
vector vector::operator*(scalar s)
{
  int i;
  vector sv(n);

  for (i=0; i < n; i++)
    sv.c[i] = s*c[i];
  sv.basis = basis;
  return sv;
}

/* Scaling vector :  s*v */
vector operator*(scalar s,const vector &v)
{
  int i;
  vector sv(v);

  for (i=0; i < v.n; i++)
    sv.c[i] *= s;
  return sv;
}

/* Scaling vector inplace */
void vector::operator*=(scalar s)
{
  int i;
  for (i=0; i < n; i++)
    c[i] *= s;
}

/* Vector dotproduct */
scalar vector::operator^(const vector &v1)
{
  scalar dot;
  int i;

  if (v1.n != n)
    die("In vector::operator^, n != v1.n\n");
  dot = (scalar)0.0;
  for (i=0; i < v1.n; i++)
#if defined SCALAR_IS_COMPLEX
    dot += conjugate(c[i])*v1.c[i];
#elif defined SCALAR_IS_REAL
    dot += c[i]*v1.c[i];
#else
#error scalar is neither real nor complex!
#endif

  return dot;
}

/* Write vector in binary form to the file fname */
void
vector::write(char *fname)
{
  FILE *fp;

  fp = dft_fopen(fname,"w");
  dft_fwrite(c,sizeof(scalar),n,fp);
  dft_fclose(fp);
}


void
vector::write(FILE *fp)
{
  dft_fwrite(c, sizeof(scalar), n, fp);
}


/* read vector in binary form from the file fname */
void
vector::read(char *fname)
{
  FILE *fp;
  fp = dft_fopen(fname,"r");
  dft_fread(c,sizeof(scalar),n,fp);
  dft_fclose(fp);
}

  
void
vector::read(FILE *fp)
{
  dft_fread(c, sizeof(scalar), n, fp);
}


/* Write real part of vector in binary form to the file fname */

void
vector::write_real(char *fname)
{
  FILE *fp;
  real *data;
  int i;

  fp = dft_fopen(fname,"w");
  data = (real *)mymalloc(sizeof(real)*n,"data","write_real");
  for (i=0; i < n; i++)
    data[i] = REAL(c[i]);
  dft_fwrite(data,sizeof(real),n,fp);
  dft_fclose(fp);
  
  myfree(data);
}

void
vector::write_real(FILE *fp)
{
  real *data;
  data = (real *)mymalloc(sizeof(real)*n,"data","write_real");
  for (int i=0; i < n; i++)
    data[i] = REAL(c[i]);
  dft_fwrite(data,sizeof(real),n,fp);
  myfree(data);
}


/* Zero out the vector */
void
vector::zero_out(void)
{
  int i;

  for (i=0; i < n; i++)
    c[i] = (scalar)0.0;
}

/* Euclidean square norm of a vector */
real
abs2(const vector &v)
{
  real v2;
  int i;

  v2 = (real)0.0;
  for (i=0; i < v.n; i++)
#if defined SCALAR_IS_COMPLEX
    v2 += v.c[i].x*v.c[i].x + v.c[i].y*v.c[i].y;
#elif defined SCALAR_IS_REAL
    v2 += v.c[i]*v.c[i];
#else
#error scalar is neither real nor complex!
#endif
  return v2;
}

/* Euclidean norm of a vector */
real
abs(const vector &v)
{
  return sqrt(abs2(v));
}

/* Sum up the elements of a vector */
scalar
sum_vector(const vector &v)
{
  register int i;
  scalar s;

  s = 0.0;
  for (i=0; i < v.n; i++)
    s += v.c[i];
  return s;
}
  

/* return a vector that is the pointwise multiply of a and b; i.e.
 * (a.*b)_i = a_i*b_i */
vector
pointwise_mult(const vector &a,const vector &b)
{
  if (a.n != b.n)
    die("a.n != b.n in pointwise_mult\n");

  vector ab(a);
  int i;

  for (i=0; i < a.n; i++)
    ab.c[i] *= b.c[i];

  return ab;
}
