/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/************************************************************************/
/*	                                                                */
/*            Tairan Wang                       Nov. 6, 1997            */
/*	                                                                */
/*  A set of timer access routines for MPI performance profiling        */
/*									*/
/************************************************************************/

/* $Id: timer.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <string.h>
#include "header.h"
#include "parallel.h"

#ifdef DFT_PROFILING

#define ON              1
#define OFF		0
#define NUM_TIMERS	64
#define NUM_COUNTERS    64

int    timerState[NUM_TIMERS];
char   timerTitle[NUM_TIMERS][DFT_MSG_LEN];
double  lastStart[NUM_TIMERS],accTime[NUM_TIMERS];
int  num_timers;

double counterState[NUM_COUNTERS];
char counterTitle[NUM_TIMERS][DFT_MSG_LEN];
int num_counters;

//////////////////////////////////////////////////////////////////

/*  To turn on a timer.  Returns the prior state of the timer.	*/

int
timerOn(int timer)
{
  int priorState = timerState[timer];

  if (priorState == OFF)
    {
      timerState[timer] = ON;
      lastStart[timer] = System::get_time();
    }
  else
    {
      dft_log("\nWARNING: timer %d already on at:%f",
	      timer,lastStart[timer]);
    }
  return(priorState);
}

/*  To turn a timer off.  Again, returns the prior state.	*/

int
timerOff(int timer)
{
  int priorState = timerState[timer];

  if (priorState == ON)
    {
      timerState[timer] = OFF;
      accTime[timer] += System::get_time() - lastStart[timer];
    }
  return(priorState);
}

/*  To set the timer to a given state.	Returns prior state.	*/

int
timerSetState(int timer, int state)
{
  if (state == ON)
    return timerOn(timer);
  else
    return timerOff(timer);
}

/*  To initialize all timers.					*/
void
timerInitAll(void)
{
  num_timers = 0;
  for (int i = 0; i < NUM_TIMERS; i++)
    timerInit(i);
}

/*  To initialize a timer.					*/
void
timerInit(int timer, char* title)
{
  timerState[timer] = OFF;
  accTime[timer]    = 0;
  strcpy(timerTitle[timer], title);
  if ((timer+1) > num_timers)
    num_timers = timer+1;
}

void
timerInit(int timer)
{
  timerState[timer] = OFF;
  accTime[timer]    = 0;
}


/*  To reset a timer, possibly running.				*/
int
timerReset(int timer)
{
  int priorState = timerOff(timer);

  timerInit(timer);
  timerSetState(timer,priorState);
  return(priorState);
}


/*  To read a timer, regardless of state.			*/
double
timerRead(int timer)
{
  double timeTemp;

  timeTemp = accTime[timer];
  if (timerState[timer] == ON)
    timeTemp += System::get_time() - lastStart[timer];
  return(timeTemp);
}


/*  To get the title of a timer */
char *
timerGetTitle(int timer)
{
  return (timerTitle[timer]);
}


/*  To get the number of initialized timers */
int
timerGetTotal(void)
{
  return num_timers;
}


/* 
 * Convenience routines to turn on all the timers 
 */
int
timerActivateAll(void)
{
  timerInitAll();
  timerInit(0, "Initializations");
  timerInit(1, "Total computation time");
  timerInit(2, "Electron runs");
  timerInit(3, "Y1dag Y2  operation");
  timerInit(4, "YM = Y * M  operation");
  timerInit(5, "YM += Y * M operation");
  timerInit(6, "Column bundle transpose");
  timerInit(7, "diaginner time"); 
  timerInit(8, "memory allocation time");
  timerInit(9, "do_linmin time");
  timerInit(10, "calc_elecgrad_... ");
  timerInit(11, "calc_UCn_d_elec_dependent_e");
  timerInit(12, "Y1dag Y2, MPI_Allreduce");
  timerInit(13, "Other MPI_Allreduce");
  timerInit(14, "\tcalc_U");
  timerInit(15, "\tcalc_C");
  timerInit(16, "\tcalc_n");
  timerInit(17, "\tsolve_poisson");
  timerInit(18, "\tcalc_KE");
  timerInit(19, "\tcalc_Eloc");
  timerInit(20, "\tcalc_Enl");
  timerInit(21, "\tcalc_EH");
  timerInit(22, "\tcalc_Exc");
  timerInit(23, "\tcalc_Ecore");
  timerInit(24, "\tcalc_Eewald");
  timerInit(25, "\tcalc_Vscloc");
  timerInit(26, "\tapply_Hsp");
  timerInit(27, "\tdiagonalize_herm");
  timerInit(28, "calc_UCn_d_... in do_linmin");
  timerInit(29, "\tO");
  timerInit(30, "\tPbar");
  timerInit(31, "\tL");
  timerInit(32, "\tinvL");
  timerInit(33, "\tprecond");
  timerInit(34, "\tI");
  timerInit(35, "\tJ");
  timerInit(36, "\tIdag");
  timerInit(37, "\tJdag");
  
  return 37;
}

/* 
 * Report the timer content
 */
int
timerReport(int iter)
{
  int itimer, n_timers = timerGetTotal();
  if (n_timers > 0)
      dft_log("\nTiming Reports at iteration %d:\n",iter);
  for (itimer = 0; itimer < n_timers; itimer++) 
    {
      dft_log("\t%4d %s :\t\t%f sec\n",
	      itimer,
	      timerGetTitle(itimer), 
	      timerRead(itimer));
    }
  return 0;
}

/*
 * Report the timer content and counter content:
 */
int
timer_counter_Report();
{

  dft_log("\n");

  // Timing Report:
  int n_timers = timerGetTotal(); 
  if (n_timers > 0)
    dft_log("\n>Timing Reports:\n");
  for (int itimer = 0; itimer < n_timers; itimer++) 
    {
      dft_log(">Timing\t%4d %s :\t\t%f sec\n", 
	      itimer,
	      timerGetTitle(itimer), 
	      timerRead(itimer));
    }
  dft_log("\n\n");
  // Counter Report:
  dft_log("\n>Counter Reports:\n");
  int n_counters = counterGetTotal(); 
  for (int icounter = 0; icounter < n_counters; icounter++)
    {
      dft_log(logfile,">Counter\t%4d %s :\t\t%f\n",
	      icounter,
	      counterGetTitle(icounter),
	      counterRead(icounter));
    }
  dft_log("\n\n");
  dft_log_flush();
  return 1;
}

/////////////////////////////////////////////////////////////////

void
counterInitAll(void)
{
  num_counters = 0;
  for (int i=0; i < NUM_COUNTERS; i++)
    counterState[i] = 0.0;
}

void
counterInit(int counter, char* title)
{
  counterState[counter] = 0.0;
  strcpy(counterTitle[counter], title);
  if ((counter+1) > num_counters) {
    num_counters = counter + 1;
  }
}

void
counterIncr(int counter)
{
  counterState[counter] += 1.0;
}

void
counterIncr(int counter, double amount)
{
  counterState[counter] += amount;
}

double
counterRead(int counter)
{
  return counterState[counter];
}

char *
counterGetTitle(int counter)
{
  return (counterTitle[counter]);
}

int
counterGetTotal(void)
{
  return num_counters;
}

#endif // DFT_PROFILING
