/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,   Jan. 1997
 *
 * Signal handling routines...mostly to be used to dynamically
 * change the state of affairs during a CG minimization of the electronic
 * degrees of freedom.
 *
 * The basic idea is to
 *  (1) reroute SIGQUIT and SIGINT so that before quitting they dump
 *      the wave-functions.
 *  (2) have the user SIGUSR1 make the program read the file "sigs.in"
 *      and do what it is asked to do (dump data, reset CG, change
 *      to diagonal subspace, etc.)
 *  (3) have SIGUSR2 reset CG.
 *
 * See the comments of sig_hanlder() for details.
 *
 */

/* $Id: signals.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include "math.h"
#include "stdio.h"
#include "signal.h"
#include "unistd.h"
#include "time.h"

/* My header file */
#include "header.h"

/* Global flags for signal processing */
int resetcg_signal = 0;
int subspace_diag_signal = 0;
int copy_C_to_Y_signal = 0;

/* Pointers in this file for the main programs density and wave-functions */
static Elecinfo *elecinfo_ptr;
static Elecvars *elecvars_ptr;

/*
 * Signal handling procedure:
 *
 * SIGHUP == ignored
 * 
 * SIGQUIT, SIGINT, SIGTERM == Dump wavefunctions/density and die
 *
 * SIGUSR1 == Read flags in "sigs.in" and do the required action:
 *            the "sigs.in" in file is read one line at a time, with
 *            only the first integer on the line being scanned for (
 *            the rest of the line, up to 80 characters, is ignored).
 *            The order of the flags (1==active, else==inactive) is:
 *                  (1) die_flag
 *                  (2) dump_flag
 *                  (3) resetcg_flag
 *                  (4) subspace_diag_flag
 *                  (5) copy_C_to_Y_signal
 *            subspace_diag_flag = 1 also sets reset_cg_signal = 1.
 *            copy_C_to_Y_signal = 1 also sets subspace_diag_signal = 0
 *                                    and reset_cg_signal = 1.
 *
 * SIGUSR2 == SIGUSR1.
 *
 */
void
sig_handler(int sig)
{
  time_t timenow;
  int die_flag,dump_flag;
  int resetcg_flag,subspace_diag_flag,copy_C_to_Y_flag;
  // FILE *sigs_file;
  // char line[85];
  char *signame=NULL;

  /* Zero out the flags */
  die_flag = dump_flag = resetcg_flag = 
             subspace_diag_flag = copy_C_to_Y_flag = 0;

  /* SIGHUP is ignored */
  if (sig == SIGHUP)
    {
      signame = "SIGHUP";
    }
  /* SIGQUIT, SIGTERM, or SIGINT mean dump wavefunction/density n and die */
  else if (sig == SIGQUIT || sig == SIGINT || sig == SIGTERM)
    {
      die_flag = dump_flag = 1;
      resetcg_flag = subspace_diag_flag = copy_C_to_Y_flag = 0;
      if (sig == SIGQUIT)
	signame = "SIGQUIT";
      else if (sig == SIGINT)
	signame = "SIGINT";
      else
	signame = "SIGTERM";
    }
  /* Otherwise, read "sigs.in" and set the flags accordingly */
  else if (sig == SIGUSR1 || sig == SIGUSR2)
    {
      /*** Tairan 10/22 let's see if disabling SIGUSR1 works.
      sigs_file = fopen("sigs.in","r");
      if (sigs_file == (FILE *)0)
	{
	  if (logfile_sigs != NULL)
	    {
	      fprintf(logfile_sigs,
		      "\nReceived SIGUSR1 but can't read sigs.in");
	      fprintf(logfile_sigs,"Ignoring signal.\n");
	      fflush(logfile_sigs);
	    }
	  goto getout;
	}
      fgets(line,80,sigs_file); sscanf(line,"%d",&die_flag);
      fgets(line,80,sigs_file); sscanf(line,"%d",&dump_flag);
      fgets(line,80,sigs_file); sscanf(line,"%d",&resetcg_flag);
      fgets(line,80,sigs_file); sscanf(line,"%d",&subspace_diag_flag);
      fgets(line,80,sigs_file); sscanf(line,"%d",&copy_C_to_Y_flag);
      fclose(sigs_file); ***/
      if (sig == SIGUSR1)
	signame = "SIGUSR1";
      else
	signame = "SIGUSR2";
    }

  /* Dump wave-functions and density and time stamp the file names:
   * The dump name start with 'C.' and 'n.' and the date/time stamp added is
   * month.day.hour:min:sec. */
  timenow = time(0);
  /* if (logfile_sigs != NULL)
    {
      fprintf(logfile_sigs,"\n********************************************\n");
      fprintf(logfile_sigs,"Received %s on %s",signame,ctime(&timenow));
      fflush(logfile_sigs);
    } */
  dft_log("\n********************************************\n");
  dft_log("Proc %d received %s on %s",
	  System::Get_procID(), signame, ctime(&timenow));
  dft_log_flush();

  if (dump_flag == 1)
    dump_and_stamp_variables(*elecinfo_ptr,*elecvars_ptr);

  /* Die! */
  if (die_flag == 1)
    {
      dft_log(DFT_SILENCE,"Exiting.\n");
      dft_log(DFT_SILENCE,"********************************************\n\n");
      die("");
    }
  /* Reset CG cycle...i.e. alpha = 0 */
  if (resetcg_flag == 1)
    {
      resetcg_signal = 1;
    }
  /* Change to the diagonal basis of the subspace Hamiltonian */
  if (subspace_diag_flag == 1)
    {
      resetcg_signal = 1;
      subspace_diag_signal = 1;
      copy_C_to_Y_signal = 0;
    }
  /* Orthonormalize the Y's:  i.e. set Y <- C */
  if (copy_C_to_Y_flag == 1)
    {
      resetcg_signal = 1;
      subspace_diag_signal = 0;
      copy_C_to_Y_signal = 1;
    }
  dft_log("resetcg_signal = %d\n",resetcg_signal);
  dft_log("subspace_diag_signal = %d\n",subspace_diag_signal);
  dft_log("copy_C_to_Y_signal = %d\n",copy_C_to_Y_signal);
  dft_log("********************************************\n\n");
  dft_log_flush();

  /* Reset the signal handling for the next round... */
  // getout: ;
  signal(sig,sig_handler);
}

/*
 * Setup signal handling, global flags, and pointers to the data
 * we need to dump if signalled.
 */
void
setup_signals(Elecinfo *einfo,Elecvars *evars)
{
  /* Set up pointers and flags */
  elecinfo_ptr = einfo;
  elecvars_ptr = evars;
  resetcg_signal = 0;
  subspace_diag_signal = 0;
  copy_C_to_Y_signal = 0;

  /* Set up the signal handler */
  // Tairan 2/5/98 
  // Temporarily disable signal handler
  return;
  signal(SIGHUP ,sig_handler);
  signal(SIGQUIT,sig_handler);
  signal(SIGINT ,sig_handler);
  signal(SIGTERM,sig_handler);
  signal(SIGUSR1,sig_handler);
  signal(SIGUSR2,sig_handler);
}

/*
 * Resets signal handling back to normal.
 */
void
unset_signals(void)
{
  // Tairan 2/5/98 
  // Temporarily disable signal handler
  return;
  signal(SIGHUP ,SIG_DFL);
  signal(SIGQUIT,SIG_DFL);
  signal(SIGINT ,SIG_DFL);
  signal(SIGTERM,SIG_DFL);
  signal(SIGUSR1,SIG_DFL);
  signal(SIGUSR2,SIG_DFL);
}
