/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,           April 7 1997
 *
 * Does {ncg_iter of CG on the electronic variables, then moves the ions}
 * ntotal times.
 *
 * The signal handling ensures that if certain signals are received,
 * wave-functions are dumped, etc.  See signal.c.
 *
 * There are two invocations of the program from the command-line:
 * (1) with no arguments, the file INPUTFILE (below) is read
 * (2) with one argument, which is the name of the input file to be read.
 *
 * The format of the input file is:
 *
 * #
 * # comments are lines that start with '#'
 * #
 * ntotal     # total number of ionic moves
 * ncg_iter    # number of CG iterations
 * init_step   # size of initial linmin step (-1.0 is okay probably)
 * elecfile    # name of file with electronic state information
 * latfile     # name of file with the lattice vectors
 * ionsfile    # name of file with ionic info:  Z, positions, pseudopots,...
 * ewaldfile   # name of file with ewald setup information
 * <Yflag>     # <Yflag> is a string describing initial wave-functions
 * [filename]  # name of file to read for wave-functions if Yflag == 'read'
 * <FFTflag>   # a string describing how FFT box sizes are picked
 * [nx ny nz]  # x,y,z FFT box sizes if FFTflag == 'specified'
 * maskfile    # filename containing name of the force "masks" (see below)
 * K_spring    # spring constant to use for diagonal preconditioner
 *
 * <Yflag> must be either 'read' or 'random' (without quotes).  'random'
 * means that the initial wave-functions are filled with random-numbers.
 * 'read' means that a file should be read:  the name is given on the
 * next line.
 *
 * <FFTflag> must be either 'automatic' or 'specified'.  'automatic'
 * means the program figures out its on FFT box sizes.  'specified'
 * means it tries to use the sizes provided on the next line.
 *
 * The mask file contains the integers 1 or 0 as many as there are atoms 
 * in the system.  The masks go in order of species, and for each species,
 * in order of atoms as given in the position file for that species.
 * The mask value for an atom multiplies the displacement for that atom;
 * in this way, we can have some ions not move.
 *
 * K_spring is used for preconditioning the forces (i.e. just scaling them
 * here...).  The atomic positions are updated via:
 *        r = r + force/K_spring
 * where r and force are in real x-y-z units (NOT LATTICE UNITS), i.e. r
 * is in Bohr and force is in Hartree/Bohr.
 *
 */

/* $Id: relaxions.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
/* My header file */
#include "header.h"

#define INPUTFILE    "relaxions.in"
#define REPORT_LEVEL 1
#define LOGFILE      stdout

#define LN_LEN  150
#define STR_LEN 80


int
main(int argc,char**argv)
{
  System::GlobalInit(&argc,&argv);  

  Basis *basis;         /* The basis set for the calculation */
  Ioninfo ioninfo;     /* Ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */
  Energies ener;       /* Holds energies */
  Control cntrl;       /* Holds convergence control data */

  /* various other local vars */
  int k;
  int niter_cg,ntotal,ionmovloop;
  real stepsize;
  MPI_FILE *filep;
  char line[LN_LEN], elecfilename[STR_LEN], latticefilename[STR_LEN],
    ionsfilename[STR_LEN], init_Y_filename[STR_LEN], init_Y_action[STR_LEN],
    FFT_action[STR_LEN], ewaldfilename[STR_LEN], mask_filename[STR_LEN];
  int nx,ny,nz;
  int sp;
  real K_spring;
  time_t timenow;
  int my_report_level;

  /* If we're processor responsible for IO, then we get to print stuff to the screen! */
  if ( System::Get_ID() == System::Get_IO() )
    my_report_level = REPORT_LEVEL;
  else
    my_report_level = 0;


  /* Read input file for information */
  if (argc == 1)
    {
      if ( (filep = MPI_fopen(INPUTFILE,"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],INPUTFILE);
	  die(line);
	}
    }
  else if (argc == 2)
    {
      if ( (filep = MPI_fopen(argv[1],"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],argv[1]);
	  die(line);
	}
    }
  else
    {
      sprintf(line,"\nUsage:  %s [inputfile]\n\n",argv[0]);
      die(line);
    }

  timenow = time(0);
  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"Current date and time: %s\n",ctime(&timenow));
    if (argc==1)
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],INPUTFILE);
    else
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],argv[1]);
  }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%d",&ntotal);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"ntotal (# of ionic moves) = %d\n",ntotal);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%d",&niter_cg);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"number of CG iterations = %d\n",niter_cg);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%lg",&stepsize);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"stepsize = %lg\n",stepsize);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",elecfilename);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"electronic state file = '%s'\n",elecfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",latticefilename);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"lattice  file = '%s'\n",latticefilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ionsfilename);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"ions file = '%s'\n",ionsfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ewaldfilename);
  if (my_report_level > 0) 
    fprintf(LOGFILE,"ewald file = '%s'\n",ewaldfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",init_Y_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"Yflag = %s ",init_Y_action);

  if (strcmp(init_Y_action,"random") == 0) 
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(init_Y_action,"read") == 0)
    {
      do { fgets(line, LN_LEN, filep); } while(line[0] == '#');
      sscanf(line,"%s",init_Y_filename);
      if (my_report_level > 0)
	fprintf(LOGFILE," file '%s'\n",init_Y_filename);
    }
  else
    {
      sprintf(line,"%s:  initial Yflag must be 'random' or 'read'.\n\n",
	      argv[0]);
      die(line);
    }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",FFT_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"FFTflag = %s",FFT_action);

  nx = ny = nz = 0;
  if (strcmp(FFT_action,"automatic") == 0)
    {
      if (my_report_level > 0) 
	fprintf(LOGFILE,"\n");
    }
  else if (strncmp(FFT_action,"specified",9) == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&nx,&ny,&nz);
      if (my_report_level > 0) 
	fprintf(LOGFILE,":  %d by %d by %d\n",nx,ny,nz);
    }
  else
    {
      sprintf(line,
	      "%s:  initial FFTflag must be 'specified' or 'automatic'.\n\n",
	      argv[0]);
      die(line);
    }

  /*
   * The ion masks capability is already included in ion setup 
   * routine. See comment in setup_ions.c
   *
   * This is to provide backward compatibility.
   */
  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",mask_filename);

  char number_test[STR_LEN];
  if ( sscanf(mask_filename,"%[.0-9]",number_test) == 1 )
    {
      // Humm, it's all positive numerical. Must be the K_spring rather than
      // the mask filename.  Assume that.
      sscanf(line, "%lg", &K_spring);
      mask_filename[0] = '\0';
      if (my_report_level > 0)
	fprintf(LOGFILE,"no maskfile provided, use masks in ion files.\n");
    }
  else 
    { 
      // Well, this is probably the mask filename.
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%lg",&K_spring);
      if (my_report_level > 0) 
	fprintf(LOGFILE,"maskfile = '%s'\n",mask_filename);
    }

  if (my_report_level > 0) 
    fprintf(LOGFILE,"K_spring = %lg\n",K_spring);

  if (K_spring <= 0.0)
    die("\nK_spring <= 0.0!!!\n\nAborting\n\n");

  if (my_report_level > 0) {
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"\n");
  }

  fclose(filep);


  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,my_report_level,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,my_report_level,LOGFILE);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ioninfo,ionsfilename,&elecinfo,my_report_level,LOGFILE);

  if (mask_filename[0] != '\0') {
    /* 
     * Read in the atomic masks. Overwrite whatever mask the 
     * ion file come with.
     */
    if ( (filep = MPI_fopen(mask_filename,"r")) == (MPI_FILE *)0)
      {
	sprintf(line,"\nCan't open maskfile='%s' for reading.\n",mask_filename);
	die(line);
      }
    if (my_report_level > 0) {
      fprintf(LOGFILE,"\n---------    Read overriding ion masks    ----------\n");
      fflush(LOGFILE);
    }

    for (sp=0; sp < ioninfo.nspecies; sp++)
      {
	int i;

	if (my_report_level > 0) 
	  fprintf(LOGFILE,"Reading masks for species=%d  natoms=%d\n",
		  sp,ioninfo.species[sp].natoms);
	for (i=0; i < ioninfo.species[sp].natoms; i++)
	  {
	    fscanf(filep,"%d",&(ioninfo.species[sp].move[i]));
	    if (my_report_level > 0)
	      fprintf(LOGFILE,"%d ",ioninfo.species[sp].move[i]);
	  }
	if (my_report_level > 0) 
	  fprintf(LOGFILE,"\n");
      }
    if (my_report_level > 0) 
      fprintf(LOGFILE,"\n");

    fclose(filep);
  }

  /* Setup Ewald calculation */
  setup_Ewald(ewaldfilename,my_report_level,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* If the flag==1, then randomize initial wavefunctions and then
   * orthonormalize them. */
  if (strcmp(init_Y_action,"random") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n-------> Setting Y to random values\n\n");
      System::seed_with_time();
      randomize_column_bundle_array(elecinfo.nkpts,elecvars.Y);
      /* Orthonormalize the Y */
      calc_UVC(&elecinfo,&elecvars);
      for (k=0; k < elecinfo.nkpts; k++)
	elecvars.Y[k] = elecvars.C[k];
    }
  else
    {
      /* Try to read Y-file */
      if (my_report_level > 0)
	{
	  // cannot use MPI_fopen, since we don't want to read in
	  // the whole binary file!!
	  FILE* testfilep = fopen(init_Y_filename,"r");
	  if (testfilep == (FILE *)0)
	    {
	      sprintf(line,
		      "\nCan't open '%s' to read initial wave-functions.\n\n",
		      init_Y_filename);
	      die(line);
	    }
	  fclose(testfilep);

	  fprintf(LOGFILE,
		  "\n-------> Reading Y from '%s'\n\n",init_Y_filename);
	}
      read_column_bundle_array(init_Y_filename,elecinfo.nkpts,elecvars.Y);
    }

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     my_report_level,LOGFILE);

  /* Setup signal handling */
  setup_signals(&elecinfo,&elecvars,LOGFILE);

  for (ionmovloop=0; ionmovloop < ntotal; ionmovloop++)
    {
      /* status */
      if (my_report_level > 0) {
	fprintf(LOGFILE,"\n>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<\n");
	fprintf(LOGFILE,"Ionic move %d out of %d\n",ionmovloop,ntotal-1);
      }

      /* Calculate core and Ewald energies */
      calc_core_ewald_pulay_energies(basis,&ioninfo,&elecinfo,&ener,
			       my_report_level, LOGFILE);

      /* Calculate the local pseudopotential */
      Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c,my_report_level,LOGFILE);
      /* Calculate the non-local pseudopotential for FHI CPI type psp. */
      Vnl_pseudo(basis,&ioninfo,&elecinfo,my_report_level,LOGFILE);

      
      /* Do conjugate gradients on electronic variables*/
      stepsize = minimize_elec_pcg_nocosgpg(niter_cg,stepsize,
					    basis,&ioninfo,&elecinfo,&elecvars,
					    &ener,cntrl,my_report_level,LOGFILE);

      /* Calculate the forces on the ions */
      calc_ionic_forces(&elecinfo,&elecvars,&ioninfo,basis,
			my_report_level, LOGFILE);

      /* Print them out! */
      int sp;
      
      if (my_report_level > 0)
	{
	  real fabs2 = 0.0; /* = 2 * sqrt(f dot f) */
	  for (sp=0; sp < ioninfo.nspecies; sp++)
	    {
	      int i;
	  
	      fprintf(LOGFILE,"\nspecies = %d,  natoms = %d,  forces follow:\n\n",
		      sp,ioninfo.species[sp].natoms);
	      for (i=0; i < ioninfo.species[sp].natoms; i++) {
		fprintf(LOGFILE,"%3d %4d %20.10le %20.10le %20.10le %d\n",
			sp,i,
			ioninfo.species[sp].forces[i].v[0],
			ioninfo.species[sp].forces[i].v[1],
			ioninfo.species[sp].forces[i].v[2],
			ioninfo.species[sp].move[i]);
		if (ioninfo.species[sp].move[i])
		  fabs2 += ioninfo.species[sp].forces[i] 
		    * ioninfo.species[sp].forces[i]; 
	      }
	    }
	  fabs2 = 2.0 * sqrt(fabs2);
	  fprintf(LOGFILE,"|force_sum| = %lg\n\n",fabs2);
	}
      
      /* Move along forces in real-space with a a spring const of */
      /* K_spring and print their final positions */
      for (sp=0; sp < ioninfo.nspecies; sp++)
	{
	  int i;

	  /* Move the atoms (depending on the masks) */
	  for (i=0; i < ioninfo.species[sp].natoms; i++)
	    {
	      vector3 dtau,f;
	      matrix3 R;
	      
	      R = basis[elecinfo.nkpts].latvec;
	      f = ioninfo.species[sp].forces[i];
	      dtau = (1.0/K_spring)*inv3((~R)*R)*f;
	      ioninfo.species[sp].atpos[i] += dtau*ioninfo.species[sp].move[i];
	    }
	  
	  if (my_report_level > 0)
	    {
	      fprintf(LOGFILE,"Updated positions for species = %d  natoms = %d\n",
		      sp,ioninfo.species[sp].natoms);
	      for (i=0; i < ioninfo.species[sp].natoms; i++)
		fprintf(LOGFILE,"%3d %4d %20.14lf %20.14lf %20.14lf\n",
			sp,i,
			ioninfo.species[sp].atpos[i].v[0],
			ioninfo.species[sp].atpos[i].v[1],
			ioninfo.species[sp].atpos[i].v[2]);
	      fprintf(LOGFILE,"\n");
	    }
	}
    } /* ionmovloop */

  /* Write out final electronic variables */
  if (my_report_level > 0) 
    {
      fprintf(LOGFILE,"\nDone!  Dumping final variables:\n\n");
      fflush(LOGFILE);
    }
  dump_and_stamp_elecvars(elecinfo.nkpts,elecvars.C,&(elecvars.n),
			  &(elecvars.d),&(elecvars.Vlocps),
			  &(elecvars.Vscloc),elecvars.Hsub_evecs,
			  my_report_level,
			  LOGFILE);
  if (my_report_level > 0)
    {
      fprintf(LOGFILE,"\n");
      fflush(stdout);
    }

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  free_ioninfo(&ioninfo);
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  /* End the MPI stuff */
  System::GlobalFinalize();

  return 0;
}

