/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *     Sohrab Ismail-Beigi           Mar. 30, 1997
 *
 * Minimizes the energy over the electronic degrees of freedom
 * using conjugate gradients.
 *
 * Plus a large number of other useful and simpler routines/building blocks.
 *
 */

/* $Id: minimize.c,v 1.3 2000/01/25 23:11:00 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <time.h>
#include <unistd.h>
#include "header.h"
// #include "parallel.h"

/* Global flag for signal processing */
extern int resetcg_signal;
extern int subspace_diag_signal;
extern int copy_C_to_Y_signal;

/*
 * Calculates U, V, C, n, then solves poisson's equation (d).
 */
void
calc_UVCn_d(Elecinfo *einfo,Elecvars *evars,Ioninfo *ioninfo)
{
  calc_UVCn(einfo,evars,ioninfo);
  solve_poisson(evars);
}

/*
 * Calculates U, V, C, n, then solves poisson's equation (d), and
 * then calculates the energies depending on the electronic variables.
 * This is the basic energy evaluation step in conjugate gradients
 * or any other minimization technique I know.
 */
void
calc_UVCn_d_elec_dependent_energies(Basis *basis,
				    Ioninfo *ioninfo,
				    Elecinfo *einfo,Elecvars *evars,
				    Energies *ener)
{
#ifdef DFT_PROFILING
  timerOn(11); // Turn on calc_UCn_d_elec_dependent_e timer
#endif // DFT_PROFILING

  dft_log("Computing U, V, C, n, d, and energies\n");
  dft_log_flush();

  calc_UVCn_d(einfo,evars,ioninfo);
  calc_elec_dependent_energies(basis,ioninfo,einfo,evars,ener);

#ifdef DFT_PROFILING
  timerOff(11); // Turn off calc_UCn_d_elec_dependent_e timer
#endif // DFT_PROFILING
}

/*
 * Calculates U,V,C,n,d,Hsub,and fillings and resets CG flag
 */
void
calc_UVCn_d_Hsub_fillings(Elecinfo *elecinfo,Elecvars *elecvars,
			  Ioninfo *ioninfo,Control &cntrl)
{

  dft_log("Computing U, V, C, n, d, and Hsub\n");
  dft_log_flush();

  /* Calc U, V, C, n, d, Hsub & diagonalize it, and fillings */
  calc_UVCn_d(elecinfo,elecvars,ioninfo);
  calc_Hsub(ioninfo,elecinfo,elecvars);
  calc_fermi_fillings(elecinfo,elecvars,elecinfo->kT, cntrl);
  resetcg_signal = 1;
}

/*
 * Copies C -> Y and resets the copy flag and CG restart flag
 */
void
copy_C_to_Y(Elecinfo *elecinfo,Elecvars *elecvars)
{
  int k;
  
  dft_log("\n### Orthonormalizing Y:  copying C -> Y ###\n\n");

  /* copy C -> Y */
  for (k=0; k < elecinfo->nkpts; k++)
    elecvars->Y[k] = elecvars->C[k];
  /* Reset the signal flag and CG restart flag */
  copy_C_to_Y_signal = 0;
  resetcg_signal = 1;
}

/*
 * Sets Y to the the eigenbasis of the subspace Hamiltonians
 * and rests the signal and CG signal
 */
void
set_Y_to_Hsub_eigenbasis(Elecinfo *elecinfo,Elecvars *elecvars,
			 Ioninfo *ioninfo)
{
  int k;

  dft_log("Computing Hsub\n");
  dft_log_flush();

  /* calc and diagonalize Hsub */
  calc_Hsub(ioninfo,elecinfo,elecvars);

  dft_log("\n### Switching to the diagonal bases of Hsub[k] ###\n\n");
  dft_log_flush();

  /* change to eigenbases of Hsub */
  for (k=0; k < elecinfo->nkpts; k++)
    /* does:    elecvars->Y[k] = elecvars->C[k]*elecvars->Hsub_evecs[k]; */
    do_column_bundle_matrix_mult(elecvars->C[k],
				 elecvars->Hsub_evecs[k],
				 elecvars->Y[k],0);
  /* Reset the signal flag and go up and recalculate everthing */
  subspace_diag_signal = 0;
  resetcg_signal = 1;
}

// Jan 25, 2000: the potential infinite loop in do_linmin has to be controlled.
// here, we set a reasonable limit on how many tries one should do.
#define DO_LINMIN_LIMIT 5

/*
 * Does a linmin on the electronic degrees of freedom in the direction
 * given by dir.  The starting configuration in evars has energy given
 * by ener0 and gradient given by grad.
 *
 * The linmin is done using a quadratic fitting with given stepsize
 * which may be adjusted during the linmin.
 *
 * The routine returns the final stepsize used (which is the
 * recommended value for the next call).
 */
real
do_linmin(column_bundle *Ydir,
	  matrix *Bdir,
	  Basis *basis,
	  Ioninfo *ioninfo,
	  Elecinfo *einfo,Elecvars *evars,
	  Energies *ener0,
	  column_bundle *Ygrad,
	  matrix *Bgrad,
	  real stepsize,
	  Control &cntrl)
{
#ifdef DFT_PROFILING
  timerOn(9); // Turn on do_linmin timer
#endif // DFT_PROFILING

  real gamma,dderiv,curvature;
  Energies ener;
  int nkpts = einfo->nkpts;
  column_bundle *Y = evars->Y;
  matrix *B = evars->B;
  int subspace_rotation = einfo->subspace_rotation;

  /* Directional derivative along dir */
  dderiv = REAL(2.0*dot(nkpts,Ygrad,Ydir));
  if (subspace_rotation)
    dderiv += REAL(dot(nkpts,Bgrad,Bdir));
  for (int i = 0; i < DO_LINMIN_LIMIT; i++)
    {
      /* Shift Y by stepsize: Y[k] += stepsize*Ydir[k] */
      scale_accumulate(nkpts,stepsize,Ydir,Y);
      if (subspace_rotation)
	scale_accumulate(nkpts,stepsize,Bdir,B);

      /* Calculate the energy at the shifted position */
      ener = *ener0;

#ifdef DFT_PROFILING
      timerOn(28); // Turn on calc_UCn_d_... in do_linmin timer
#endif // DFT_PROFILING

      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,einfo,evars,&ener);

#ifdef DFT_PROFILING
      timerOff(28); // Turn off calc_UCn_d_... in do_linmin timer
#endif // DFT_PROFILING

      /* Shift back */
      scale_accumulate(nkpts,-stepsize,Ydir,Y);
      if (subspace_rotation)
	scale_accumulate(nkpts,-stepsize,Bdir,B);

      /* Do a parabolic fit to the Etot0, Etot, and dderiv
       * to get curvature for quadratic fit and proposed minimum (gamma) */
      curvature = 2.0*(ener.Etot-ener0->Etot-stepsize*dderiv) / 
	(stepsize*stepsize);
      gamma = -dderiv/curvature;
      dft_log("dderiv = %14.7le  curvature = %14.7le\n",dderiv,curvature);
      dft_log("stepsize = %14.7le    gamma = %14.7le\n",stepsize,gamma);

      /* If curvature is wrong way, take a bigger step */
      if (curvature < 0.0)
	{ stepsize *= 4.0; continue; }
      /* If the proposed minimum is much larger than the stepsize,
       * increase stepsize */
      else if (fabs(gamma/stepsize) > 10.0)
	{ stepsize *= 10.0; continue; }
      /* If much smaller, decrease */
      // Tairan (10/15/1998): but the stepsize should not be too small.
      //  otherwise, numerical error will overwhelm the calculation
      //   Y += stepsize * dir
      else if ( (fabs(gamma/stepsize) <  0.1)
		&& (fabs(stepsize) > cntrl.stepsize_min) )
	{ stepsize *= 0.1; continue; }
      /* Otherwise, it was a good linmin so stop! */
      else
	break;
    }

  /* Move to the bottom of the parabola, given by gamma */
  scale_accumulate(nkpts,gamma,Ydir,Y);
  if (subspace_rotation)
    scale_accumulate(nkpts,gamma,Bdir,B);

#ifdef DFT_PROFILING
  timerOff(9);  // Turn off do_linmin timer
#endif // DFT_PROFILING

  /* If update_stepsize flag is set, use gamma as the new stepsize
     if it is larger than the stepsize minimum.
     Otherwise, update stepsize only if it has been updated by 
     factors of 10 or 0.1 */
  if (cntrl.update_stepsize && (fabs(gamma) > cntrl.stepsize_min))
    {
      dft_log("stepsize updated to  %e : (min) %e\n",
	      gamma, cntrl.stepsize_min);
      return gamma;
    }
  return stepsize;
}

/*
 * Minimize energy over electronic degrees of freedom with conjugate
 * gradients.
 *
 * niter CG iterations are done
 * stepsize is how far to go along the search direction for the linmins;
 *     it is changed dynamically by the linmins.
 * ener contains the energy of the configuration whose energy is printed
 * report and logfile are passed on to all lower-level routines.
 *
 * The routine returns the last stepsize used in the linmin.
 */
real
minimize_elec_cg(Basis *basis,
		 Ioninfo *ioninfo,
		 Elecinfo *elecinfo,
		 Elecvars *elecvars,
		 Energies *ener,
		 Control &cntrl)
{
  int iter,k;
  time_t timenow;
  real linminfact=0.0, cosgg=0.0, alpha,abs2_grad_now=0.0,abs2_grad_old,
    abs2_Ygrad_now=0.0,abs2_Bgrad_now=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad_now,*Ygrad_old,*Ydir;
  matrix *Bgrad_now,*Bgrad_old,*Bdir;

  /* Allocate space for the gradients and search directions */
  Ygrad_now = alloc_column_bundle_array(nkpts,nbands,basis);
  Ygrad_old = alloc_column_bundle_array(nkpts,nbands,basis);
  Ydir = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation)
    {
      Bgrad_now = alloc_matrix_array(nkpts,nbands,nbands);
      Bgrad_old = alloc_matrix_array(nkpts,nbands,nbands);
      Bdir = alloc_matrix_array(nkpts,nbands,nbands);
    }
  else
    {
      Bgrad_now = NULL;
      Bgrad_old = NULL;
      Bdir = NULL;
    }

  dft_log("\n----- minimize_elec_cg() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Conjugate Gradients with\n",
	  cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	    elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of CG iterations! */
  iter = 0;
  do {
    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,
					  elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}

    } while (UCnd_flag);

    /* Calculate energy gradient versus Y and Hsub and diagonalize Hsub*/
    dft_log("Computing Gradient and Hsub\n");
    dft_log_flush();

    calc_elecgrad_and_Hsub(Ygrad_now,Bgrad_now,ioninfo,elecinfo,elecvars);

    /* Square length of gradients */
    abs2_grad_old = abs2_grad_now;
    abs2_Ygrad_now = 4.0*abs2(nkpts,Ygrad_now);
    abs2_grad_now = abs2_Ygrad_now;
    if (subspace_rotation)
      {
	abs2_Bgrad_now = abs2(nkpts,Bgrad_now);
	abs2_grad_now += abs2_Bgrad_now;
      }

    /* Some linimin statistics from last iteration */
    if (iter > 0)
      {
	if (subspace_rotation) {
	  linminfact = 
	    REAL(2.0*dot(nkpts,Ygrad_now,Ydir)+dot(nkpts,Bgrad_now,Bdir))/
	    REAL(2.0*dot(nkpts,Ygrad_old,Ydir)+dot(nkpts,Bgrad_old,Bdir));
	  cosgg = REAL(4.0*dot(nkpts,Ygrad_old,Ygrad_now)+
		       dot(nkpts,Bgrad_old,Bgrad_now))/
	    sqrt( abs2_grad_now*abs2_grad_old  );
	} else {
	  linminfact = 
	    REAL(dot(nkpts,Ygrad_now,Ydir))/REAL(dot(nkpts,Ygrad_old,Ydir));
	  cosgg = REAL(4.0*dot(nkpts,Ygrad_old,Ygrad_now))/
	    sqrt( abs2_grad_now*abs2_grad_old  );
	}
	dft_log("\nlinmin = %8.1le   cosgg = %8.1le\n\n", linminfact,cosgg);
      }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log("------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't linmin, but
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot convergence.
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    }
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
      cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else {

      /* Calculate search direction */
      alpha = 0.0;
      /* If we're given the reset CG signal, keep alpha at zero and reset the
       * signal flag. */ 
      if (resetcg_signal == 1)
	resetcg_signal = 0;
      /* Otherwise, if we have done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      else if (iter>0 && fabs(linminfact) < 0.05 && fabs(cosgg) < 0.1)
	alpha = abs2_grad_now/abs2_grad_old;

      if (subspace_rotation) 
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now),
		sqrt(abs2_Bgrad_now));
      else
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now));
      dft_log("\n");

      /* Calculate current search direction:
       * dir = 2.0*grad_now + alpha*dir_old */
      for (k=0; k < nkpts; k++)
	{
	  /* On first iteration, zero out dir[k]. */
	  if (iter == 0)
	    Ydir[k].zero_out();
	  else 
	    Ydir[k] *= alpha;
	  Ydir[k] += Ygrad_now[k];
	  Ydir[k] += Ygrad_now[k];
	  if (subspace_rotation)
	    {
	      if (iter == 0)
		Bdir[k].zero_out();
	      else
		Bdir[k] *= (scalar)alpha;
		Bdir[k] += Bgrad_now[k];
	    }
	}

      /* Do a linmin along dir */
      cntrl.elec_stepsize = do_linmin(Ydir,Bdir,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad_now,Bgrad_now,cntrl.elec_stepsize,
				      cntrl);

      /* Copy old gradients */
      for (k=0; k < nkpts; k++)
	{
	  Ygrad_old[k] = Ygrad_now[k];
	  if (subspace_rotation)
	    Bgrad_old[k] = Bgrad_now[k];
	}

#ifdef DFT_PROFILING
      // Timing report at the end of the iteration
      timerReport(iter);
#endif // DFT_PROFILING

    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad_now);
  free_column_bundle_array(nkpts,Ygrad_old);
  free_column_bundle_array(nkpts,Ydir);
  if (subspace_rotation) {
    free_matrix_array(nkpts,Bgrad_now);
    free_matrix_array(nkpts,Bgrad_old);
    free_matrix_array(nkpts,Bdir);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}

/*
 * Does the same as above, except it does not report 'cosgg' and so
 * does not allocate memory for the old gradient (grad_old).
 */
real
minimize_elec_cg_nocosgg(Basis *basis,
			 Ioninfo *ioninfo,
			 Elecinfo *elecinfo,
			 Elecvars *elecvars,
			 Energies *ener,
			 Control &cntrl)
{
  int iter,k;
  time_t timenow;
  real linminfact=0.0,alpha,abs2_grad_now=0.0,abs2_grad_old,
    abs2_Ygrad_now=0.0,abs2_Bgrad_now=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad,*Ydir;
  matrix *Bgrad,*Bdir;


  /* Allocate space for the gradients and search directions */
  Ygrad = alloc_column_bundle_array(nkpts,nbands,basis);
  Ydir = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation) {
    Bgrad = alloc_matrix_array(nkpts,nbands,nbands);
    Bdir = alloc_matrix_array(nkpts,nbands,nbands);
  } else {
    Bgrad = NULL;
    Bdir = NULL;
  }

  dft_log("\n----- minimize_elec_cg_nocosgg() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Conjugate Gradients with\n",
	  cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	    elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of CG iterations! */
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,
					  elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag); 

    /* Square length of old gradient; linmin is ratio of dot() of dir
     * with old and new gradient...  so here we do the denominator. */
    abs2_grad_old = abs2_grad_now;
    if (subspace_rotation)
      linminfact = 1.0/REAL(2.0*dot(nkpts,Ygrad,Ydir)+dot(nkpts,Bgrad,Bdir));
    else
      linminfact = 1.0/REAL(2.0*dot(nkpts,Ygrad,Ydir));

    /* Calculate energy gradient versus Y and Hsub and diagonalize Hsub*/
    dft_log("Computing Gradient and Hsub\n");
    dft_log_flush();

    calc_elecgrad_and_Hsub(Ygrad,Bgrad,ioninfo,elecinfo,elecvars);

    /* Square length of current gradient */
    abs2_Ygrad_now = 4.0*abs2(nkpts,Ygrad);
    abs2_grad_now = abs2_Ygrad_now;
    if (subspace_rotation) {
      abs2_Bgrad_now = abs2(nkpts,Bgrad);
      abs2_grad_now += abs2_Bgrad_now;
    }

    /* Some linimin statistics from last iteration */
    if (iter > 0)
      {
	if (subspace_rotation)
	  linminfact *= REAL(2.0*dot(nkpts,Ygrad,Ydir)+dot(nkpts,Bgrad,Bdir));
	else
	  linminfact *= REAL(2.0*dot(nkpts,Ygrad,Ydir));
	dft_log("\nlinmin = %8.1le\n\n",linminfact);
      }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log("------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't linmin, but
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    } 
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    } 
    // go ahead with linmin
    else {

      /* Calculate search direction */
      alpha = 0.0;
      /* If we're given the reset CG signal, keep alpha at zero and reset the
       * signal flag. */ 
      if (resetcg_signal == 1)
	resetcg_signal = 0;
      /* Otherwise, if we have done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      else if (iter>0 && fabs(linminfact) < 0.05)
	alpha = abs2_grad_now/abs2_grad_old;

      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now),
		sqrt(abs2_Bgrad_now));
      else
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now));
      dft_log("alpha = %8.1le\n",alpha);


      /* Calculate current search direction:
       * dir = 2.0*grad_now + alpha*dir_old */
      for (k=0; k < nkpts; k++)
	{
	  /* On first iteration, zero out dir[k]. */
	  if (iter == 0)
	    Ydir[k].zero_out();
	  else
	    Ydir[k] *= alpha;
	  Ydir[k] += Ygrad[k];
	  Ydir[k] += Ygrad[k];
	  if (subspace_rotation) {
	    if (iter == 0)
	      Bdir[k].zero_out();
	    else 
	      Bdir[k] *= (scalar)alpha;
	    Bdir[k] += Bgrad[k];
	  }
	}

      /* Do a linmin along dir */
      cntrl.elec_stepsize = do_linmin(Ydir,Bdir,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad,Bgrad,cntrl.elec_stepsize,
				      cntrl);

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING

    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad);
  free_column_bundle_array(nkpts,Ydir);
  if (subspace_rotation) { 
    free_matrix_array(nkpts,Bgrad);
    free_matrix_array(nkpts,Bdir);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}


/*
 * Minimize energy over electronic degrees of freedom with
 * preconditioned conjugate gradients.
 *
 * niter PCG iterations are done
 * stepsize is how far to go along the search direction for the linmins;
 *     it is changed dynamically by the linmins.
 * ener contains the energy of the configuration whose energy is printed
 * report and logfile are passed on to all lower-level routines.
 *
 * The routine returns the last stepsize used in the linmin.
 */
real
minimize_elec_pcg(Basis *basis,
		  Ioninfo *ioninfo,
		  Elecinfo *elecinfo,
		  Elecvars *elecvars,
		  Energies *ener,
		  Control &cntrl)
{
  int iter,k;
  time_t timenow;
  real linminfact=0.0, cosgpg=0.0, alpha;
  real abs2_pgrad_now=0.0,abs2_grad_old,abs2_grad_now=0.0,
    grad_old_dot_pgrad_old=0.0,
    grad_now_dot_pgrad_now=0.0,grad_old_dot_pgrad_now=0.0,
    dir_dot_grad_old,dir_dot_grad_now=0.0,
    abs2_Ygrad_now=0.0,abs2_Bgrad_now=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad_now,*Ygrad_old,*pYgrad_now,*Ydir;
  matrix *Bgrad_now,*Bgrad_old,*pBgrad_now,*Bdir;

  /* Allocate space for the gradients and search directions */
  Ygrad_now = alloc_column_bundle_array(nkpts,nbands,basis);
  Ygrad_old = alloc_column_bundle_array(nkpts,nbands,basis);
  pYgrad_now = alloc_column_bundle_array(nkpts,nbands,basis);
  Ydir = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation) {
    Bgrad_now = alloc_matrix_array(nkpts,nbands,nbands);
    Bgrad_old = alloc_matrix_array(nkpts,nbands,nbands);
    pBgrad_now = alloc_matrix_array(nkpts,nbands,nbands);
    Bdir = alloc_matrix_array(nkpts,nbands,nbands);
  } else {
    Bgrad_now = NULL;
    Bgrad_old = NULL;
    pBgrad_now = NULL;
    Bdir = NULL;
  }

  dft_log("\n----- minimize_elec_pcg() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Precond. Conjugate Gradients with\n",
	  cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	    elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of PCG iterations! */
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,
					  elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag);

    /* Calculate energy gradient versus Y, preconditioned gradient,
     * and Hsub, and also diagonalize Hsub.  We also calulate some
     * dot products and vector square lengths we'll need. */
    dft_log("Computing Gradient and Hsub\n");
    dft_log_flush();

    /* This one we must copy before we recalculate them */
    grad_old_dot_pgrad_old = grad_now_dot_pgrad_now;
    abs2_grad_old = abs2_grad_now;

    calc_elecgrad_pgrad_and_Hsub(Ygrad_now,pYgrad_now,
				 Bgrad_now,pBgrad_now,
				 ener,ioninfo,elecinfo,elecvars);

    grad_now_dot_pgrad_now = REAL(4.0*dot(nkpts,Ygrad_now,pYgrad_now));
    grad_old_dot_pgrad_now = REAL(4.0*dot(nkpts,Ygrad_old,pYgrad_now));
    abs2_pgrad_now = 4.0*abs2(nkpts,pYgrad_now);
    abs2_Ygrad_now = 4.0*abs2(nkpts,Ygrad_now);
    abs2_grad_now = abs2_Ygrad_now;
    dir_dot_grad_now = REAL(2.0*dot(nkpts,Ydir,Ygrad_now));
    dir_dot_grad_old = REAL(2.0*dot(nkpts,Ydir,Ygrad_old));
    if (subspace_rotation) {
      grad_now_dot_pgrad_now += REAL(dot(nkpts,Bgrad_now,pBgrad_now));
      grad_old_dot_pgrad_now += REAL(dot(nkpts,Bgrad_old,pBgrad_now));
      abs2_pgrad_now += abs2(nkpts,pBgrad_now);
      abs2_Bgrad_now = abs2(nkpts,Bgrad_now);
      abs2_grad_now += abs2_Bgrad_now;
      dir_dot_grad_now += REAL(dot(nkpts,Bdir,Bgrad_now));
      dir_dot_grad_old += REAL(dot(nkpts,Bdir,Bgrad_old));
    }

    /* Some linimin statistics from last iteration */
    if (iter > 0)
      {
	linminfact = dir_dot_grad_now/dir_dot_grad_old;
	cosgpg = grad_old_dot_pgrad_now/sqrt(abs2_grad_old*abs2_pgrad_now);
	dft_log("\nlinmin = %8.1le   cosgpg = %8.1le\n\n",
		  linminfact,cosgpg);
      }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log(
	      "------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't linmin, but
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	  cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    } 
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else { 

      /* Calculate search direction */
      alpha = 0.0;
      /* If we're given the reset CG signal, keep alpha at zero and reset the
       * signal flag. */ 
      if (resetcg_signal == 1)
	resetcg_signal = 0;
      /* Otherwise, if we have done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      else if (iter>0 && fabs(linminfact) < 0.05 && fabs(cosgpg) < 0.1)
	alpha = grad_now_dot_pgrad_now/grad_old_dot_pgrad_old;

      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now),
		sqrt(abs2_Bgrad_now));
      else
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now));
      dft_log("alpha = %8.1le\n",alpha);
      dft_log_flush();

      /* Calculate current search direction:
       * dir = 2.0*pgrad_now + alpha*dir_old */
      for (k=0; k < nkpts; k++)
	{
	  /* On first iteration, zero out dir[k]. */
	  if (iter == 0)
	    Ydir[k].zero_out();
	  else 
	    Ydir[k] *= alpha;
	  Ydir[k] += pYgrad_now[k];
	  Ydir[k] += pYgrad_now[k];
	  if (subspace_rotation) {
	    if (iter == 0)
	      Bdir[k].zero_out();
	    else 
	      Bdir[k] *= (scalar)alpha;
	    Bdir[k] += pBgrad_now[k];
	  }
	}

      /* Do a linmin along dir */
      cntrl.elec_stepsize = do_linmin(Ydir,Bdir,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad_now,Bgrad_now,cntrl.elec_stepsize,
				      cntrl);

      /* copy old gradient */
      for (k=0; k < nkpts; k++)
	{
	  Ygrad_old[k] = Ygrad_now[k];
	  if (subspace_rotation)
	    Bgrad_old[k] = Bgrad_now[k];
	}

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING
    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad_now);
  free_column_bundle_array(nkpts,Ygrad_old);
  free_column_bundle_array(nkpts,Ydir);
  free_column_bundle_array(nkpts,pYgrad_now);
  if (subspace_rotation) {
    free_matrix_array(nkpts,Bgrad_now);
    free_matrix_array(nkpts,Bgrad_old);
    free_matrix_array(nkpts,Bdir);
    free_matrix_array(nkpts,pBgrad_now);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}


/*
 * The same thing as the above routine, except that the cos of 
 * the old gradient and the new preconditioned gradient is not reported.
 * This means that the old gradient need not be stored, so this
 * version uses less memory.
 */
real
minimize_elec_pcg_nocosgpg(Basis *basis,
			   Ioninfo *ioninfo,
			   Elecinfo *elecinfo,
			   Elecvars *elecvars,
			   Energies *ener,
			   Control &cntrl)
{
  int iter,k;
  time_t timenow;
  real linminfact=0.0, alpha;
  real abs2_grad_now=0.0,grad_old_dot_pgrad_old,grad_now_dot_pgrad_now=0.0,
    dir_dot_grad_old,dir_dot_grad_now=0.0,
    abs2_Ygrad_now=0.0,abs2_Bgrad_now=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad,*pYgrad,*Ydir;
  matrix *Bgrad,*pBgrad,*Bdir;

  /* Allocate space for the gradients and search directions */
  Ygrad = alloc_column_bundle_array(nkpts,nbands,basis);
  pYgrad = alloc_column_bundle_array(nkpts,nbands,basis);
  Ydir = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation) {
    Bgrad = alloc_matrix_array(nkpts,nbands,nbands);
    pBgrad = alloc_matrix_array(nkpts,nbands,nbands);
    Bdir = alloc_matrix_array(nkpts,nbands,nbands);
  } else {
    Bgrad = NULL;
    pBgrad = NULL;
    Bdir = NULL;
  }

  dft_log("\n----- minimize_elec_pcg_nocosgpg() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Precond. Conjugate Gradients with\n",
	    cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	      elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of PCG iterations! */
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,
					  elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag);

    /* Calculate energy gradient versus Y, preconditioned gradient,
     * and Hsub, and also diagonalize Hsub.  We also calulate some
     * dot products and vector square lengths we'll need. */
    dft_log("Computing Gradient and Hsub\n");
    dft_log_flush();

    /* These ones we must calculate before grad and pgrad are over-written */
    grad_old_dot_pgrad_old = grad_now_dot_pgrad_now;
    dir_dot_grad_old = REAL(2.0*dot(nkpts,Ydir,Ygrad));
    if (subspace_rotation)
      dir_dot_grad_old += REAL(dot(nkpts,Bdir,Bgrad));

    calc_elecgrad_pgrad_and_Hsub(Ygrad,pYgrad,Bgrad,pBgrad,
				 ener,ioninfo,elecinfo,elecvars);

    grad_now_dot_pgrad_now = REAL(4.0*dot(nkpts,Ygrad,pYgrad));
    dir_dot_grad_now = REAL(2.0*dot(nkpts,Ydir,Ygrad));
    abs2_Ygrad_now = 4.0*abs2(nkpts,Ygrad);
    abs2_grad_now = abs2_Ygrad_now;
    if (subspace_rotation) {
      grad_now_dot_pgrad_now += REAL(dot(nkpts,Bgrad,pBgrad));
      dir_dot_grad_now += REAL(dot(nkpts,Bdir,Bgrad));
      abs2_Bgrad_now = abs2(nkpts,Bgrad);
      abs2_grad_now += abs2_Bgrad_now;
    }

    /* Some linimin statistics from last iteration */
    if (iter > 0)
      {
	linminfact = dir_dot_grad_now/dir_dot_grad_old;
	dft_log("\nlinmin = %8.1le\n",linminfact);
      }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log(
	      "------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't linmin, but
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	  cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    }
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else {

      /* Calculate search direction */
      alpha = 0.0;
      /* If we're given the reset CG signal, keep alpha at zero and reset the
       * signal flag. */ 
      if (resetcg_signal == 1)
	resetcg_signal = 0;
      /* Otherwise, if we have done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      else if (iter>0 && fabs(linminfact) < 0.05)
	alpha = grad_now_dot_pgrad_now/grad_old_dot_pgrad_old;

      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now),
		sqrt(abs2_Bgrad_now));
      else 
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now));
      dft_log("alpha = %8.1le\n",alpha);

      /* Calculate current search direction:
       * dir = 2.0*pgrad_now + alpha*dir_old */
      for (k=0; k < nkpts; k++)
	{
	  /* On first iteration, zero out dir[k]. */
	  if (iter == 0)
	    Ydir[k].zero_out();
	  else 
	    Ydir[k] *= alpha;
	  Ydir[k] += pYgrad[k];
	  Ydir[k] += pYgrad[k];
	  if (subspace_rotation) {
	    if (iter == 0)
	      Bdir[k].zero_out();
	    else 
	      Bdir[k] *= (scalar)alpha;
	    Bdir[k] += pBgrad[k];
	  }
	}

      /* Do a linmin along dir */
      cntrl.elec_stepsize = do_linmin(Ydir,Bdir,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad,Bgrad,cntrl.elec_stepsize,
				      cntrl);

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING

    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad);
  free_column_bundle_array(nkpts,Ydir);
  free_column_bundle_array(nkpts,pYgrad);
  if (subspace_rotation) {
    free_matrix_array(nkpts,Bgrad);
    free_matrix_array(nkpts,Bdir);
    free_matrix_array(nkpts,pBgrad);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}

/*
 * This does EOM (equation of motion) minimization of the energy
 * versus the electronic degrees of freedom.  By EOM I mean we move
 * along the gradient scaled by a stepsize.  Stepsize is dynamically
 * adjusted so the energy keeps decreasing.  The algorithm is:
 *
 *     do niter times
 *        {
 *             Y <- Y - stepsize*grad ; 
 *             if E(Y) decreased then stepsize *= 1.1; 
 *             else stepsize *= 0.3;
 *        }
 *
 *
 * niter iterations of EOM are done.
 * stepsize is the initial stepsize to use.
 *
 */
real
minimize_elec_eom(Basis *basis,
		  Ioninfo *ioninfo,
		  Elecinfo *elecinfo,
		  Elecvars *elecvars,
		  Energies *ener,
		  Control &cntrl)
{
  int iter;
  time_t timenow;
  real abs2_grad,abs2_Ygrad, abs2_Bgrad=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* The energy from the previous iteration */
  Energies ener_old;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad;
  matrix *Bgrad;


  /* Allocate space for the gradients and search directions */
  Ygrad = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation)
    Bgrad = alloc_matrix_array(nkpts,nbands,nbands);
  else
    Bgrad = NULL;

  dft_log("\n----- minimize_elec_eom() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Equation of Motion with\n",
	    cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	      elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of EOM iterations! */
  cntrl.elec_stepsize = -fabs(cntrl.elec_stepsize);
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Save the old energy and calculate energy of current config */
      ener_old = *ener;
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag);

    /* Calculate energy gradient versus Y and Hsub and diagonalize Hsub*/
    dft_log("Computing Gradient and Hsub\n\n");
    dft_log_flush();

    calc_elecgrad_and_Hsub(Ygrad,Bgrad,ioninfo,elecinfo,elecvars);

    /* Square length of current gradient */
    abs2_Ygrad = 4.0*abs2(nkpts,Ygrad);
    abs2_grad = abs2_Ygrad;
    if (subspace_rotation) {
      abs2_Bgrad = abs2(nkpts,Bgrad);
      abs2_grad += abs2_Bgrad;
    }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log("------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();


    /* Dynamically adjust stepsize depending on whether the energy
     * went up or down. */
    if (iter > 0)
      {
	if (ener->Etot < ener_old.Etot)
	  cntrl.elec_stepsize *= 1.05;
	else
	  cntrl.elec_stepsize *= 0.3;
      }

    /* If this is the last iteration, don't move but just 
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	  cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    }
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else {

      dft_log("stepsize = %le\n",cntrl.elec_stepsize);
      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		  sqrt(abs2_grad),sqrt(abs2_Ygrad),sqrt(abs2_Bgrad));
      else
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad),sqrt(abs2_Ygrad));

      /* Move by stepsize along gradient */
      scale_accumulate(nkpts,cntrl.elec_stepsize,Ygrad,elecvars->Y);
      if (subspace_rotation)
	scale_accumulate(nkpts,cntrl.elec_stepsize,Bgrad,elecvars->B);

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING
    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad);
  if (subspace_rotation)
    free_matrix_array(nkpts,Bgrad);

  /* return stepsize used */
  return cntrl.elec_stepsize;
}

/*
 * This does PSD (precond. steepest descent) minimization of the energy
 * versus the electronic degrees of freedom.  By PSD I mean we linmin
 * along the precond. gradient.
 * niter iterations of PSD are done.
 * stepsize is the initial stepsize to use.
 *
 */
real
minimize_elec_psd(Basis *basis,
		  Ioninfo *ioninfo,
		  Elecinfo *elecinfo,
		  Elecvars *elecvars,
		  Energies *ener,
		  Control &cntrl)
{
  int iter;
  time_t timenow;
  real abs2_grad,abs2_Ygrad,abs2_Bgrad=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad,*pYgrad;
  matrix *Bgrad,*pBgrad;


  /* Allocate space for the gradients and search directions */
  Ygrad = alloc_column_bundle_array(nkpts,nbands,basis);
  pYgrad = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation) {
    Bgrad = alloc_matrix_array(nkpts,nbands,nbands);
    pBgrad = alloc_matrix_array(nkpts,nbands,nbands);
  } else {
    Bgrad = NULL;
    pBgrad = NULL;
  }

  dft_log("\n----- minimize_elec_psd() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Preconditioned Steepest Descent with\n",
	    cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	      elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of PSD iterations! */
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Save the old energy and calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag);

    /* Calculate energy gradient versus Y and Hsub and diagonalize Hsub*/
    dft_log("Computing Gradient and Hsub\n\n");
    dft_log_flush();

    calc_elecgrad_pgrad_and_Hsub(Ygrad,pYgrad,Bgrad,pBgrad,
				 ener,ioninfo,elecinfo,elecvars);

    /* Square length of current gradient */
    abs2_Ygrad = 4.0*abs2(nkpts,Ygrad);
    abs2_grad = abs2_Ygrad;
    if (subspace_rotation) {
      abs2_Bgrad = abs2(nkpts,Bgrad);
      abs2_grad += abs2_Bgrad;
    }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log("------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't move but just 
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	  cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    }
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
      cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else {

      dft_log("stepsize = %le\n",cntrl.elec_stepsize);
      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		  sqrt(abs2_grad),sqrt(abs2_Ygrad),sqrt(abs2_Bgrad));
      else
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad),sqrt(abs2_Ygrad));

      /* Do linmin along precond. gradient */
      cntrl.elec_stepsize = do_linmin(pYgrad,pBgrad,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad,Bgrad,cntrl.elec_stepsize,
				      cntrl);

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING
    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad);
  free_column_bundle_array(nkpts,pYgrad);
  if (subspace_rotation) {
    free_matrix_array(nkpts,Bgrad);
    free_matrix_array(nkpts,pBgrad);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}




/*
 * The same thing as the above routine, except that the cos of 
 * the old gradient and the new preconditioned gradient is not reported.
 * This means that the old gradient need not be stored, so this
 * version uses less memory.
 */
real
minimize_elec_pcg_nocosgpg_bands(Basis *basis,
				 Ioninfo *ioninfo,
				 Elecinfo *elecinfo,
				 Elecvars *elecvars,
				 Energies *ener,
				 Control &cntrl)
{
  int iter,k;
  time_t timenow;
  real linminfact=0.0, alpha;
  real abs2_grad_now=0.0,grad_old_dot_pgrad_old,grad_now_dot_pgrad_now=0.0,
    dir_dot_grad_old,dir_dot_grad_now=0.0,
    abs2_Ygrad_now=0.0,abs2_Bgrad_now=0.0;
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int subspace_rotation = elecinfo->subspace_rotation;

  /* gradient, search direction, and the gradient from last iter */
  column_bundle *Ygrad,*pYgrad,*Ydir;
  matrix *Bgrad,*pBgrad,*Bdir;

  /* Allocate space for the gradients and search directions */
  Ygrad = alloc_column_bundle_array(nkpts,nbands,basis);
  pYgrad = alloc_column_bundle_array(nkpts,nbands,basis);
  Ydir = alloc_column_bundle_array(nkpts,nbands,basis);

  if (subspace_rotation) {
    Bgrad = alloc_matrix_array(nkpts,nbands,nbands);
    pBgrad = alloc_matrix_array(nkpts,nbands,nbands);
    Bdir = alloc_matrix_array(nkpts,nbands,nbands);
  } else {
    Bgrad = NULL;
    pBgrad = NULL;
    Bdir = NULL;
  }

  dft_log("\n----- minimize_elec_pcg_nocosgpg() -----\n");
  dft_log("Called on %s",ctime(&(timenow=time(0))));
  dft_log("Starting %d iterations of Precond. Conjugate Gradients with\n",
	    cntrl.max_elecs_steps);
  dft_log("initial stepsize = %lg.\n",cntrl.elec_stepsize);
  dft_log("calc_fillings_flag = %d\n",elecinfo->calc_fillings);
  if (elecinfo->calc_fillings!=0)
    dft_log("kT = %lg   niter_recalc_fillings = %d\n",
	      elecinfo->kT,elecinfo->niter_recalc_fillings);
  dft_log("\n");
  dft_log_flush();

#ifdef DFT_PROFILING
  timerOn(2);  // turn on electron iteration timer
#endif // DFT_PROFILING

  /* The big loop of PCG iterations! */
  iter = 0;
  do {

    /* See if we have to recompute fillings */
    if ( elecinfo->calc_fillings == 1 )
      if ( iter % elecinfo->niter_recalc_fillings == 0 )
	calc_UVCn_d_Hsub_fillings(elecinfo,elecvars,ioninfo,cntrl);

    /* Reentry point to recalculate U,C,n,d,and energies */
    int UCnd_flag;
    do {

      UCnd_flag = 0;

      /* Calculate energy of current config */
      calc_UVCn_d_elec_dependent_energies(basis,ioninfo,
					  elecinfo,elecvars,ener);

      /* If signalled, orthonormalize wavefuncs (i.e. copy Y <- C) */
      if (copy_C_to_Y_signal == 1)
	{
	  copy_C_to_Y(elecinfo,elecvars);
	  UCnd_flag = 1;
	}
      /* If signalled, switch to the diagonal basis of Hsub */
      else if (subspace_diag_signal == 1)
	{
	  set_Y_to_Hsub_eigenbasis(elecinfo,elecvars,ioninfo);
	  UCnd_flag = 1;
	}
    } while (UCnd_flag);

    /* Calculate energy gradient versus Y, preconditioned gradient,
     * and Hsub, and also diagonalize Hsub.  We also calulate some
     * dot products and vector square lengths we'll need. */
    dft_log("Computing Gradient and Hsub\n");
    dft_log_flush();

    /* These ones we must calculate before grad and pgrad are over-written */
    grad_old_dot_pgrad_old = grad_now_dot_pgrad_now;
    dir_dot_grad_old = REAL(2.0*dot(nkpts,Ydir,Ygrad));
    if (subspace_rotation)
      dir_dot_grad_old += REAL(dot(nkpts,Bdir,Bgrad));

    calc_elecgrad_pgrad_and_Hsub(Ygrad,pYgrad,Bgrad,pBgrad,
				 ener,ioninfo,elecinfo,elecvars);

    grad_now_dot_pgrad_now = REAL(4.0*dot(nkpts,Ygrad,pYgrad));
    dir_dot_grad_now = REAL(2.0*dot(nkpts,Ydir,Ygrad));
    abs2_Ygrad_now = 4.0*abs2(nkpts,Ygrad);
    abs2_grad_now = abs2_Ygrad_now;
    if (subspace_rotation) {
      grad_now_dot_pgrad_now += REAL(dot(nkpts,Bgrad,pBgrad));
      dir_dot_grad_now += REAL(dot(nkpts,Bdir,Bgrad));
      abs2_Bgrad_now = abs2(nkpts,Bgrad);
      abs2_grad_now += abs2_Bgrad_now;
    }

    /* Some linimin statistics from last iteration */
    if (iter > 0)
      {
	linminfact = dir_dot_grad_now/dir_dot_grad_old;
	dft_log("\nlinmin = %8.1le\n",linminfact);
      }

    /* Time stamp and print out current energies and subspace Hamiltonian */
    timenow = time(0);
    dft_log(
	      "------------------------------------------------------\n");
    dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
    print_energies(elecinfo,ener,dft_global_log);
    dft_log("\n");
    print_Hsub_eigs(elecinfo,elecvars,dft_global_log);
    dft_log("\n");
    dft_log_flush();

    /* If this is the last iteration, don't linmin, but
     * quit the iteration loop and exit the routine */
    if (iter==cntrl.max_elecs_steps-1) {
      if (! cntrl.if_e_converged(ener->Etot)) // check Etot converge.
	  cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
    }
    // If the energy is converged, break the loop.
    else if (cntrl.if_e_converged(ener->Etot)) {
	cntrl.print(dft_global_log,iter);
      cntrl.reset_converge();  // reset convergence counter
      iter = cntrl.max_elecs_steps;
    }
    // go ahead with linmin
    else {

      /* Calculate search direction */
      alpha = 0.0;
      /* If we're given the reset CG signal, keep alpha at zero and reset the
       * signal flag. */ 
      if (resetcg_signal == 1)
	resetcg_signal = 0;
      /* Otherwise, if we have done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      else if (iter>0 && fabs(linminfact) < 0.05)
	alpha = grad_now_dot_pgrad_now/grad_old_dot_pgrad_old;

      if (subspace_rotation)
	dft_log("|grad| = %le     |Ygrad| = %le    |Bgrad| = %le\n",
		sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now),
		sqrt(abs2_Bgrad_now));
      else 
	dft_log("|grad| = %le     |Ygrad| = %le\n",
		  sqrt(abs2_grad_now),sqrt(abs2_Ygrad_now));
      dft_log("alpha = %8.1le\n",alpha);

      /* Calculate current search direction:
       * dir = 2.0*pgrad_now + alpha*dir_old */
      for (k=0; k < nkpts; k++)
	{
	  /* On first iteration, zero out dir[k]. */
	  if (iter == 0)
	    Ydir[k].zero_out();
	  else 
	    Ydir[k] *= alpha;
	  Ydir[k] += pYgrad[k];
	  Ydir[k] += pYgrad[k];
	  if (subspace_rotation) {
	    if (iter == 0)
	      Bdir[k].zero_out();
	    else 
	      Bdir[k] *= (scalar)alpha;
	    Bdir[k] += pBgrad[k];
	  }
	}

      /* Do a linmin along dir */
      cntrl.elec_stepsize = do_linmin(Ydir,Bdir,
				      basis,ioninfo,elecinfo,elecvars,ener,
				      Ygrad,Bgrad,cntrl.elec_stepsize,
				      cntrl);

#ifdef DFT_PROFILING
      // Timing Report:
      timerReport(iter);
#endif // DFT_PROFILING

    }

    iter++;
  } while (iter < cntrl.max_elecs_steps);


#ifdef DFT_PROFILING
  timerOff(2); // turn off electron iteration timer
#endif // DFT_PROFILING

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,Ygrad);
  free_column_bundle_array(nkpts,Ydir);
  free_column_bundle_array(nkpts,pYgrad);
  if (subspace_rotation) {
    free_matrix_array(nkpts,Bgrad);
    free_matrix_array(nkpts,Bdir);
    free_matrix_array(nkpts,pBgrad);
  }

  /* return stepsize used */
  return cntrl.elec_stepsize;
}

