/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* 
 * Gabor Csanyi, June 1999
 * 
 * generalized I/O functions to handle serial and MPI binary I/O in a unified way
 * 
 * dft_fopen()
 * dft_fclose()
 * dft_fread() BUG: assumes data is type double
 * dft_fwrite()
 *
 */

/* $Id: dft_io.c,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#include <stdio.h>
#include "header.h"
#include "parallel.h"


/* 
 * dft_fopen() opens a file and returns the file pointer. arguments
 * are as in standard fopen(). If called in a serial context, it
 * simply executes a system fopen(). If called in an MPI context, it
 * executes a fopen() on the I/O node, and returns a NULL on all the
 * other nodes.
 */
FILE *
dft_fopen(char *filename, char *access_string)
{

  FILE *stream;

#ifdef DFT_MPI
  // we are running on an MPI system
  int ready_flag = 0;
  int my_id = System::Get_procID();
  int io_node = System::Get_IOprocID();

  if (my_id == io_node){
    stream = fopen(filename,access_string);

    // check if fopen was successful
    if (stream == NULL){
      // it wasnt... tell all processes to quit.
      MPI_Bcast ( &ready_flag, 1, MPI_INT, io_node, MPI_COMM_WORLD );
      die("Can't open %s in dft_fopen()\n",filename);
    }
    ready_flag = 1; // tell all processes that we are fine
    MPI_Bcast ( &ready_flag, 1, MPI_INT, io_node, MPI_COMM_WORLD );
    return(stream);
  }else{
    // Get signal from the I/O node
    MPI_Bcast ( &ready_flag, 1, MPI_INT, io_node, MPI_COMM_WORLD );

    if (ready_flag != 1) {
      // Error. Abort.
      die("Proc %d abort.\n", my_id);
    }

    return(NULL);
  }



#else //DFT_MPI

  // we are running in a serial context
  stream = fopen(filename,access_string);
  if (stream == NULL){
    die("Can't open %s in dft_fopen()\n",filename);
  }

  return(stream);



#endif //DFT_MPI

}


/* 
 * dft_fclose() closes the stream pointed to by its argument. if the
 * argument is NULL, it does nothing.
 *
 */
void
dft_fclose(FILE *stream)
{

#ifdef DFT_MPI
  // we are in an MPI context

  int my_id = System::Get_procID();
  int io_node = System::Get_IOprocID();

  if (my_id == io_node){
    fclose(stream);
  }

#else // DFT_MPI

  // we are in a serial context, so just close the stream
  fclose(stream);

#endif //DFT_MPI
}


/*
 * dft_fwrite() called in a scalar context just does an fwrite(). in an
 * MPI context, it does an fwrite if executed in the I/O node, and nothing
 * on any other node. returns the number of bytes written.
 */
size_t 
dft_fwrite(const void *data, size_t size, size_t nmemb, FILE *stream)
{
#ifdef DFT_MPI
  int my_id = System::Get_procID();
  int io_node = System::Get_IOprocID();

  if (my_id == io_node){
    return(fwrite(data, size, nmemb, stream));
  }else{
    return(0);
  }
#else // DFT_MPI
  // we are in a serial context

  return(fwrite(data, size, nmemb, stream));

#endif //DFT_MPI
}


/*
 * dft_fread() called in a scalar context just does an fread(). in an
 * MPI context, it does an fread if executed in the I/O node, and nothing
 * on any other node. then it broadcases the results, and returns the return vallue
 * of the system fread.
 */

size_t
dft_fread(void *data, size_t size, size_t nmemb, FILE *stream)
{
#ifdef DFT_MPI
  // we are in an MPI context

  // the problem is that dft_fread() is not aware of the type of the data
  // but the MPI broadcast needs that information. so at a minimum we assume
  // that its a double, and we trap if its not.
  if(size != sizeof(double)*SCALAR_SIZE)
    die("dft_fread() can only handle doubles at this point\n");
  
  int my_id = System::Get_procID();
  int io_node = System::Get_IOprocID();
  size_t items;

  if (my_id == io_node)
    items = fread(data, size, nmemb, stream);
  else
    items=0;

  // this assumes that the data is type double. TO BE FIXED
  MPI_Bcast ( data, nmemb*SCALAR_SIZE, MPI_DOUBLE, io_node, MPI_COMM_WORLD );

  return(items);
#else  // DFT_MPI
  // we are in a serial context
  return(fread(data, size, nmemb, stream));

#endif // DFT_MPI
}


// Flush the file if we're the IO node
int
dft_fflush(FILE *stream)
{
#ifdef DFT_MPI
  // we are in an MPI context
  if (System::Get_procID() == System::Get_IOprocID())
    return fflush(stream);
  else
    return 0;
#else // DFT_MPI
  // we are in a serial context, so just flush the stream
  return fflush(stream);
#endif //DFT_MPI
}


/* Does vfprintf */
int
dft_vfprintf(FILE *stream, char *format, va_list ap)
{
#ifdef DFT_MPI
  // we are in an MPI context
  if (System::Get_procID() == System::Get_IOprocID())
    return vfprintf(stream,format,ap);
  else
    return 0;
#else // DFT_MPI
  // we are in a serial context, so just flush the stream
  return vfprintf(stream,format,ap);
#endif //DFT_MPI
}
