/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *  Sohrab Ismail-Beigi,   Dec. 1996
 *
 * A simple complex number class for C++.  I tried to make it as efficient
 * as possible.
 *
 */

/* $Id: complex.h,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#ifndef DFT_COMPLEX_H
#define DFT_COMPLEX_H

#include <stdio.h>
#include <math.h>

class complex {

  /* data */
  public:
    real x,y;

 /* constructors */
  inline complex() { x = y = (real)0.0; }
  inline complex(const real x0) { x = x0; y = (real)0.0; }
  inline complex(const real x0,const real y0) { x = x0; y = y0; }
  inline complex(const complex &z) { x = z.x; y = z.y; } /* copy construc. */

  /* operators */
  inline void operator=(const complex &z1) { x = z1.x; y = z1.y; }
  inline void operator=(const real x1) { x = x1; y = (real)0.0; }
  inline complex operator-()  /* unary - */
    {
      complex z( -x, -y );
      return z;
    }
  inline complex operator+(const complex &z1)  /* z + z1 */
    {
      complex z( x + z1.x, y + z1.y );
      return z;
    }
  inline void operator+=(const complex &z1) /* z += z1 */
    {
      x += z1.x;
      y += z1.y;
/* for standard behavior, replace void by complex and uncomment: */
/*    return *this; */
    }
  inline complex operator-(const complex &z1)
    {
      complex z( x - z1.x, y - z1.y );
      return z;
    }
  inline void operator-=(const complex &z1) /* z -= z1 */
    {
      x -= z1.x;
      y -= z1.y;
    }
  inline complex operator*(const complex &z1) /* z*z1 */
    {
      complex z( x*z1.x - y*z1.y, x*z1.y + y*z1.x );
      return z;
    }
  inline complex operator*(const real s)    /* z*r */
    {
      complex z( x*s, y*s );
      return z;
    }
  inline friend complex operator*(const real &s,const complex &z1);   /* r*z */
  inline void operator*=(const real s)  /* z *= r */
    {
      x *= s;
      y *= s;
    }
  inline void operator*=(const complex &z1) /* z *= z1 */
    {
      real tx = x*z1.x - y*z1.y;
      y = x*z1.y + y*z1.x;
      x = tx;
    }
  inline complex operator/(const complex &z1) /* z/z1 */
    {
      real s = (real)1.0/(z1.x*z1.x + z1.y*z1.y);
      complex z( s*(x*z1.x + y*z1.y), s*(y*z1.x - x*z1.y) );
      return z;
    }
  inline complex operator/(const real s)   /* z/r */
    {
      real sinv = (real)1.0/s;
      complex z( x*sinv, y*sinv );
      return z;
    }
  inline friend complex operator/(const real s,const complex &z1);  /* r/z */

  /* functions */
  inline real abs2(void)  { return x*x+y*y; }
  inline real abs(void)   { return sqrt(x*x+y*y); }
  inline real phase(void) { return atan2(y,x); }
  inline complex conjugate(void)
    {
      complex z( x, -y );
      return z;
    }
  inline void auto_conjugate() { y = -y; }
  inline void set(real x0,real y0) { x = x0; y = y0; }
};

/* friend operator* overload to handle real*complex case */
inline complex operator*(const real &s,const complex &z1)
{
  complex z( z1.x*s, z1.y*s );
  return z;
}

/* friend operator/ overload to handle real/complex case */
inline complex operator/(const real s,const complex &z1)
{
  real soverz2 = s/(z1.x*z1.x+z1.y*z1.y);
  complex z( soverz2*z1.x, -soverz2*z1.y );
  return z;
}

/* Returns square magnitude of z */
inline real
abs2(const complex &z)
{
  return z.x*z.x + z.y*z.y;
}

/* Returns length of z */
inline real
abs(const complex &z)
{
  return sqrt(z.x*z.x+z.y*z.y);
}

/* Returns phase of z from -pi to pi */
inline real
phase(const complex &z)
{
  return atan2(z.y,z.x);
}

/* Returns the complex conjugate of z1 */
inline complex
conjugate(const complex &z1)
{
  complex z( z1.x, -z1.y );
  return z;
}

/* Exponentiates z1 */
inline complex
exp(const complex &z1)
{
  real s = exp(z1.x);
  complex z( s*cos(z1.y), s*sin(z1.y) );
  return z;
}

/* Returns the natural log of z1 with phase from -pi to pi */
inline complex
ln(complex &z1)
{
  complex z( (real)(0.5*log(z1.abs2())), z1.phase() );
  return z;
}

/* Returns sqrt of z1 with phase of z1 taken from -pi to pi */
inline complex
sqrt(complex &z1)
{
  real r = sqrt(z1.abs()),
    theta = (real)(0.5 * z1.phase());
  complex z( r*cos(theta), r*sin(theta) );
  return z;
}

/* Returns complex x+i*y */
inline complex
complex_xy(const real &x,const real &y)
{
  complex z( x, y );
  return z;
}
  
/* Returns complex r*exp(i*t) */
inline complex
complex_rt(const real &r,const real &t)
{
  complex z( r*cos(t), r*sin(t) );
  return z;
}

#endif // DFT_COMPLEX_H
