/*----------------------  column_bundle  --------------------------------*
 *                                                                       *
 * column_bundle class:  a collection of vectors composed of scalars--   *
 *                                                                       *
 * scalar_column's put into a larger overall structure (column_bundle.c) *
 *                                                                       *
 *-----------------------------------------------------------------------*/

/* $Id: column_bundle.h,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#ifndef DFT_COLUMN_BUNDLE_H
#define DFT_COLUMN_BUNDLE_H

class scalar_column
{
public:
  int n;
  scalar *c;
};

class column_bundle : public System
{
  /*************************************************
   **                                              **
   ** This data structure is one of the few that   **
   ** needs to know about the system architecture. **
   **                                              **
   *************************************************/
public:
  int start_ncol, end_ncol;
  int tot_ncols, my_ncols;
  int distributed;  // 1 stands for distributed object; 0 for local
  int col_length;
  scalar_column *col;
  vector *Vscloc;
  vector3 k;
  Basis *basis;

  /*
   * Comments:
   * Right now, the distributing strategy is static. This ensures that 
   * column_bundles of the same dimension will be distributed in the same 
   * way, thus making operations such as addition, substraction purly 
   * local.
   *
   * If some kind of load balancing is to be performed in the future,
   * special attention needs to be put here.
   */

  /* other data */

  /* Constructors and destructor */
  // constructors distribute the column-bundle into all processors,
  column_bundle();
  column_bundle(int nc,int len);
  column_bundle(int nc,int len,Basis *b);
  column_bundle(const column_bundle &Y); /* copy constructor */
  ~column_bundle(); /* destructor */
  /*
   * These constructors force each processor to create a local
   * copy instead of distributing the object.
   *
   * This is purly for the convenient of creating local temporary 
   * workspaces.
   *
   * WARNING!! other member functions might not work for objects
   * created this way!!
   */
  column_bundle(const char* str);
  column_bundle(int nc,int len,const char* str);
  column_bundle(int nc,int len,Basis *b,const char* str);
  column_bundle(const column_bundle &Y,const char* str); /* copy constructor */


  /* Operators */
  void operator=(const column_bundle &Y);    /* Nonstandard: returns void */
  void operator=(scalar s); /* sets all elements to s */
  friend column_bundle operator+(const column_bundle &Y1,
				 const column_bundle &Y2);
  void operator+=(const column_bundle &Y);
  friend column_bundle operator-(const column_bundle &Y1,
				 const column_bundle &Y2);
  void operator-=(const column_bundle &Y);
  friend column_bundle operator*(const column_bundle &Y, const matrix &M);
  friend column_bundle operator*(scalar s,const column_bundle &Y);
  friend column_bundle operator*(const column_bundle &Y,scalar s);
  void operator*=(scalar s);
  friend matrix operator^(const column_bundle &Y1,const column_bundle &Y2);

  /* member functions: */
  void distribute_cols();  // generate my_ncols, etc. from tot_ncols
  int ncol_to_ID(int col) const; // get processor ID from column number
  /* Do what the constructor does, but as a member function */
  void init(int nc,int len,Basis *b,int type);
  /* Free up the memory used by the structure by hand */
  void freemem(void);
  /* read and write column_bundles from/to binary format named files */
  void write(char *fname);
  void read(char *fname);
  /* read and write column_bunldes from/to binary format to already
   * opened IO streams...these functions don't close the streams. */
  void write_stream(FILE *stream);
  void read_stream(FILE *stream);
  /* zero out all the columns */
  void zero_out(void);
  /* Fill with random numbers */
  void randomize(void);
  /* Flips the sign of every entry in the column_bundle: i.e. y = (-1)*y */
  void negate(void);
};

/* Does YM = Y*M multiplication (operator* above) but with less memory use.
 * accum == 1 does YM += Y*M instead. */
void do_column_bundle_matrix_mult(const column_bundle &Y,
				  const matrix &M,
				  column_bundle &YM,
				  int accum);

/* Copies all the internal data of Y1 into Y2:  Y1 -> Y2 */
void copy_innards_column_bundle(const column_bundle *Y1,column_bundle *Y2);

/* Read/write an array of column_bundles from/to a file */
void read_column_bundle_array(char *fname,int nbundles,column_bundle *Y);
void write_column_bundle_array(char *fname,int nbundles,column_bundle *Y);

/* Allocate/free an array of column bundles */
column_bundle *alloc_column_bundle_array(int nbundles,int ncols,Basis* basis);
void free_column_bundle_array(int nbundles,column_bundle *Y);

/* Fill column_bundle_array with random numbers */
void randomize_column_bundle_array(int nbundles,column_bundle *Y);

/* Sum of absolute squares of all members in a column_bundle */
real abs2(const column_bundle &Y);
/* Sum of absolute squares of all members in an array of column_bundles */
real abs2(int nbundles,column_bundle *Y);
/* Take "dot-product" of two column_bundles:  sum the diagonals of Y1^Y2 */
scalar dot(const column_bundle &Y1,const column_bundle &Y2);
/* Take "dot-product" of nbundles pairs of column_bundles: i.e.
 * do dot_column_bundles does, but just loop over the pairs. */
scalar dot(int nbundles,column_bundle *Y1,column_bundle *Y2);

/* Does Yout += s*Yin */
void scale_accumulate(scalar s,const column_bundle &Yin,column_bundle &Yout);
void scale_accumulate(int nbundles,
		      scalar s,column_bundle *Yin,column_bundle *Yout);


#endif // DFT_COLUMN_BUNDLE_H
