/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *     Sohrab Ismail-Beigi           Mar. 30, 1997
 *
 * Calculates the gradient of the energy versus the electronic variables
 * Y (and also all the various routines needed to do this).
 *
 */

/* $Id: calcelecgrad.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include "header.h"

#include "parallel.h"

/*
 * Calculates the local part of the self-consistent potential.
 */
void
calc_Vscloc(Elecvars *evars)
{
#ifdef DFT_PROFILING
  timerOn(25);   // Turn on calc_Vscloc timer
#endif // DFT_PROFILING

  vector &Vscloc = evars->Vscloc;
  vector &Vlocps = evars->Vlocps;
  vector &d = evars->d;
  vector &n = evars->n;

  // enable non-linear core correction
  if (evars->ncore.n > 0)
    n += evars->ncore;  // add core charge to charge density.

  /*
  Vscloc = Jdag(Vlocps) + Jdag(O(d)) + Jdag(O(J(exc(n)))) + 
                 pointwise_mult(excprime(n),Jdag(O(J(n))));
		 */
  // Include options for GGA
  if (evars->ex_opt != DFT_EXCORR_GGA)
    Vscloc = Jdag(Vlocps+O(d)) + Jdag_O_J(exc(n)); // LDA by default
  else
    Vscloc = Jdag(Vlocps+O(d)) + Jdag_O_J(exc(n)+exGCprime(n));
  Vscloc += pointwise_mult(excprime(n),Jdag_O_J(n));

  if (evars->ncore.n > 0) 
    n -= evars->ncore;  // recover valence charge density.

#ifdef DFT_PROFILING
  timerOff(25);   // Turn off calc_Vscloc timer
#endif // DFT_PROFILING
}

/*
 * Calls calc_Vscloc() [above] and then
 * calculates and diagonalizes the subspace Hamiltonian.
 */
void
calc_Hsub(Ioninfo *ioninfo,
	  Elecinfo *einfo,Elecvars *evars)
{
  int k;
  /* Local copies to make the formulae easier to read */
  column_bundle *C = evars->C;
  matrix *Hsub = evars->Hsub;
  matrix *Hsub_evecs = evars->Hsub_evecs;
  real **Hsub_eigs = evars->Hsub_eigs;

  /* Local workspaces */
  /*
   * to avoid allocate/deallocate many times, create for largest
   * col_length. 
   */
  int max_col_length = 0;
  for (k=0; k < einfo->nkpts; k++)
    if (max_col_length < C[k].col_length)
      max_col_length = C[k].col_length;

  column_bundle HspC(C[0].tot_ncols,max_col_length);
  calc_Vscloc(evars);

  for (k=0; k < einfo->nkpts; k++)
    {
      // manually adjust column_bundle length;
      HspC.col_length = C[k].col_length;
      copy_innards_column_bundle(&(C[k]),&HspC);

      apply_Hsp(C[k],ioninfo,HspC);
      Hsub[k] = C[k]^HspC;
      Hsub[k].hermetian = 1;
      diagonalize_herm(Hsub_eigs[k],Hsub_evecs[k],Hsub[k],Hsub[k].nr);
    }
}

/*
 * Call calc_Vscloc() [above] and then
 * calculates the energy gradient versus Y and B
 * and calculates and diagonalizes the subspace Hamiltonian.
 */
void
calc_elecgrad_and_Hsub(column_bundle *Ygrad,
		       matrix *Bgrad,
		       Ioninfo *ioninfo,
		       Elecinfo *einfo,Elecvars *evars)
{

#ifdef DFT_PROFILING
  timerOn(10); // Turn on calc_elecgrad_... timer
#endif // DFT_PROFILING

  int k;
  /* Local copies to make the formulae easier to read */
  real *w = einfo->w;
  diag_matrix *F = einfo->F;
  column_bundle *C = evars->C;
  matrix *W = evars->W;
  real **mu = evars->mu;
  matrix *Umhalf = evars->Umhalf;
  matrix *Hsub = evars->Hsub;
  matrix *Hsub_evecs = evars->Hsub_evecs;
  real **Hsub_eigs = evars->Hsub_eigs;
  matrix *V = evars->V;
  matrix *Z = evars->Z;
  real **beta = evars->beta;

  /* Local workspaces */
  /*
   * to avoid allocate/deallocate many times, create for largest
   * col_length. 
   */
  int max_col_length = 0;
  for (k=0; k < einfo->nkpts; k++)
    if (max_col_length < C[k].col_length)
      max_col_length = C[k].col_length;

  column_bundle tmp(C[0].tot_ncols,max_col_length);

  calc_Vscloc(evars);
  for (k=0; k < einfo->nkpts; k++)
    {
      // manually adjust column_bundle length;
      tmp.col_length = C[k].col_length;
      copy_innards_column_bundle(&(C[k]),&tmp);

      /* Apply Hsp, calculate Hsub, and diagonalize Hsub */
      apply_Hsp(C[k],ioninfo,Ygrad[k]);
      Hsub[k] = C[k]^Ygrad[k];
      Hsub[k].hermetian = 1;

      diagonalize_herm(Hsub_eigs[k],Hsub_evecs[k],Hsub[k],Hsub[k].nr);

      /* Calc Y gradient */
/* The code/mess below does: */
/*       HspC = Hsp(C[k],ioninfo) */
/*       Ygrad[k]  = Pbar(C[k],HspC*(F[k]*V[k]*Umhalf[k])); */
/*       Ygrad[k] += O(C[k]*V[k]*Q(Vdag[k]*(Hsub[k]*F[k]-F[k]*Hsub[k])*V[k],W[k],mu[k])); */
/*       Ygrad[k] *= w[k]; */
      if (einfo->subspace_rotation)
	{
	  do_column_bundle_matrix_mult(Ygrad[k],F[k]*V[k]*Umhalf[k],tmp,0);
	  apply_Pbar(C[k],tmp,Ygrad[k]);
	  do_column_bundle_matrix_mult(
	     C[k],
	     V[k]*Q(herm_adjoint(V[k])*(Hsub[k]*F[k]-F[k]*Hsub[k])*
		               V[k],W[k],mu[k]),
	     tmp,0);
      } else {
	do_column_bundle_matrix_mult(Ygrad[k],F[k]*Umhalf[k],tmp,0);
	apply_Pbar(C[k],tmp,Ygrad[k]);
	do_column_bundle_matrix_mult(
		     C[k],
		     Q(Hsub[k]*F[k]-F[k]*Hsub[k],W[k],mu[k]),
		     tmp,0);
      }
      apply_O_inplace(tmp);
      Ygrad[k] += tmp;
      Ygrad[k] *= w[k];

      if (einfo->subspace_rotation) {
	/* Calc B gradient and symmetrize it */
	/* Bgrad[k] = R([Hsub[k],F[k]]) */
	Bgrad[k] = R((Hsub[k]*F[k]-F[k]*Hsub[k]),Z[k],beta[k]);
	Bgrad[k] += herm_adjoint(Bgrad[k]);
	Bgrad[k] *= (scalar)(0.5*w[k]);
      }
    }
#ifdef DFT_PROFILING
  timerOff(10); // Turn off calc_elecgrad_... timer
#endif // DFT_PROFILING
}

/*
 * Does what the above routine does plus preconditioning the gradient.
 * The preconditioning is diagonal inverse kinetic for the plane-wave basis.
 * Namely, the average kinetic energy per band is calculated 
 * (via ener->KE/einfo->nelectrons) and this (times 2.0) is used
 * as the roll-over value of the kinetic energy for the preconditioning.
 */
void
calc_elecgrad_pgrad_and_Hsub(column_bundle *Ygrad,
			     column_bundle *pYgrad,
			     matrix *Bgrad,
			     matrix *pBgrad,
			     Energies *ener,
			     Ioninfo *ioninfo,
			     Elecinfo *einfo,
			     Elecvars *evars)
{
#ifdef DFT_PROFILING
      timerOn(10); // Turn on calc_elecgrad_... timer
#endif // DFT_PROFILING

  int k;
  /* Local copies to make the formulae easier to read */
  real *w = einfo->w;
  diag_matrix *F = einfo->F;
  column_bundle *C = evars->C;
  column_bundle *Y = evars->Y;
  matrix *W = evars->W;
  real **mu = evars->mu;
  matrix *Umhalf = evars->Umhalf;
  matrix *Hsub = evars->Hsub;
  matrix *Hsub_evecs = evars->Hsub_evecs;
  real **Hsub_eigs = evars->Hsub_eigs;
  matrix *V = evars->V;
  matrix *Z = evars->Z;
  real **beta = evars->beta;

  /* Local workspaces */
  /*
   * to avoid allocate/deallocate many times, create for largest
   * col_length. 
   */
  int max_col_length = 0;
  for (k=0; k < einfo->nkpts; k++)
    if (max_col_length < C[k].col_length)
      max_col_length = C[k].col_length;

  column_bundle tmp(C[0].tot_ncols,max_col_length);

  /* Roll-over value for kinetic preconditioning:  2.0*average kinetic
   * energy per band. */
  real KErollover;
  KErollover = 2.0*ener->KE/einfo->nelectrons;

  calc_Vscloc(evars);
  for (k=0; k < einfo->nkpts; k++)
    {
      // manually adjust column_bundle length;
      tmp.col_length = C[k].col_length;
      copy_innards_column_bundle(&(C[k]),&tmp);

      /*
       * In this wonderful mess below, we calculate the subspace Hamiltonian,
       * the gradient, and the preconditioned gradient!  The formulae:
       *  C[k] = Y[k]*Umhalf[k]*Vdag[k]
       *  Hsub[k] = C[k]*Hsp(C[k])
       *  Ygrad[k] = w[k]*(g1 + g2)
       *  g1 = Pbar(Hsp(C[k]*F[k]*V[k]*Umhalf[k]))
       *     = Pbar(Hsp(Y[k]*Umhalf[k]*Vdag[k]*F[k]*V[k]*Umhalf[k]))
       *  g2 = O(C[k]*V[k]*Q(Vdag[k]*[Hsub[k],F[k]]*V[k]))
       *  pYgrad[k] = w[k]*(Pbar(precond(Pbar(Hsp(Y[k])))) + g2)
       */

      /* First we calculate Hsp(Y[k]), Hsp(Y[k])*Umhalf[k]*Vdag[k]=Hsp(C[k]), and Hsub[k] */

      apply_Hsp(Y[k],ioninfo,pYgrad[k]);
      if (einfo->subspace_rotation)
	do_column_bundle_matrix_mult(pYgrad[k],
				     Umhalf[k]*herm_adjoint(V[k]),
				     tmp,0);
      else
	do_column_bundle_matrix_mult(pYgrad[k],Umhalf[k],tmp,0);
      Hsub[k] = C[k]^tmp;

      /* Digonalize Hsub[k] */
      Hsub[k].hermetian = 1;
      diagonalize_herm(Hsub_eigs[k],Hsub_evecs[k],Hsub[k],Hsub[k].nr);

      /* Now calculate g1 and the first term of pYgrad[k] */
      apply_Pbar(C[k],pYgrad[k],tmp);
      if (einfo->subspace_rotation)
	do_column_bundle_matrix_mult(
		   tmp,
	           Umhalf[k]*herm_adjoint(V[k])*F[k]*V[k]*Umhalf[k],
		   Ygrad[k],0);
      else
	do_column_bundle_matrix_mult(tmp,Umhalf[k]*F[k]*Umhalf[k],Ygrad[k],0);
      precond_inv_kinetic(tmp,KErollover);
      apply_Pbar(C[k],tmp,pYgrad[k]);

      /* Add in g2 = O*C*Q() term to Ygrad[k] and pYgrad[k] */
      if (einfo->subspace_rotation)
	do_column_bundle_matrix_mult(
		     C[k],
		     V[k]*Q(herm_adjoint(V[k])*(Hsub[k]*F[k]-F[k]*Hsub[k])*
                              V[k],W[k],mu[k]),
		     tmp,0);
      else
	do_column_bundle_matrix_mult(C[k],
				     Q(Hsub[k]*F[k]-F[k]*Hsub[k],W[k],mu[k]),
				     tmp,0);

      apply_O_inplace(tmp);
       Ygrad[k] += tmp;
      pYgrad[k] += tmp;

      /* scale by weights */
       Ygrad[k] *= w[k];
      pYgrad[k] *= w[k];

      /* Calc B gradient and symmetrize it */
      /* Bgrad[k] = R([Hsub[k],F[k]]) */
      if (einfo->subspace_rotation) {
	Bgrad[k] = R((Hsub[k]*F[k]-F[k]*Hsub[k]),Z[k],beta[k]);
	Bgrad[k] += herm_adjoint(Bgrad[k]);
        Bgrad[k] *= (0.5 * w[k]);
	pBgrad[k] = ((scalar) einfo->pcond_for_Bgrad) * Bgrad[k];
      }
    }

#ifdef DFT_PROFILING
      timerOff(10); // Turn off calc_elecgrad_... timer
#endif // DFT_PROFILING

}

/*
 * Prints the eigenenergies of the subspace Hamiltonian for each
 * k-point.
 */
void
print_Hsub_eigs(Elecinfo *einfo,Elecvars *evars,Output *out)
{
  int k,i;

  out->printf("Band energies:\n");
  for (k=0; k < einfo->nkpts; k++)
    {
      out->printf("\nkpt = %d   k = [ %lg %lg %lg ]   w = %lg\n",
		k,einfo->kvec[k].v[0],einfo->kvec[k].v[1],einfo->kvec[k].v[2],
		einfo->w[k]);
      out->printf("%4s  %13s  %13s  %13s\n",
		"band","filling   ","diag(Hsub) ","epsilon   ");
      out->printf("-------------------------------------------------\n");
      for (i=0; i < evars->Hsub[k].nr; i++)
	out->printf("%4d  %13.6le  %13.6le  %13.6le\n",
		  i,REAL(einfo->F[k].c[i]),
		  REAL(evars->Hsub[k](i,i)),evars->Hsub_eigs[k][i]);
    }
  out->flush();
}
