/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *     Sohrab Ismail-Beigi           Mar. 29, 1997
 *
 * Calculate the U matrices, the C column_bundles, and charge density n.
 *
 */

/* $Id: calcUVCn.c,v 1.2 1999/12/19 16:31:54 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include "header.h"

/*
 * Calculate the U matrices and their various powers:
 *
 *                       U[k] = Y[k]^O(Y[k])
 */
void
calc_U(Elecinfo *einfo,Elecvars *evars)
{
#ifdef DFT_PROFILING
  timerOn(14);   // Turn on calc_U timer
#endif // DFT_PROFILING

  int k;

  /* Get local pointer to the relevant variables to make the formulae
   * legible */
  column_bundle *Y = evars->Y;
  matrix *U = evars->U;
  matrix *W = evars->W;
  real **mu = evars->mu;
  matrix *Umhalf = evars->Umhalf;

  /* temporary workspace */
  int max_col_length = 0;
  for (k=0; k < einfo->nkpts; k++)
    if (max_col_length < Y[k].col_length)
      max_col_length = Y[k].col_length;

  column_bundle OY(Y[0].tot_ncols,max_col_length);

  for (k=0; k < einfo->nkpts; k++)
    {
      // manually adjust col_length;
      OY.col_length = Y[k].col_length;
      copy_innards_column_bundle(&(Y[k]),&OY);

/* does:  U[k] = Y[k]^(O(Y[k])); */
      apply_O(Y[k],OY);
      U[k] = Y[k]^OY;
      // U is hermetian, so make it so!
      U[k].hermetian = 1;
      {
	int i,j,flag;

	flag = 0;

	for (j=0; j < Y[k].my_ncols; j++)
	  for (i=0; i < Y[k].col_length; i++)
	    if ( isnan(Y[k].col[j].c[i].x) || isnan(Y[k].col[j].c[i].y) )
	      {
		flag = 1;
		dft_log("&&&&&&&&&&& Y[%d](%d,%d) has NaN ",k,i,j);
		if (isnan(Y[k].col[j].c[i].x))
		  dft_log("r");
		if (isnan(Y[k].col[j].c[i].y))
		  dft_log("i");
		dft_log("\n");
	      }

	if (flag)
	  die("\n\nProgram is quitting due to NaNs!!!\n\n");
      }

      Umhalf[k] = Uminusonehalf(U[k],W[k],mu[k]);
    }

#ifdef DFT_PROFILING
  timerOff(14);   // Turn off calc_U timer
#endif // DFT_PROFILING

}

/*
 * Calculate the subspace rotation matrix  :  V[k] = exp(iB[k]) 
 */
void
calc_V(Elecinfo *einfo,Elecvars *evars)
{
  int k,i;

  /* If we aren't doing subspace rotations, then we should do nothing! */
  if (!einfo->subspace_rotation)
    return;

  /* Get pointers to the data */
  matrix *V = evars->V;
  matrix *B = evars->B;
  matrix *Z = evars->Z;
  real **beta = evars->beta;

  diag_matrix expibeta(B[0].nr);

  for (k=0; k < einfo->nkpts; k++)
    {
      /* B[k] = Z[k]*beta[k]*Zdag[k] */
      diagonalize_herm(beta[k],Z[k],B[k],B[k].nr);
      /* V[k] = Z[k]*exp(i*beta[k])*Zdag[k] */
      for (i=0; i < B[k].nr; i++)
	{
	  register complex ibeta;

	  ibeta.x = 0.0;
	  ibeta.y = beta[k][i];
	  expibeta.c[i] = exp(ibeta);
	}
      V[k] = Z[k]*(expibeta*herm_adjoint(Z[k]));
    }
}

/*
 * Calculate the orthonormal wavefunctions:
 *
 * C[k] = Y[k]*Umhalf[k]*Vdag[k]
 */
void
calc_C(Elecinfo *einfo,Elecvars *evars)
{
#ifdef DFT_PROFILING
  timerOn(15);   // Turn on calc_C timer
#endif // DFT_PROFILING

  int k;

  /* Get local pointer to the relevant variables to make the formulae
   * legible */
  column_bundle *Y = evars->Y;
  column_bundle *C = evars->C;
  matrix *Umhalf = evars->Umhalf;
  matrix *V = evars->V;

  if (einfo->subspace_rotation)
    for (k=0; k < einfo->nkpts; k++)
      {
	/*       C[k] = Y[k]*Umhalf[k]; */
	do_column_bundle_matrix_mult(Y[k],
				     Umhalf[k]*herm_adjoint(V[k]),
				     C[k],0);
      }
  else
    for (k=0; k < einfo->nkpts; k++)
      {
	/*       C[k] = Y[k]*Umhalf[k]; */
	do_column_bundle_matrix_mult(Y[k],
				     Umhalf[k],
				     C[k],0);
      }

#ifdef DFT_PROFILING
  timerOff(15);   // Turn off calc_C timer
#endif // DFT_PROFILING
}

/* Combines U, V, and C */
void
calc_UVC(Elecinfo *einfo,Elecvars *evars)
{
  calc_U(einfo,evars);
  calc_V(einfo,evars);
  calc_C(einfo,evars);
}

/* Calculate electron density: n = sum_k { w[k]*diagouter(F[k],I*C[k]) } */
void
calc_n(Elecinfo *einfo, Elecvars *evars, Ioninfo *ioninfo)
{
#ifdef DFT_PROFILING
  timerOn(16);   // Turn on calc_n timer
#endif // DFT_PROFILING

  int k;

  /* Get local pointer to the relevant variables to make the formulae
   * legible */
  diag_matrix *F = einfo->F;
  real *w = einfo->w;
  column_bundle *C = evars->C;
  vector &n = evars->n;   /* reference */

  n.zero_out();
  for (k=0; k < einfo->nkpts; k++) {
    n += ((scalar)w[k])*diagouterI(F[k],C[k]);
  }

  // symmetrize the charge density.
  symmetrize_n(n, ioninfo->nrot, ioninfo->n_sym, ioninfo->done);

  // sum of n  should be  number of electrons per unit cell.
  real sum = REAL(sum_vector(n));
  sum = sum * n.basis->unit_cell_volume / n.n;
  if ( fabs(sum/einfo->nelectrons-1) > 1e-4 ) {
    dft_log(">>>>Real space charge integrate to %g != %g\n",
	    sum, einfo->nelectrons);
    die("Quit due to violation of charge conservation.");
  }

#ifdef DFT_PROFILING
  timerOff(16);   // Turn off calc_n timer
#endif // DFT_PROFILING
}

/* Combines U, V, C and n */
void
calc_UVCn(Elecinfo *einfo,Elecvars *evars, Ioninfo *ioninfo)
{
  calc_U(einfo,evars);
  calc_V(einfo,evars);
  calc_C(einfo,evars);
  calc_n(einfo,evars,ioninfo);
}

