/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *            Tairan Wang                        January 8, 1998
 * 
 * System.c -- provide MPI setup.
 *
 *
 */

/* $Id: System.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>

#include "header.h"
#include "parallel.h"   // include MPI related info.

#include <time.h>


///////////////////////////////
//                           //
// Define Static Data Member //
//                           //
///////////////////////////////
int System::N_Procs = 0;
int System::My_procID = 0;
Output * System::global_log = NULL;

//////////////////////////////////////////////
/*                                          *
 * Static Member Function. Initialize       *
 * system configuration info.               *
 *                                          *
 * Must be called before any object of the  *
 * class is actually created                *
 *                                          */
//////////////////////////////////////////////

void
System::GlobalInit(int* argc, char*** argv)
{
#ifdef DFT_MPI
  int id, num_procs;

  MPI_Init(argc, argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &id);
  MPI_Comm_size(MPI_COMM_WORLD, &num_procs);

  System::N_Procs = num_procs;
  System::My_procID = id;
#else   // DFT_MPI
  System::N_Procs = 1;
  System::My_procID = 0;
#endif  // DFT_MPI

}

void
System::GlobalFinalize()
{
#ifdef DFT_MPI
  MPI_Finalize();
#endif // DFT_MPI
}



// This class is for system-info storage.
// No object of this class should ever be created.
System::System()
{
  ;
}

// platform-dependent timer
double
System::get_time(void)
{
#ifdef DFT_MPI
  return MPI_Wtime();
#else
  return (double) time((time_t *)0);
#endif
}

// platform-dependent seeding of random numbers.
// Seed the system random number generator with the current time.
void
System::seed_with_time(int flag)
{
  int seed;

  if (flag == DFT_SMALL_DATA) {

#if defined DFT_MPI    
    if (System::Get_procID() == System::Get_IOprocID())
      seed = (int)System::get_time();
    MPI_Bcast(&seed,1,MPI_INT,System::Get_IOprocID(),MPI_COMM_WORLD);
#else
    seed = (int)System::get_time();
#endif // DFT_MPI    
    srand(seed);

  } else {
    // must be DFT_LARGE_DATA
    srand((unsigned int)System::get_time()+
	  System::Get_procID()+
	  (unsigned int)System::get_time()/(System::Get_procID()+1) );
    /* if running on parallel platform, this will make sure that each processor 
     * gets a different seed so that random numbers won't be the same across
     * processors. */
  }
}




/* Bite the dust with some last words */
void
die(const char *last_words, ...)
{

  /* variable argument list */
  va_list ap;

  va_start(ap, last_words);

#ifdef DFT_MPI

  if(System::Get_procID() == System::Get_IOprocID() )
    {
      System::global_log->vprintf(DFT_SILENCE, last_words, ap);
      System::global_log->flush();
    }
  MPI_Abort(MPI_COMM_WORLD, 1);
  exit(1);

#else // DFT_MPI

  System::global_log->vprintf(DFT_SILENCE, last_words, ap);
  System::global_log->flush();
  exit(1);

#endif // DFT_MPI


}
