/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* $Id: Speciesinfo.c,v 1.3 1999/12/03 20:57:34 tairan Exp $ */

#include "header.h"
#include <stdio.h>
#include <math.h>

//
// Implement member functions of Speciesinfo
//
void
Speciesinfo::print() const
{
  int i;
  dft_log(">> Species %s:\n",name);
  dft_log(">>  Z = %f\tmass = %f\tnatoms = %d\n",
	  Z, mass, natoms);
  for (i=0; i < natoms; i++) {
    dft_log(">>  %d [ %f %f %f ] %d\n",
	    i, atpos[i].v[0], atpos[i].v[1], atpos[i].v[2],
	    move[i]);
  }
}

void Speciesinfo::print_force(const matrix3 &invR) const
{
  int i;
  matrix3 invRT = ~invR; // inverse transpose of R (the lattice vector matrix)
  vector3 forcetemp;
  
  dft_log("\nspecies(%s) = %d, natoms = %d,  forces follow (Hartrees/Bohr):\n\n", 
	  name, sp_id, natoms);
  for (i=0; i < natoms; i++) {
    forcetemp = invRT * forces[i];
    dft_log("%3d %4d %20.10le %20.10le %20.10le %d\n",
	    sp_id,i,
	    forcetemp.v[0], forcetemp.v[1], forcetemp.v[2],
	    move[i]);
  }
  dft_log("\n");
}

void
Speciesinfo::setup_pulay(const char *pulayfilename, 
   		         real einfoEcut)
{
  /* If Pulay correction filename is "none", then set the
   * Pulay correction to zero and skip reading the file */
  if ( strcmp(pulayfilename,"none") == 0 ) {
    dft_log("Pulay file is specified as 'none':\n");
    dft_log("setting Pulay correction to zero.\n");
    dEperNatoms_dNGperVol = 0.0;
  } else {
    dft_text_FILE *pulayfile;

    /* Read file of Pulay correction factors for the species */
    if ( (pulayfile = dft_text_fopen(pulayfilename,"r")) == (dft_text_FILE *)0)
      die("Can't open pulay file for reading.\n");
    dft_log("Reading pulay file %s\n",pulayfilename);

    /*
     * Read number of energy cutoff values listed in file
     */
    int NEcuts,n,flag;
    real Ecut;
    char line[DFT_LINE_LEN];

    skip_comment(pulayfile, line);
    sscanf(line,"%d",&NEcuts);
    dft_log("NEcuts=%d\n",NEcuts);
    flag = 0;
    for (n=0; n < NEcuts; n++) {
      skip_comment(pulayfile, line);
      sscanf(line,"%lg %lg",
	     &Ecut,&dEperNatoms_dNGperVol);
      dft_log("Ecut = %g     dEperNatoms_dNGperVol = %g\n",
		Ecut,dEperNatoms_dNGperVol);
      if (Ecut == einfoEcut) {
	dft_log("Using Ecut=%g as it agrees with Elecinfo Ecut\n",Ecut);
	flag = 1;
	break;
      }
    }
    if (flag==0)
      die("\nCan't find appropriate energy cutoff in Pulay file!!!\n\n");
  }
}
