/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* $Id: Ioninfo.c,v 1.4 2000/01/25 04:53:44 tairan Exp $ */

#include "header.h"
#include <stdio.h>
#include <math.h>

//
// Implement member functions of Ioninfo
//
void Ioninfo::print() const
{
  int sp;

  dft_log(">> Display Ioninfo internals:\n");
  dft_log(">> Found %d species:\n",nspecies);
  for (sp = 0; sp < nspecies; sp++)
    species[sp].print();
}


/*
 * Make sure no atoms are layed on top of each other.
 */
void Ioninfo::check_pos()
{
  int dieflag = 0;
  real sizetest = 0;
  vector3 vtest[2];

  for (int sp=0; sp < nspecies; sp++)
    for (int n=0; n < species[sp].natoms; n++) {
      vtest[0] = species[sp].atpos[n];
      for (int sp1 = sp; sp1 < nspecies; sp1++)
	for (int n1 = ((sp1==sp) ? (n+1) : 0);
	     n1 < species[sp1].natoms; n1++) {
	  vtest[1] = vtest[0] - species[sp1].atpos[n1];
	  sizetest = vtest[1] * vtest[1];
	  if (sizetest < MIN_ION_DISTANCE) {
	    dieflag = 1;
	    dft_log(">>Atoms' positions coincide!!\n");
	    dft_log(">>(sp %d : at %d) = (sp %d : at %d) : ",
		       sp, n, sp1, n1);
	    species[sp].atpos[n].print(dft_global_log," %f ");
	  }
	}
    }

  if(dieflag != 0)
    die(">> You put atoms on top of each other, you moron. I have to quit now. Bye\n");
}


/* Maxim added on 11/1/99, subroutine used to check
 *  whether an atom [at] of species [sp] is constrained to move along a line.
 */
int Ioninfo::check_line_constraint(int sp, int at) const
{
  if (number_of_line_constraints == 0)
    return -1;
  else {
    int i;
    for (i=0; i<number_of_line_constraints; i++) {
      if ((line_constraint_n_sp[i]==sp)&&(line_constraint_n_at[i]==at))
	return i;
    }
    return -1;
  }
}

void Ioninfo::constrain_forces_to_lines()
{
  int i, sp, nat;

  // Maxim added on 11/1/99 to implement line constraints on ion dynamics
  dft_log("number of line constraints: %d\n", number_of_line_constraints);
  for (i = 0; i < number_of_line_constraints; i++) {
    sp = line_constraint_n_sp[i];
    nat = line_constraint_n_at[i];
    line_constraint_forces[i] = species[sp].forces[nat];
    species[sp].forces[nat] = 
      ( species[sp].forces[nat] 
	* line_constraint_direction[i] )
      * line_constraint_direction[i];
  }
}

int Ioninfo::symmetrize_force()
{
  if (nrot <= 1) // no need to do any symmetrization
    return 0;

  // for each atom, each symmetry maps it to one of its own species.
  int sp, nat, irot, natoms; 
  vector3 *forces_tmp, *forces;
  for (sp = 0; sp < nspecies; sp++) {
    natoms = species[sp].natoms;
    forces = species[sp].forces;
    forces_tmp = (vector3*)mymalloc(sizeof(vector3)*natoms,"forces_tmp","symmetrize_force");
    for (nat = 0; nat < natoms; nat++)
      forces_tmp[nat] = 0;
    for (nat = 0; nat < natoms; nat++)
      for (irot = 0; irot < nrot; irot++)
	forces_tmp[species[sp].maps[irot][nat]] += f_sym[irot] * forces[nat];
    for (nat = 0; nat < species[sp].natoms; nat++)
      forces[nat] = forces_tmp[nat] / nrot;
    myfree(forces_tmp);
  }
  return 1;

}


void Ioninfo::print_force(const matrix3 &invR) const
{
  int sp, nat;
  int idir;

  for (sp=0; sp < nspecies; sp++) {
    species[sp].print_force(invR);
    if (number_of_line_constraints > 0) {
      dft_log("\nForces on constrained atoms\n");
      for (sp=0; sp < nspecies; sp++)
	for (nat=0; nat < species[sp].natoms; nat++)
	  if ((idir=check_line_constraint(sp,nat))!=(-1))
	    dft_log("%3d %4d %20.10le %20.10le %20.10le %d\n",
		    sp,nat,
		    line_constraint_forces[idir].v[0],
		    line_constraint_forces[idir].v[1],
		    line_constraint_forces[idir].v[2]);
    }
  }
}
