/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* $Id: Elecinfo.c,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#include "header.h"
#include <stdio.h>
#include <math.h>

/*=================================================================*
 *                                                                 *
 * Implement the member function of Elecinfo                       *
 *                                                                 *
 *=================================================================*/
Elecinfo::Elecinfo()
{
  Ecut = 0.0;
  nkpts = nbands = 0;
  nelectrons = 0.0;

  calc_fillings = DFT_FALSE;
  niter_recalc_fillings = 0;
  kT = mu = 0.0;
  
  subspace_rotation = DFT_FALSE;
  pcond_for_Bgrad = 0.0;
  
  ex_opt = DFT_EXCORR_LDA;

  nl_core_flag = DFT_FALSE;

  kvec = NULL;
  w = NULL;
  F = NULL;

  initial_fillings_flag = 0;
  strcpy(initial_fillings_filename,"<fillings-file>"); // default name
}


void
Elecinfo::print() const
{
  int i;

  dft_log(">> Displaying Elecinfo internals:\n");
  dft_log(">> Ecut = %f\n",Ecut);
  dft_log(">> nelectrons = %f\n",nelectrons);
  dft_log(">> nbands = %d\tnkpts = %d\n",nbands,nkpts);
  if (( kvec != NULL ) && ( w != NULL ))
    for (i = 0; i < nkpts; i++) {
      dft_log(">>  %d [ %f %f %f ] %f\n",
		i, kvec[i].v[0], kvec[i].v[1], kvec[i].v[2], w[i]);
      dft_log(">> ");
      F[i].print();
    }
}

// Sets up the fillings of the bands depending
// on the value of the initial_fillings_flag:
// 0 = set them up automatically to be 2.0 or 0.0 (so the total=nelectrons)
// 1 = read them from the file initial_fillings_file
void
Elecinfo::setup_initial_fillings(Basis *basis)
{
  int k, b;
  dft_text_FILE *fillings_FILE = NULL;

  // allocate the fillings matrices.
  F = (diag_matrix *)mymalloc(sizeof(diag_matrix)*nkpts,
			      "F","Elecinfo::setup_initial_fillings");

  // If we have to read the fillings from a file, open the file first
  if (initial_fillings_flag == 1)
    {
      dft_log("Reading initial fillings from file %s.\n",
	      initial_fillings_filename);
      fillings_FILE = dft_text_fopen(initial_fillings_filename,"r");
      if (fillings_FILE == NULL)
	die("Can't open file %s to read initial fillings!\n",
	    initial_fillings_filename);
    }
  // otherwise, automatic fillings are being performed, so get # of electrons
  else
    dft_log("Computing initial fillings automatically.\n");
  
  // Fill in the fillings!
  for (k = 0; k < nkpts; k++)
    {
      real ne = nelectrons;
      F[k].init(nbands, &basis[k]);
      for (b = 0; b < nbands; b++)
	{
	  real fnk;
	  // either read fnk from file...
	  if (initial_fillings_flag == 1)
	    dft_text_fscanf(fillings_FILE,"%lg",&fnk);
	  // or calculate what is should be so as to fill up # of electrons
	  else
	    {
	           if (ne > 2.0) fnk = 2.0;
	      else if (ne > 0.0) fnk = ne;
	      else               fnk = 0.0;
	      ne -= fnk;
	    }
	  F[k].c[b] = fnk;
	}
    }
  // If the file is open, close it.
  if (initial_fillings_flag == 1)
    dft_text_fclose(fillings_FILE);
}

/*
 * Elecinfo::print_fillings : 
 *
 * Print out filling info:
 */
void
Elecinfo::print_fillings(Output *out) const
{ 
  dft_log("Dumping latest fillings to '%s'\n", out->filename);
  dft_log_flush();

  int k,b;
  for (k=0; k < nkpts; k++)
    {
      for (b=0; b < nbands; b++) 
	out->printf("%16.10le ",REAL(F[k].c[b]));
      out->printf("\n");
    }
}
