/*
 *     Nikolaj Moll                 April 10, 1999
 *
 * Add support for symmetry calculation.
 */

/* $Id: symm.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "header.h"

// Maxim, 10/13/1999, hack this to get it compiled in T3E.
int rint(float fnumber)
{
  int nnumber;
  nnumber=(int)fnumber;
  if ((fnumber-nnumber)<0.5) return nnumber;
  else return (nnumber+1);
}


/*
 * symmetry recongnition of the ionic configuration
 */
void 
symmetries(Ioninfo *ioninfo, Basis *basis)
{
  int sp, n1, n2, i;
  int bnrot = 0, tnrot = 0;
  vector3 r, tr;
  matrix3 bsym[48], tsym[48], tmp, identity(1.0, 1.0, 1.0);

  ioninfo->nrot = 0;

  dft_log("\nautomatic search of point group symmetries:\n");
  
  /* find symmetries of bravais lattice */
  bravais_symmetries(basis, bnrot, bsym);

  /* first find symmetries without translation */
  basis_symmetries(ioninfo, bnrot, bsym, r, ioninfo->nrot, ioninfo->sym);

  /* use the atomic positions and the center of two atoms of the same species 
   * as symmetry point */
  for (sp = 0; sp < ioninfo->nspecies; sp++) {
    for (n1 = 0; n1 < ioninfo->species[sp].natoms; n1++) {
      for (n2 = 0; n2 <= n1; n2++) {
	tr = (ioninfo->species[sp].atpos[n1]
	     + ioninfo->species[sp].atpos[n2])/2.0;
	/* find subgroup of symmetries */
	basis_symmetries(ioninfo, bnrot, bsym, tr, tnrot, tsym);
	if (tnrot > ioninfo->nrot) {
	  r = tr;
	  ioninfo->nrot = tnrot;
	  for (i = 0; i < ioninfo->nrot; i++)
	       ioninfo->sym[i] = tsym[i];
	}
      }
    }
  }
  
  /* sort, so identity is the first matrix */ 
  for (i = 0; i < ioninfo->nrot; i++)
    if (matcmp(ioninfo->sym[i], identity) < MIN_SYMM_TOL) {
      tmp = ioninfo->sym[0];
      ioninfo->sym[0] = ioninfo->sym[i];
      ioninfo->sym[i] = tmp;
    }

  /* print out the symmetries */
  dft_log("reduced to %d total symmetries with basis:\n\n", ioninfo->nrot);
  for (i = 0; i < ioninfo->nrot; i++)
    {
      ioninfo->sym[i].print(dft_global_log,"%4.0f ");
      dft_log("\n");
    }
  dft_log_flush();

  /* move atoms to symmetry point */
  if (abs(r) > MIN_SYMM_TOL) {
    dft_log("moving atoms to new symmetry point: ");
    r.print(dft_global_log,"%g ");
    for (sp = 0; sp < ioninfo->nspecies; sp++) {
      for (n1 = 0; n1 < ioninfo->species[sp].natoms; n1++) {
	ioninfo->species[sp].atpos[n1] -= r;
      }
    }
  }
}

void 
bravais_symmetries(Basis *basis, int &bnrot, matrix3 *bsym)
{
  int i; 
  matrix3 a, g, m, mgm, t(1.0, 1.0, 1.0), identity(1.0, 1.0, 1.0);
  
  /* transpose lattice vector matrix */
  a = ~basis->latvec;

  /* find reduced basis and transmission matrix t */
  minimize_basis(a, t);

  /* print transmission matrix if it is not equal to the identity */
  if (matcmp(t, identity) > MIN_SYMM_TOL) {
    dft_log("\ntransmission matrix:\n\n");
    t.print(dft_global_log,"%4.0f ");
    dft_log("\nwith the corresponding reduced lattice vectors:\n\n");
    (~a).print(dft_global_log,"%12.6f ");
  }

  /* calculate metric */
  g = (~a)*a;

  /* test all possible symmetries */
  bnrot = 0;
  for (m.m[0][0] = -1.0; m.m[0][0] <= 1.0; m.m[0][0]++)
    for (m.m[0][1] = -1.0; m.m[0][1] <= 1.0; m.m[0][1]++)
      for (m.m[0][2] = -1.0; m.m[0][2] <= 1.0; m.m[0][2]++)
	for (m.m[1][0] = -1.0; m.m[1][0] <= 1.0; m.m[1][0]++)
 	  for (m.m[1][1] = -1.0; m.m[1][1] <= 1.0; m.m[1][1]++)
	    for (m.m[1][2] = -1.0; m.m[1][2] <= 1.0; m.m[1][2]++)
	      for (m.m[2][0] = -1.0; m.m[2][0] <= 1.0; m.m[2][0]++)
		for (m.m[2][1] = -1.0; m.m[2][1] <= 1.0; m.m[2][1]++)
		  for (m.m[2][2] = -1.0; m.m[2][2] <= 1.0; m.m[2][2]++) {
	
		    /* determinant of symmetry has to be +-1 */
		    if (fabs(fabs(det3(m)) - 1.0) < MIN_SYMM_TOL) {

		      /* calculate transformed metric */
		      mgm = m*g*(~m);

		      /* compare orignal and transformed metric */
		      if (matcmp(g, mgm) < MIN_SYMM_TOL)
			/* transposed symmetry matrix, because matrix a 
			   is transposed */
			bsym[bnrot++] = ~m;	
		    }
		  }

  /* transform symmetries from reduced basis to orginal basis */
  dft_log("\n%d symmtries of the bravais lattice\n", bnrot);

  for (i = 0; i < bnrot; i++)
    bsym[i] = inv3(t)*bsym[i]*t;

  dft_log_flush();
}

void 
minimize_basis(matrix3 &a, matrix3 &t)
{
  matrix3 d(1.0, 1.0, 1.0), new_a;
  int change, k1, k2, k3, i, j;

  do {
    change = DFT_FALSE;
    for (k1 = 0; k1 < 3; k1 ++) {
      k2 = (k1 + 1)%3;
      k3 = (k1 + 2)%3;
      for (i = -1; i <= 1; i++)
	for (j = -1; j <= 1; j++) {
	  d.m[k1][k1] = 1.0;
	  d.m[k1][k2] = i;
	  d.m[k1][k3] = j;
	  d.m[k2][k1] = 0.0;
	  d.m[k2][k2] = 1.0;
	  d.m[k2][k3] = 0.0;
	  d.m[k3][k1] = 0.0;
	  d.m[k3][k2] = 0.0;
	  d.m[k3][k3] = 1.0;
	  new_a = d*a;
	  if (trace3((~new_a)*new_a) + MIN_SYMM_TOL < trace3((~a)*a)) {
	    change = DFT_TRUE;
	    a = new_a;
	    t = d*t;
	  }
	}
    }
  }
  while (change);
}

real 
matcmp(matrix3 a, matrix3 b)
{
  matrix3 tmp;
  real sum = 0.0;
  int i, j;
  
  tmp = a - b;

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      sum += fabs(tmp.m[i][j]);

  return(sum);
}

void 
basis_symmetries(Ioninfo *ioninfo, int &bnrot, matrix3 *bsym, 
		      vector3 tr, int &tnrot, matrix3 *tsym)
{
  vector3 **pos, new_pos;
  int bsym_basis[48], sp, natoms, n, n1, irot, found, i;

  /* get memory for new atom positions */
  pos  = (vector3**)mymalloc(sizeof(vector3*)*ioninfo->nspecies, 
				   "pos", "basis_symmetries");
  
  for (sp = 0; sp < ioninfo->nspecies; sp++) {
    natoms = ioninfo->species[sp].natoms;
    pos[sp] = (vector3*)mymalloc(sizeof(vector3)*natoms, 
				       "pos[]", "basis_symmetries");
    for (n = 0; n < natoms; n++) {
      /* move atoms to new symmetry point */
      pos[sp][n] = ioninfo->species[sp].atpos[n] - tr;
      
      for (i = 0; i < 3; i++) {
	pos[sp][n].v[i] = fmod(pos[sp][n].v[i], 1.0);
	if (pos[sp][n].v[i] < 0.0)
	  pos[sp][n].v[i]++;
      }   
    }
  }

  for (irot = 0; irot < bnrot; irot++) {
    bsym_basis[irot] = DFT_TRUE;
    for (sp = 0; sp < ioninfo->nspecies && bsym_basis[irot]; sp++) {
      natoms = ioninfo->species[sp].natoms;
      for (n = 0; n < natoms && bsym_basis[irot]; n++) { 
	new_pos = bsym[irot]*pos[sp][n];
	
	for (i = 0; i < 3; i++) {
 	  new_pos.v[i] = fmod(new_pos.v[i], 1.0);
	  if (new_pos.v[i] < 0.0)
	    new_pos.v[i]++;
	}   

	/* look if there is an equivalent atom */
	for (found = DFT_FALSE, n1 = 0; n1 < natoms && !found; n1++) 
	  if (abs(new_pos - pos[sp][n1]) < MIN_SYMM_TOL)   
	    found = DFT_TRUE;

	bsym_basis[irot] = found;
      }
    }
  }
	
  /* copy symmetries which are symmetries of the basis */
  tnrot = 0;
  for(irot = 0; irot < bnrot; irot++)
    if (bsym_basis[irot])
      tsym[tnrot++] = bsym[irot];
  
  for (sp = 0; sp < ioninfo->nspecies; sp++)
    myfree(pos[sp]);
  myfree(pos);
}

void 
check_symmetries(Ioninfo *ioninfo)
{
  int irot, sp, natoms, n, i, n1, found;
  vector3 new_pos;

  for (irot = 0; irot < ioninfo->nrot; irot++) {
    for (sp = 0; sp < ioninfo->nspecies; sp++) {
      natoms = ioninfo->species[sp].natoms;
      for (n = 0; n < natoms; n++) { 
	new_pos = ioninfo->sym[irot]*ioninfo->species[sp].atpos[n];
	
	for (i = 0; i < 3; i++) {
 	  new_pos.v[i] = fmod(new_pos.v[i], 1.0);
	  if (new_pos.v[i] < 0.0)
	    new_pos.v[i]++;
	}   

	/* look if there is an equivalent atom */
	for (found = DFT_FALSE, n1 = 0; n1 < natoms && !found; n1++) 
	  if (abs(new_pos - ioninfo->species[sp].atpos[n1]) < MIN_SYMM_TOL)   
	    found = DFT_TRUE;

	if (!found) {
	  dft_log(DFT_SILENCE,
		  "Symmetries do not not agree with atomic positions!\n");
	  dft_log(DFT_SILENCE,
		  "symmetry: %d  specie: %d  atom: %d\n", irot, sp, n);
	  die("Symmetries do not not agree with atomic positions!");
	}
      }
    }
  }
}


/*
 * fold_kpoints
 *
 * Fold the kpoints according to symmetries.
 *
 * Ref:
 *  H.J.Monkhorst, J.D.Pack, PRB 13, 5188, 1976
 *
 * Effect: kvec will now contain a list of folded kpoints, with
 *         appropriate weights in w. nkpts = total number of folded
 *         kpoints.
 */
void 
fold_kpoints(vector3** kvec, real **w, const int* kpt_fold, int &nkpts)
{
  int i[3], j, k;
  vector3 *kvec1, *kvec0 = *kvec;
  real *w1, *w0 = *w; 

  int total_fold = kpt_fold[0] * kpt_fold[1] * kpt_fold[2];

  // Ok, let's go.
  if (total_fold <= 0) {
    dft_log(DFT_SILENCE,
	    "Why would you want fold to be 0? %d %d %d\n",
	    kpt_fold[0], kpt_fold[1], kpt_fold[2]);
    die("You are nuts. I will quit.\n");
  }
  else if (total_fold == 1)
    // do nothing
    return;

  // move all components of kpoints to between 0 and 1
  for (k = 0; k < nkpts; k++) {
    for (j = 0; j < 3; j++) {
      kvec0[k].v[j] = fmod(kvec0[k].v[j], 1.0);
      if (kvec0[k].v[j] < 0.0)
	kvec0[k].v[j] += 1.0;
    }
  }

  // allocate temporary storages.
  kvec1 = (vector3 *)mymalloc(sizeof(vector3)*total_fold*nkpts,
			      "kvec","fold_kpoints");
  w1    = (real *)   mymalloc(sizeof(real)*total_fold*nkpts,
			      "w","setup_elecinfo");

  int new_n = 0;
  for (i[0] = 0; i[0] < kpt_fold[0]; i[0]++)
    for (i[1] = 0; i[1] < kpt_fold[1]; i[1]++)
      for (i[2] = 0; i[2] < kpt_fold[2]; i[2]++)
	for (k = 0; k < nkpts; k++) {
	  for (j = 0; j < 3; j++) {
	    if (kpt_fold[j] > 1) {
	      kvec1[new_n].v[j] =
		(kvec0[k].v[j] + i[j])/kpt_fold[j];
	    } else {
	      kvec1[new_n].v[j] = kvec0[k].v[j];
	    }
	  }
	  w1[new_n] = w0[k]/total_fold;
	  new_n++;
	}


  // modify the number of kpoints.
  nkpts *= total_fold;

  // get the kvec and w to point to the newly folded lists.
  myfree(*kvec);
  myfree(*w);
  *kvec = kvec1;
  *w    = w1;

  // output the folded kpoint coordinates
  dft_log("kpoint folding with mesh: %d x %d x %d\n", 
	  kpt_fold[0], kpt_fold[1], kpt_fold[2]);
  for (k = 0; k < nkpts; k++) {
    dft_log("%5d\t[ %4.2f %4.2f %4.2f ]  %4.2f\n",k,
	    kvec1[k].v[0], kvec1[k].v[1], kvec1[k].v[2], w1[k]);
  }
  dft_log("\n");
  
  return;
}


/*
 * reduce_kpoints
 *
 * Reduce the kpoints according to system symmetries.
 *
 * Requirement: all components of kvec[] must be between 0 and 1.
 * Effect: elecinfo.kvec/w are allocated and written;
 *         elecinfo.nkpts is the number of reduced kpoints.
 */
void
reduce_kpoints(const vector3 *kvec, real *w, const Ioninfo &ioninfo,
	       Elecinfo &elecinfo, const Basis &basis, int reduce_kpts_flag)
{
  const matrix3 &GGT = basis.GGT;
  matrix3 invGGT = (~basis.latvec) * (basis.latvec) / (4*M_PI*M_PI);
  int &nkpts = elecinfo.nkpts;
  const matrix3 identity(1.0,1.0,1.0);
  const vector3 vec_one(1.0,1.0,1.0);
  matrix3 sym[48];
  vector3 k_new, k_tmp;
  int i, ii, j, k, newkpts, *found = NULL, nrot = ioninfo.nrot;
  real diff1, diff2, total_w;

  // check that first one is identity
  if ( identity != ioninfo.sym[0] ) {
    die("reduce_kpoints: first symmetry operation is not identity");
  }
  
  /*
   * if not reduce_kpts_flag, then set symmetry numbers to 0.
   * (i.e.  not used)
   * so that the rest of the subroutine just allocates and copy
   * over the kpoint information to elecinfo.
   */
  if ( !reduce_kpts_flag )
    nrot = 0;

  // get the symmetry matrix for k space
  /*
   * Sk = (~A A) Sr (G ~G) / (2pi)^2
   *   note:  (~A A / (2pi)^2) = inv ( G ~G )
   * 
   */
  for (i = 0; i < nrot; i++) {
    sym[i] = invGGT * ioninfo.sym[i] * GGT;
  }

  // initialize found array to false.
  found = (int*)mymalloc(sizeof(int)*nkpts,"found","reduce_kpoints");
  for (i = 1; i < nkpts; i++)
    found[i] = DFT_FALSE;

  /* for each original k-point,
   * if not found yet, put it in new k-point list,
   *  for each symmetry, if transformed k-point matches another
   *  original k-point, list that one as found, add its weight 
   *  to the present k-point.
   */
  newkpts = 0;
  for (i = 0; i < nkpts; i++) { // for each original k-point,
    if ( !found[i] ) {          // if not yet found.
      newkpts++;                 // add to new-kpoint list
      for (j = 0; j < nrot; j++) {  // for each symmetry
	k_new = sym[j] * kvec[i];   // produce transformed k.
	// get k_new in betweenn 0..1;
	for (ii = 0; ii < 3; ii++) {
	  k_new.v[ii] = fmod(k_new.v[ii], 1.0);
	  if (k_new.v[ii] < 0.0)
	    k_new.v[ii] += 1.0;
	}
	// compare with remaining kpoints
	for (k = i+1; k < nkpts; k++) {
	  // get k_tmp in betweenn 0..1 to compare to k_new;
	  for (ii = 0; ii < 3; ii++) {
	    k_tmp.v[ii] = fmod(kvec[k].v[ii], 1.0);
	    if (k_tmp.v[ii] < 0.0)
	      k_tmp.v[ii] += 1.0;
	  }
	  diff1 = abs2(k_new - k_tmp);
	  // check also for inversion symmetry.
	  diff2 = abs2(k_new + k_tmp - vec_one);
	  if ( ((diff1 < MIN_KPT_DISTANCE) || (diff2 < MIN_KPT_DISTANCE))
	       && (! found[k]) ) {
	    found[k] = DFT_TRUE;
	    w[i] += w[k];
	  }
	}
      }
    }
  }

  if (newkpts == nkpts)
    dft_log("reduce_kpoints: No reducable k-point discovered.\n");
  else 
    dft_log("reduce_kpoints: number of k-points reduced to %d.\n", newkpts);

  elecinfo.kvec = (vector3 *)mymalloc(sizeof(vector3)*newkpts,
				      "elecinfo.kvec","reduce_kpoints");
  elecinfo.w    = (real *)   mymalloc(sizeof(real)*newkpts,
				      "elecinfo.w","reduce_kpoints");

  // calculate weight renormalization in case error has been accumulated.
  for (i = 0, total_w = 0.0; i < nkpts; i++)
    if ( !found[i] )
      total_w += w[i];


  for (i = j = 0; i < nkpts; i++) {
    if ( !found[i] ) {
      elecinfo.kvec[j] = kvec[i];
      elecinfo.w[j] = w[i] / total_w;
      j++;
    }
  }
  nkpts = newkpts;

  myfree(found);

  // output the reduced kpoint coordinates
  for (k = 0; k < nkpts; k++) {
    dft_log("%5d\t[ %4.2f %4.2f %4.2f ]  %4.2f\n",k,
	    elecinfo.kvec[k].v[0], 
	    elecinfo.kvec[k].v[1], 
	    elecinfo.kvec[k].v[2],
	    elecinfo.w[k]);
  }
  dft_log("\n");

  return;
}


/*
 * check_symm_fftbox
 *
 * Make sure that the FFT-box doesn't destroy the intrinsic symmetries of 
 * the system.
 *
 * And calculate the symmetry matrix for charge density mesh.
 */
int
check_symm_fftbox(Ioninfo &ioninfo, const Basis &basis)
{
  int N[3] = { basis.Nx, basis.Ny, basis.Nz };
  matrix3 tmp, Nmat(basis.Nx, basis.Ny, basis.Nz), Nmat_inv(1.0/basis.Nx, 1.0/basis.Ny, 1.0/basis.Nz);
  int nrot = ioninfo.nrot, irot, i1, i2, error = DFT_FALSE;

  for (irot = 0; irot < nrot; irot++) {
    tmp = ioninfo.sym[irot] * Nmat;
    for (i1 = 0; i1 < 3; i1++) {
      for (i2 = 0; i2 < 3; i2++) {
	if (fmod(tmp.m[i1][i2], N[i1]) > MIN_SYMM_TOL) {
	  if (!error) 
	    dft_log(">>>> FFT box not comensurate with symmetries:\n");
	  ioninfo.sym[irot].print(dft_global_log,"%4.0f ");
	  dft_log(">>>> \n");
	  error = DFT_TRUE;
	}
      }
    }
    ioninfo.n_sym[irot] = Nmat * ioninfo.sym[irot] * Nmat_inv;
  }
  if (error) {
    dft_log(DFT_SILENCE,">>>> \t[ %d %d %d ]\n",N[0], N[1], N[2]);
    dft_log(DFT_SILENCE,">>>> You have two options:\n\t1. disable symmetry\n");
    dft_log(DFT_SILENCE,"\t2. manually input FFT box size\n");
    die("");
  }

  return (! error);
}


/*
 * symmetrize_n 
 *
 * Symmetrize the charge density
 */
int
symmetrize_n(vector & n, int nrot, const matrix3 * n_sym, int * done)
{
  Basis * basis = n.basis;
  // Z is the outer index, Y the next, X the inner index.
  vector3 r, rnew;
  int irot, i;
  int index, index2, N[3], ind[48];
  real one_over_nrot, n_sum;

  if (nrot <= 1)  // no need to do any symmetrization
    return 0;

  one_over_nrot = 1.0/nrot;
  for (i = 0; i < n.n; i++)
    done[i] = DFT_FALSE;

  N[0] = basis->Nx; N[1] = basis->Ny; N[2] = basis->Nz;
  index = 0;
  for (r.v[0]=0; r.v[0] < N[0]; r.v[0]+=1) {
    for (r.v[1]=0; r.v[1] < N[1]; r.v[1]+=1) {
      for (r.v[2]=0; r.v[2] < N[2]; r.v[2]+=1, index++) {
	// for all points in the box

	if ( !done[index] ) { // if not yet related by symmetry to a previous point

	  n_sum = REAL( n.c[index] );   // first symmetry is identity.
	  ind[0] = index;
	  for (irot = 1; irot < nrot; irot++) { // for all other symmetries
	    rnew = n_sym[irot] * r; // get the point related by symmetry.
	  
	    for (i=0;i<3;i++) {	    // project back into range.
	      rnew.v[i] = rint(rnew.v[i]);
	      rnew.v[i] = fmod(rnew.v[i], N[i]);
	      if (rnew.v[i] < 0.0)
		rnew.v[i] += N[i];
	    }
	    index2 = (int)(rnew.v[2] + N[2]*(rnew.v[1] + N[1]*rnew.v[0]));

	    n_sum += REAL(n.c[ index2 ]); // add the charge density.

	    ind[irot] = index2; // remember the index.
	  }
	  
	  n_sum *= one_over_nrot;  // get averaged charge density.

	  for (irot = 0; irot < nrot; irot++) { // put symmetrized value in the star.
	    n.c[ ind[irot] ] = n_sum;
	    done[ ind[irot] ] = DFT_TRUE;
	  }
	}
      }
    }
  }

  return(1);
}
