/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *     Sohrab Ismail-Beigi           Mar. 5, 1997
 *
 * Setup up from a file (or frees) the information and data describing the
 * electronic states.
 *
 */

/* $Id: setup_elecs.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "header.h"


/*
 * Read the file <elecfilename> to get all the specifications for the
 * electronic states:  energy cutoff (in Hartrees), number of bands, number
 * of k-points, number of electrons, the k-vectors, weights, and fillings.
 * The format of of the <elecinfofile> is described in the comments below.
 *
 ***************************************************************************
 * (1) The format of the <elecfilename> file:
 *
 *         Ecut  [# comment]
 *         Nkpoints Nbands Nelectrons  [# comment]
 *         calc_fillings_flag [ kT niter_recalc_fillings] [# comment]
 *         # for i = 1 to Nkpoints {
 *             kx_i ky_i kz_i w_i      [# comment]
 *             f_i1 f_i2 f_i3 ... f_iNbands
 *         # }
 *
 * "# for i = 1 to Nkpoints" is my shorthand for saying that there must be
 * Nkpoints instances of the lines in the braces.
 *
 * -> Ecut is real and is the kinetic energy cutoff in Hartrees.
 * -> Nkpoints is an integer:  # of k-points
 * -> Nbands is an integer: # of bands at each k-point
 * -> Nelectrons is an integer: total # of electrons (should be the
 *                              total sum of all fillings; is a real number.)
 * -> calc_fillings_flag: either 1 or 0; if 0, the fillings are read
 *                        and never changed.  If 1, fillings are recomputed
 *                        during a calculation via a Fermi-Dirac distrib.
 * -> kT: Temperature of electronic system (only used if calc_fillings_flag==1)
 * -> niter_recalc_fillings: we recompute electronic fillings every
 *                           so many CG iterations (only used if
 *                           calc_fillings_flag == 1).
 * -> kx_i ky_i kz_i are reals giving k-vectors in recip. lattice coords.
 * -> w_i is the weight of the k-point (real)
 * -> f_ij (real) is the filling for the i-th k'point's j'th band.
 *
 * Note:
 * -> any line whose first character is '#' is a comment line and ignored
 * -> [# comment] means any text is ignored until the end of the line
 *
 * NO COMMENTS ALLOWED BETWEEN THE VALUES OF THE FILLLINGS.
 * THE LAST FILLLING MUST BE ON A LINE BY ITSELF.
 *
 ***************************************************************************
 *
 * elecs: the Elecinfo structure to fill up with information in elecfilename.
 * basis: pointer to the basis set used to do the calculation.
 * logfile: the file to send the reports to.
 *
 */


void
setup_elecinfo(Elecinfo *elecs,char *elecfilename,
	       Basis **basis, Control &cntrl)
{
  dft_text_FILE *elecfile;

  char line[DFT_LINE_LEN];
  int k,b;
  real f,nel;

  if ( (elecfile = dft_text_fopen(elecfilename,"r")) == (dft_text_FILE *)0)
    die("Can't read electronic information file %s\n",elecfilename);

  /* Read the Kinetic energy cutoff */
  do { dft_text_fgets(line,DFT_LINE_LEN,elecfile); } while(line[0] == '#');
  sscanf(line,"%lg",&(elecs->Ecut));

  dft_log("\n----- setup_elecinfo() -----\n\n");
  dft_log("Reading electronic information from file %s\n\n",
	    elecfilename);
  dft_log("Ecut = %lf Hartrees\n\n",elecs->Ecut);


  /* Read the number of k-points, the # of bands, and the total electron # */
  do { dft_text_fgets(line,DFT_LINE_LEN,elecfile); } while(line[0] == '#');
  sscanf(line,"%d %d %lg",&(elecs->nkpts),&(elecs->nbands),
                      	  &(elecs->nelectrons));

  dft_log("Nkpoints = %d   Nbands = %d   Nelectrons = %lg\n\n",
	    elecs->nkpts,elecs->nbands,elecs->nelectrons);

  /* Read flag describing whether we should recalculated fillings
   * during the computation; if so, also read the electronic temperature
   * and the number of CG iterations to skip between calculations
   * of fillings */
  do { dft_text_fgets(line,DFT_LINE_LEN,elecfile); } while(line[0] == '#');
  sscanf(line,"%d",&(elecs->calc_fillings));
  if (elecs->calc_fillings!=0 && elecs->calc_fillings!=1)
    die("\n\ncalc_fillings_flag must be 1 or 0 in setup_elecinfo()!!!\n\n");
  else if (elecs->calc_fillings == 0)
    {
      elecs->kT = 0.0;
      elecs->niter_recalc_fillings = 0;
      dft_log("calc_fillings_flag = 0\n\n");
    }
  else
    {
      sscanf(line,"%d %lg %d",&(elecs->calc_fillings),
	                  &(elecs->kT),
	                  &(elecs->niter_recalc_fillings));
      elecs->mu = 0.0;  // initial chemical potential guess.
      dft_log("calc_fillings_flag = 1\tkT = %lg\tniter_recalc_fillings = %d\n\n",
		elecs->kT,elecs->niter_recalc_fillings);
    }

  /* Loop over k-points and read k-vectors, weights, and fillings */
  elecs->kvec = (vector3 *)mymalloc(sizeof(vector3)*elecs->nkpts,
				    "kvec","setup_elecinfo");
  elecs->w = (real *)mymalloc(sizeof(real)*elecs->nkpts,
			      "w","setup_elecinfo");
  elecs->F = (diag_matrix *)mymalloc(sizeof(diag_matrix)*elecs->nkpts,
				     "F","setup_elecinfo");
  *basis = (Basis *)mymalloc(sizeof(Basis)*(elecs->nkpts+1),"basis","setup_elecinfo");
  // the last basis is for charge density.
  

  nel = 0.0;
  for (k=0; k < elecs->nkpts; k++)
    {
      /* Read kx ky kz w */
      do { dft_text_fgets(line,DFT_LINE_LEN,elecfile); } while(line[0] == '#');
      sscanf(line,"%lg %lg %lg %lg",
	     &(elecs->kvec[k].v[0]),
	     &(elecs->kvec[k].v[1]),
	     &(elecs->kvec[k].v[2]),
	     &(elecs->w[k]));

      dft_log("kpt = %d: k = ",k);
      elecs->kvec[k].print(dft_global_log,"%lg ");
      dft_log(" w = %lg\n",elecs->w[k]);

      /* Set up filling structure and read the fillings */
      elecs->F[k].init(elecs->nbands,&(*basis)[k]);
      dft_log("kpt = %d: Fillings = [ ",k);
      for (b=0; b < elecs->nbands; b++)
	{
	  dft_text_fscanf(elecfile,"%lg",&f);
	  elecs->F[k].c[b] = f;
	  nel += f*elecs->w[k];
	  dft_log("%lg ",f);
	}
      /* Get useless stuff to end of line */
      dft_text_fgets(line,DFT_LINE_LEN,elecfile);
      dft_log("]\n\n");

    }

  if (nel != elecs->nelectrons)
    {
      dft_log("&&&&&&&&&&&&&&&&&& WARNING: &&&&&&&&&&&&&&&&&&&\n");
      dft_log("Nelectrons DOESN'T EQUAL SUM OF WEIGHTS*FILLINGS = %lg\n",nel);
      dft_log("THE DIFFERENCE IS %le.\n",nel-elecs->nelectrons);
      dft_log("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&\n\n");

      dft_log("I will renormalize the WEIGHTS for you.\n");

      real ratio = elecs->nelectrons / nel;
      for (k=0; k < elecs->nkpts; k++) elecs->w[k] *= ratio;
    }

  // Read total energy convergence tolerance, and other additional controls.
  // initialize the default values not in Control:
  elecs->subspace_rotation = 1;  // enable subspace rotation by default.
  elecs->pcond_for_Bgrad = 30.0;  // set preconditioning number for Bgrad.
  elecs->ex_opt = DFT_EXCORR_LDA; // set exchange-correlation to LDA by default.

  char key[DFT_LINE_LEN];
  while ( dft_text_fgets(line,DFT_LINE_LEN,elecfile) != NULL ) {
    // look for tolerance input, if not find, then assume 0 tolerance.
    // in which case, full number of iterations are done.
    // The line format should be:
    //       <value> # e_tol <comments>
    // Flag for do_linmin in electronic minimization for
    // updating stepsize with gamma:
    //       <1,0> # update_stepsize <comments>
    if ( line[0] == '#' ) continue;  // skip comments.
    if (sscanf(line, "%*s # %s", key) > 0) {
      if (strncmp(key,"e_tol",5)==0) {
	sscanf(line, "%lg", &(cntrl.E_tolerance));
      } else if (strncmp(key,"update_stepsize",15)==0) {
	sscanf(line, "%d", &(cntrl.update_stepsize));
      } else if (strncmp(key,"stepsize_min",12)==0) {
	sscanf(line, "%lg", &(cntrl.stepsize_min));
      } else if (strncmp(key,"nel_fermi_tol",13)==0) {
	sscanf(line, "%lg", &(cntrl.nel_fermi_tol));
      } else if (strncmp(key,"subspace_rotation",17)==0) {
	sscanf(line, "%d", &(elecs->subspace_rotation));
	if (elecs->subspace_rotation != 0)
	  elecs->subspace_rotation = 1;
      } else if (strncmp(key,"pcond_Bgrad",11)==0) {
	sscanf(line,"%lg", &(elecs->pcond_for_Bgrad));
      } else if (strncmp(key,"ex_corr",7)==0) {
	sscanf(line,"%s",key);
	if (strncmp(key,"gga",3)==0) {
	  elecs->ex_opt = DFT_EXCORR_GGA; // set ex_corr to GGA PW91
	} else if (strncmp(key,"lda",3)==0) {
	  elecs->ex_opt = DFT_EXCORR_LDA; // set ex_corr to LDA
	} else {
	  dft_log(">>(unknown ex_corr options)> %s\n",key);
	}
      } else {
	dft_log(">>(unknown options)> %s",line);
      }
    } else {
      dft_log(">>>(extra line)> %s",line);
    }
  }

  dft_log("Etot convergence tolerance = %lg Hartree.\n",
	    cntrl.E_tolerance);
  if (cntrl.update_stepsize)
    dft_log("Update stepsize with gamma in do_linmin.\n");
  dft_log("stepsize is constrainted to be >= %lg\n",cntrl.stepsize_min);
  if (elecs->calc_fillings == 1)
    dft_log("nel in fermi filling has tolerance: %lg.\n",
	      cntrl.nel_fermi_tol);
  if (elecs->subspace_rotation == 1) {
    dft_log("Perform subspace rotation during minimization,\n");
    dft_log("with precondition number for Bgrad = %lg.\n",
	      elecs->pcond_for_Bgrad);
  }
  if (elecs->ex_opt == DFT_EXCORR_GGA)
    dft_log("Using GGA PW91 exchange-correlation.\n");
  else 
    dft_log("Using LDA exchange-correlation.\n");
  dft_log_flush();

  dft_text_fclose(elecfile);
}

/* free up memory used by structure in Elecinfo */
void
free_elecinfo(Elecinfo *elecs)
{
  int k;

  for (k=0; k < elecs->nkpts; k++)
    elecs->F[k].freemem();
  myfree(elecs->F);
  myfree(elecs->w);
  myfree(elecs->kvec);
}

/* Setup and allocate space for the electronic variables */
void
init_elecvars(Elecinfo *einfo,Basis *basis,Elecvars *evars)
{
  int nkpts,k,nbands,NXYZ;
  Basis *fftbox_basis;

  /* Get key sizes */
  nkpts = einfo->nkpts;
  nbands = einfo->nbands;
  fftbox_basis = &basis[nkpts];
  NXYZ = fftbox_basis->NxNyNz;

  /* Allocate space for the electron density, electrostatic potential,
     the local pseudopotential, and the local part of the self-consistent
     potential */
  evars->n.init(NXYZ,fftbox_basis);
  evars->d.init(NXYZ,fftbox_basis);
  evars->Vlocps.init(NXYZ,fftbox_basis);
  evars->Vscloc.init(NXYZ,fftbox_basis);

  /* Allocate space for all the matrices and eigenvalues */
  evars->U = alloc_matrix_array(nkpts,nbands,nbands);
  evars->Umhalf = alloc_matrix_array(nkpts,nbands,nbands);
  evars->W = alloc_matrix_array(nkpts,nbands,nbands);
  evars->Hsub = alloc_matrix_array(nkpts,nbands,nbands);
  evars->Hsub_evecs = alloc_matrix_array(nkpts,nbands,nbands);
  evars->mu = (real **)mymalloc(sizeof(real *)*nkpts,
				"init_elecvars()","mu");
  evars->Hsub_eigs = (real **)mymalloc(sizeof(real *)*nkpts,
				       "init_elecvars()","Hsub_eigs");

  if (einfo->subspace_rotation) {
    evars->B = alloc_matrix_array(nkpts,nbands,nbands);
    evars->V = alloc_matrix_array(nkpts,nbands,nbands);
    evars->Z = alloc_matrix_array(nkpts,nbands,nbands);
    evars->beta = (real **)mymalloc(sizeof(real *)*nkpts,
				    "init_elecvars()","beta");
  } else {
    evars->B = NULL; 
    evars->V = NULL;
    evars->Z = NULL;
    evars->beta = NULL;
  }

  for (k=0; k < nkpts; k++)
    {
      /* Set hermetian flags & alloc double real arrays for eigenvalues */
      evars->U[k].hermetian = 1;
      evars->Umhalf[k].hermetian = 1;
      evars->Hsub[k].hermetian = 1;
      evars->mu[k] = (real *)mymalloc(sizeof(real)*nbands,
				      "init_elecvars()","mu[k]");
      evars->Hsub_eigs[k] = (real *)mymalloc(sizeof(real)*nbands,
					     "init_elecvars()",
					     "Hsubk_eigs[k]");
      if (einfo->subspace_rotation) {
	evars->B[k].hermetian=1;
	evars->V[k].hermetian=0;
	evars->Z[k].hermetian=0;
	evars->beta[k] = (real *)mymalloc(sizeof(real)*nbands,
					  "init_elecvars()","beta[k]");

	/* Set B to identity */
	int i;

	evars->B[k].zero_out();
	for (i=0; i < nbands; i++)
	  evars->B[k](i,i) = 1.0;
      }
    }


  /* Allocate the column_bundle arrays for the electronic wave-functions */
  evars->Y = alloc_column_bundle_array(nkpts,nbands,basis);
  evars->C = alloc_column_bundle_array(nkpts,nbands,basis);

  /* Setup all the necessary internal pointers */
  for (k=0; k < nkpts; k++)
    {
      evars->Y[k].Vscloc     = evars->C[k].Vscloc     = &(evars->Vscloc);
      evars->Y[k].k          = evars->C[k].k          = einfo->kvec[k];
      evars->Y[k].basis      = evars->C[k].basis      = &basis[k];
    }

  /*
   * due to code organization, energy calculation routines don't have 
   * access to  Elecinfo, so has to make a copy in  Elecvars
   */
  evars->ex_opt = einfo->ex_opt;

  
  /*
   * If non-linear core correction is going to be performed, allocate 
   *  ncore.  It would be filled in later by nlcore_setup().
   */
  if (einfo->nl_core_flag) {
    evars->ncore.init(NXYZ,&basis[nkpts]);
  } else {
    evars->ncore.n = 0;  // set again just to make sure.
  }

}


/* Free up the memory used by the electronic variables */
void
free_elecvars(Elecinfo *einfo,Elecvars *evars)
{
  int nkpts,k;

  nkpts = einfo->nkpts;

  free_column_bundle_array(nkpts,evars->Y);
  free_column_bundle_array(nkpts,evars->C);

  free_matrix_array(nkpts,evars->U);
  free_matrix_array(nkpts,evars->W);
  free_matrix_array(nkpts,evars->Umhalf);
  free_matrix_array(nkpts,evars->Hsub);
  free_matrix_array(nkpts,evars->Hsub_evecs);
  for (k=0; k < nkpts; k++)
    {
      myfree(evars->mu[k]);
      myfree(evars->Hsub_eigs[k]);
    }
  myfree(evars->mu);
  myfree(evars->Hsub_eigs);

  if (einfo->subspace_rotation) {
    free_matrix_array(nkpts,evars->B);
    free_matrix_array(nkpts,evars->V);
    free_matrix_array(nkpts,evars->Z);
    for (k=0; k < nkpts; k++)
      myfree(evars->beta[k]);
    myfree(evars->beta);
  }

  evars->n.freemem();
  evars->d.freemem();
  evars->Vlocps.freemem();
  evars->Vscloc.freemem();

  evars->ncore.freemem();
}  
