/*
 * Tairan Wang,                   July 21 1998
 *
 * Added Conjugate Gradient ion relaxation
 */

/*
 * Sohrab Ismail-Beigi,           April 7 1997
 *
 * Does {ncg_iter of CG on the electronic variables, then moves the ions}
 * ntotal times.
 *
 * The signal handling ensures that if certain signals are received,
 * wave-functions are dumped, etc.  See signal.c.
 *
 * There are two invocations of the program from the command-line:
 * (1) with no arguments, the file INPUTFILE (below) is read
 * (2) with one argument, which is the name of the input file to be read.
 *
 * The format of the input file is:
 *
 * #
 * # comments are lines that start with '#'
 * #
 * ntotal     # total number of ionic moves
 * ncg_iter    # number of CG iterations
 * init_step   # size of initial linmin step (-1.0 is okay probably)
 * elecfile    # name of file with electronic state information
 * latfile     # name of file with the lattice vectors
 * ionsfile    # name of file with ionic info:  Z, positions, pseudopots,...
 * ewaldfile   # name of file with ewald setup information
 * <Yflag>     # <Yflag> is a string describing initial wave-functions
 * [filename]  # name of file to read for wave-functions if Yflag == 'read'
 * <FFTflag>   # a string describing how FFT box sizes are picked
 * [nx ny nz]  # x,y,z FFT box sizes if FFTflag == 'specified'
 * maskfile    # filename containing name of the force "masks" (see below)
 * K_spring    # spring constant to use for diagonal preconditioner
 *
 * <Yflag> must be either 'read' or 'random' (without quotes).  'random'
 * means that the initial wave-functions are filled with random-numbers.
 * 'read' means that a file should be read:  the name is given on the
 * next line.
 *
 * <FFTflag> must be either 'automatic' or 'specified'.  'automatic'
 * means the program figures out its on FFT box sizes.  'specified'
 * means it tries to use the sizes provided on the next line.
 *
 * The mask file contains the integers 1 or 0 as many as there are atoms 
 * in the system.  The masks go in order of species, and for each species,
 * in order of atoms as given in the position file for that species.
 * The mask value for an atom multiplies the displacement for that atom;
 * in this way, we can have some ions not move.
 *
 * K_spring is used for preconditioning the forces (i.e. just scaling them
 * here...).  The atomic positions are updated via:
 *        r = r + force/K_spring
 * where r and force are in real x-y-z units (NOT LATTICE UNITS), i.e. r
 * is in Bohr and force is in Hartree/Bohr.
 *
 */

/* $Id: relaxions_CG.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
/* My header file */
#include "header.h"

#define INPUTFILE    "relaxions.in"
#define REPORT_LEVEL 1
#define LOGFILE      stdout

#define LN_LEN  150
#define STR_LEN 80

/* Master structure for Ion relaxation */
class IonRelax
{
public:
  Ioninfo ioninfo;

  int total_atoms;
  vector3 *f_old;
  vector3 *dir;

  real linminfact, cosgg, alpha;
  real abs2_grad_now, abs2_grad_old;

  // Member functions
private:
  real dot_product(vector3* a, vector3* b, int len);
public:
  void init(void);
  real f_dot_f(void);
  real f_dot_dir(void);
  real fold_dot_dir(void);
  real f_dot_fold(void);
  void dir_zero_out(void);
  void dir_scale(real alpha);
  void dir_add_f(void);
  void f_to_fold(void);
  void impose_sym(void);
  real do_linmin(real ion_stepsize, Basis *basis, Elecinfo &elecinfo,
		 Elecvars &elecvars, Energies &ener0, Control &cntrl,
		 int niter_cg, real elec_stepsize, 
		 int report_level, FILE* logfile);
  real testline(real ion_stepsize, Basis *basis, Elecinfo &elecinfo,
		 Elecvars &elecvars, Energies &ener0, Control &cntrl,
		int niter_cg, real elec_stepsize, 
		int report_level, FILE* logfile);
  void print_pos(FILE* logfile);
  void print_forces(FILE* logfile);
  void freemem(void);
};

void IonRelax::init(void)
{
  abs2_grad_now = 0.0;
  total_atoms = 0;
  for (int sp = 0; sp < ioninfo.nspecies; sp++) 
    total_atoms += ioninfo.species[sp].natoms; 
  f_old = (vector3 *)mymalloc(sizeof(vector3)*total_atoms,
			      "f_old[]", "IonRelax::init");
  dir   = (vector3 *)mymalloc(sizeof(vector3)*total_atoms,
			      "dir[]", "IonRelax::init");
}

real IonRelax::dot_product(vector3* a, vector3* b, int len)
{
  real result = 0.0;
  for (int i=0; i < len; i++) result += a[i] * b[i];
  return result;
}

real IonRelax::f_dot_f(void)
{
  real result = 0.0;
  for (int sp = 0; sp < ioninfo.nspecies; sp++)
    result += dot_product(ioninfo.species[sp].forces,
			  ioninfo.species[sp].forces,
			  ioninfo.species[sp].natoms);
  return result;
}

real IonRelax::f_dot_dir(void)
{
  real result = 0.0;
  int sp, i;
  for (sp = 0, i = 0; 
       sp < ioninfo.nspecies; 
       sp++, i+=ioninfo.species[sp].natoms) 
    {
      result += dot_product(ioninfo.species[sp].forces,
			    dir+i,
			    ioninfo.species[sp].natoms);
    }
  return result;
}

real IonRelax::f_dot_fold(void)
{
  real result = 0.0;
  int sp, i;
  for (sp = 0, i = 0; 
       sp < ioninfo.nspecies; 
       sp++, i+=ioninfo.species[sp].natoms) 
    {
      result += dot_product(ioninfo.species[sp].forces,
			    f_old+i,
			    ioninfo.species[sp].natoms);
    }
  return result;
}

real IonRelax::fold_dot_dir(void)
{
  return dot_product(f_old,dir,total_atoms);
}

void IonRelax::dir_zero_out(void)
{
  for (int i=0; i<total_atoms; i++)
    dir[i].v[0] = dir[i].v[1] = dir[i].v[2] = 0.0;
}

void IonRelax::dir_scale(real alpha)
{
  for (int i=0; i<total_atoms; i++) dir[i] *= alpha;
}

void IonRelax::dir_add_f(void)
{
  int sp, nat, i;
  for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++)
    for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
 	dir[i] += ioninfo.species[sp].forces[nat];
}

void IonRelax::f_to_fold(void)
{
  int sp, nat, i;
  for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++)
    for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
      f_old[i] = ioninfo.species[sp].forces[nat];
}

void IonRelax::impose_sym(void)
{
  int sp, nat;
  // Impose extra symmetry on the system by modifying the 
  // forces. This breaks the harmonic assumption, but hopefully 
  // the effect on convergence is small.
  
  // 1. for ions not suppose to move, zero out the forces
  for (sp = 0; sp < ioninfo.nspecies; sp++)
    for (nat = 0; nat < ioninfo.species[sp].natoms; nat++)
      if (ioninfo.species[sp].move[nat] != 1)
	ioninfo.species[sp].forces[nat] = 0.0;

}

real IonRelax::testline(real ion_stepsize, Basis *basis, Elecinfo &elecinfo,
			 Elecvars &elecvars, Energies &ener0, Control &cntrl,
			 int niter_cg, real elec_stepsize,
			 int report_level, FILE* logfile)
{
  int sp, nat, i, nstep = 10;
  real gamma, dderiv, curvature, stepsize;
  Energies ener;

  stepsize = ion_stepsize/nstep;
  dderiv = 2.0 * f_dot_dir();

  if (report_level > 0)
    {
      fprintf(logfile,"IonCG: dderiv = %le\n",
	      dderiv);
    }
  // shift by stepsize * dir
  for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++) 
    for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
      ioninfo.species[sp].atpos[nat] += -ion_stepsize * dir[i];

  for (int istep =1; istep <=2*nstep; istep++) {

    // shift by stepsize * dir
    for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++) 
      for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
	ioninfo.species[sp].atpos[nat] += stepsize * dir[i];

    if (report_level > 0)
      {
	fprintf(logfile,"Test step %lg : %d\n",
		stepsize,istep);
	fflush(logfile);
      }

    // do the electron degree of freedom given the ion positions
    calc_core_ewald_pulay_energies(basis,&ioninfo,&elecinfo,&ener,
				   report_level, logfile);
    Vloc_pseudo(&basis[elecinfo.nkpts], &ioninfo, elecvars.Vlocps.c, report_level, logfile);
    /* Calculate the non-local pseudopotential for FHI CPI type psp. */
    Vnl_pseudo(basis,&ioninfo,&elecinfo, report_level, logfile);

    elec_stepsize = minimize_elec_pcg_nocosgpg(niter_cg, elec_stepsize,
					       basis, &ioninfo, &elecinfo, &elecvars,
					       &ener, cntrl, report_level, logfile);

    if (report_level > 0)
      {
	fprintf(logfile,"IonCG: Etot  = %le\n", ener.Etot);
	print_pos(logfile);
	fflush(logfile);
      }

  }

  return ion_stepsize;
}

real IonRelax::do_linmin(real ion_stepsize, Basis *basis, Elecinfo &elecinfo,
			 Elecvars &elecvars, Energies &ener0, Control &cntrl,
			 int niter_cg, real elec_stepsize,
			 int report_level, FILE* logfile)
{
  int sp, nat, i;
  real gamma, dderiv, curvature;
  Energies ener;

// 2* is removed here by Shanhui
  dderiv = f_dot_dir();

  for (;;) {

    // shift by stepsize * dir
    for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++) 
      for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
	ioninfo.species[sp].atpos[nat] += ion_stepsize * dir[i];

    if (report_level > 0)
      {
	fprintf(logfile,"Calculate electronic system %lg in dir direction\n",
		ion_stepsize);
	fflush(logfile);
      }
    ener = ener0;
    // do the electron degree of freedom given the ion positions
    calc_core_ewald_pulay_energies(basis,&ioninfo,&elecinfo,&ener,
				   report_level, logfile);
    Vloc_pseudo(&basis[elecinfo.nkpts], &ioninfo, elecvars.Vlocps.c, report_level, logfile);
    /* Calculate the non-local pseudopotential for FHI CPI type psp. */
    Vnl_pseudo(basis,&ioninfo,&elecinfo, report_level, logfile);

    elec_stepsize = minimize_elec_pcg_nocosgpg(niter_cg, elec_stepsize,
					       basis, &ioninfo, &elecinfo, &elecvars,
					       &ener, cntrl, report_level, logfile);

    // shift back
    for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++) 
      for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
	ioninfo.species[sp].atpos[nat] -= ion_stepsize * dir[i];

    /* Do a parabolic fit to the Etot0, Etot, and dderiv
     * to get curvature for quadratic fit and proposed minimum (gamma) */
    curvature = 2.0*(ener.Etot - ener0.Etot + ion_stepsize*dderiv) / 
                (ion_stepsize * ion_stepsize);
    gamma = dderiv/curvature;
    if (report_level > 0)
      {
	fprintf(logfile,"IonCG: dderiv = %le, curvature = %le\n",
		dderiv, curvature);
	fprintf(logfile,"IonCG: stepsize = %le, gamma = %le\n",
		ion_stepsize, gamma);
	fprintf(logfile,"IonCG: Etot  = %le\n", ener.Etot);
	fprintf(logfile,"IonCG: Etot0 = %le\n", ener0.Etot);
	fprintf(logfile,"IonCG: ion_stepsize = %le\n", ion_stepsize);
	fprintf(logfile,"\n");
	fflush(logfile);
      }

    /* If curvature is wrong way, take a bigger step */
    if (curvature < 0.0)
      { ion_stepsize *= 4.0; continue; }
    /* If the proposed minimum is much larger than the stepsize,
     * increase stepsize */
    else if (fabs(gamma/ion_stepsize) > 10.0)
      { ion_stepsize *= 10.0; continue; }
    /* If much smaller, decrease */
    else if (fabs(gamma/ion_stepsize) <  0.1)
      { ion_stepsize *= 0.1; continue; }
    /* Otherwise, it was a good linmin so stop! */
    else
      break;

  }

  // move to minimum,
  for (sp = 0, i = 0; sp < ioninfo.nspecies; sp++) 
    for (nat = 0; nat < ioninfo.species[sp].natoms; nat++, i++)
      ioninfo.species[sp].atpos[nat] += gamma * dir[i];

  // return final gamma.
  return gamma;
}

void IonRelax::print_forces(FILE* logfile)
{
  /* Print out the forces on current ion */
  int sp;
  for (sp=0; sp < ioninfo.nspecies; sp++)
    {
      int i;      
      fprintf(LOGFILE,"\nspecies = %d,  natoms = %d,  forces follow:\n\n",
	      sp,ioninfo.species[sp].natoms);
      for (i=0; i < ioninfo.species[sp].natoms; i++) {
	fprintf(LOGFILE,"%3d %4d %20.10le %20.10le %20.10le %d\n",
		sp,i,
		ioninfo.species[sp].forces[i].v[0],
		ioninfo.species[sp].forces[i].v[1],
		ioninfo.species[sp].forces[i].v[2],
		ioninfo.species[sp].move[i]);
      }
    }  
}


void IonRelax::print_pos(FILE* logfile)
{
  /* Print out the current ion positions! */
  int sp;
  for (sp=0; sp < ioninfo.nspecies; sp++) {
    int i;
    fprintf(logfile,"\nspecies = %d,  natoms = %d,  positions follow:\n\n",
	    sp, ioninfo.species[sp].natoms);
    for (i=0; i < ioninfo.species[sp].natoms; i++)
      fprintf(logfile,"%3d %4d %20.10le %20.10le %20.10le %d\n",
	      sp,i,
	      ioninfo.species[sp].atpos[i].v[0],
	      ioninfo.species[sp].atpos[i].v[1],
	      ioninfo.species[sp].atpos[i].v[2],
	      ioninfo.species[sp].move[i]);
  }
  fprintf(logfile,"\n");
}

void IonRelax::freemem(void)
{
  free_ioninfo(&ioninfo);
  myfree(f_old);
  myfree(dir);
}

//
// Utility routines;
//

void getinput(MPI_FILE *filep,int &data,
	      int toggle,FILE *log,char* comment)
{
  char line[LN_LEN];
  do { fgets(line,LN_LEN,filep); } while (line[0] == '#');
  sscanf(line,"%d",&data);
  if (toggle > 0)
    fprintf(log,comment,data);
}

void getinput(MPI_FILE *filep,real &data,
	      int toggle,FILE *log,char* comment)
{
  char line[LN_LEN];
  do { fgets(line,LN_LEN,filep); } while (line[0] == '#');
  sscanf(line,"%lg",&data);
  if (toggle > 0)
    fprintf(log,comment,data);
}

void getinput(MPI_FILE *filep,char* data,
	      int toggle,FILE *log,char* comment)
{
  char line[LN_LEN];
  do { fgets(line,LN_LEN,filep); } while (line[0] == '#');
  sscanf(line,"%s",data);
  if (toggle > 0)
    fprintf(log,comment,data);
}

int
main(int argc,char**argv)
{
  System::GlobalInit(&argc,&argv);  

  Basis *basis;        /* The basis set for the calculation */
  IonRelax ionrelax;   /* Ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */
  Energies ener;       /* Holds energies */
  Control cntrl;       /* Holds convergence control data */

  /* various other local vars */
  int k;
  int niter_cg,ntotal;
  real stepsize;
  MPI_FILE *filep;
  char line[LN_LEN], elecfilename[STR_LEN], latticefilename[STR_LEN],
    ionsfilename[STR_LEN], init_Y_filename[STR_LEN], init_Y_action[STR_LEN],
    FFT_action[STR_LEN], ewaldfilename[STR_LEN], mask_filename[STR_LEN];
  int nx,ny,nz;
  int sp;
  real K_spring;
  time_t timenow;
  int my_report_level;

  /* If we're processor responsible for IO, then we get to print stuff to the screen! */
  if ( System::Get_ID() == System::Get_IO() )
    my_report_level = REPORT_LEVEL;
  else
    my_report_level = 0;

  /* Read input file for information */
  if (argc == 1)
    {
      if ( (filep = MPI_fopen(INPUTFILE,"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],INPUTFILE);
	  die(line);
	}
    }
  else if (argc == 2)
    {
      if ( (filep = MPI_fopen(argv[1],"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],argv[1]);
	  die(line);
	}
    }
  else
    {
      sprintf(line,"\nUsage:  %s [inputfile]\n\n",argv[0]);
      die(line);
    }

  timenow = time(0);
  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"Current date and time: %s\n",ctime(&timenow));
    if (argc==1)
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],INPUTFILE);
    else
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],argv[1]);
  }

  getinput(filep,ntotal,my_report_level,LOGFILE,"ntotal (# of ionic moves) = %d\n");
  getinput(filep,niter_cg,my_report_level,LOGFILE,"number of CG iterations = %d\n");
  getinput(filep,stepsize,my_report_level,LOGFILE,"stepsize = %lg\n");
  getinput(filep,elecfilename,my_report_level,LOGFILE,"electronic state file = '%s'\n");
  getinput(filep,latticefilename,my_report_level,LOGFILE,"lattice  file = '%s'\n");
  getinput(filep,ionsfilename,my_report_level,LOGFILE,"ions file = '%s'\n");
  getinput(filep,ewaldfilename,my_report_level,LOGFILE,"ewald file = '%s'\n");
  getinput(filep,init_Y_action,my_report_level,LOGFILE,"Yflag = %s");

  if (strcmp(init_Y_action,"random") == 0) 
    {
      if (my_report_level > 0) fprintf(LOGFILE,"\n");
    }
  else if (strcmp(init_Y_action,"read") == 0)
    {
      getinput(filep,init_Y_filename,my_report_level,LOGFILE," file '%s'\n");
    }
  else
    {
      sprintf(line,"%s:  initial Yflag must be 'random' or 'read'.\n\n",
	      argv[0]);
      die(line);
    }

  getinput(filep,FFT_action,my_report_level,LOGFILE,"FFTflag = %s");
  nx = ny = nz = 0;
  if (strcmp(FFT_action,"automatic") == 0)
    {
      if (my_report_level > 0) 
	fprintf(LOGFILE,"\n");
    }
  else if (strncmp(FFT_action,"specified",9) == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&nx,&ny,&nz);
      if (my_report_level > 0) 
	fprintf(LOGFILE,":  %d by %d by %d\n",nx,ny,nz);
    }
  else
    {
      sprintf(line,
	      "%s:  initial FFTflag must be 'specified' or 'automatic'.\n\n",
	      argv[0]);
      die(line);
    }

  /*
   * The ion masks capability is already included in ion setup 
   * routine. See comment in setup_ions.c
   *
   * This is to provide backward compatibility.
   */
  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",mask_filename);

  char number_test[STR_LEN];
  if ( sscanf(mask_filename,"%[.0-9]",number_test) == 1 )
    {
      // Humm, it's all positive numerical. Must be the K_spring rather than
      // the mask filename.  Assume that.
      sscanf(line, "%lg", &K_spring);
      mask_filename[0] = '\0';
      if (my_report_level > 0)
	fprintf(LOGFILE,"no maskfile provided, use masks in ion files.\n");
    }
  else 
    { 
      // Well, this is probably the mask filename.
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%lg",&K_spring);
      if (my_report_level > 0) 
	fprintf(LOGFILE,"maskfile = '%s'\n",mask_filename);
    }

  if (my_report_level > 0) 
    fprintf(LOGFILE,"K_spring = %lg\n",K_spring);

  if (K_spring <= 0.0)
    die("\nK_spring <= 0.0!!!\n\nAborting\n\n");

  if (my_report_level > 0) {
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"\n");
  }
  fclose(filep);


  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,my_report_level,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,my_report_level,LOGFILE);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ionrelax.ioninfo,ionsfilename,&elecinfo,my_report_level,LOGFILE);

  if (mask_filename[0] != '\0') {
    /* 
     * Read in the atomic masks. Overwrite whatever mask the 
     * ion file come with.
     */
    if ( (filep = MPI_fopen(mask_filename,"r")) == (MPI_FILE *)0)
      {
	sprintf(line,"\nCan't open maskfile='%s' for reading.\n",mask_filename);
	die(line);
      }
    if (my_report_level > 0) {
      fprintf(LOGFILE,"\n---------    Read overriding ion masks    ----------\n");
      fflush(LOGFILE);
    }

    for (sp=0; sp < ionrelax.ioninfo.nspecies; sp++)
      {
	int i;

	if (my_report_level > 0) 
	  fprintf(LOGFILE,"Reading masks for species=%d  natoms=%d\n",
		  sp,ionrelax.ioninfo.species[sp].natoms);
	for (i=0; i < ionrelax.ioninfo.species[sp].natoms; i++)
	  {
	    fscanf(filep,"%d",&(ionrelax.ioninfo.species[sp].move[i]));
	    if (my_report_level > 0)
	      fprintf(LOGFILE,"%d ",ionrelax.ioninfo.species[sp].move[i]);
	  }
	if (my_report_level > 0) 
	  fprintf(LOGFILE,"\n");
      }
    if (my_report_level > 0) 
      fprintf(LOGFILE,"\n");

    fclose(filep);
  }

  /* Setup Ewald calculation */
  setup_Ewald(ewaldfilename,my_report_level,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* If the flag==1, then randomize initial wavefunctions and then
   * orthonormalize them. */
  if (strcmp(init_Y_action,"random") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n-------> Setting Y to random values\n\n");
      System::seed_with_time();
      randomize_column_bundle_array(elecinfo.nkpts,elecvars.Y);
      /* Orthonormalize the Y */
      calc_UVC(&elecinfo,&elecvars);
      for (k=0; k < elecinfo.nkpts; k++)
	elecvars.Y[k] = elecvars.C[k];
    }
  else
    {
      /* Try to read Y-file */
      if (my_report_level > 0)
	{
	  // cannot use MPI_fopen, since we don't want to read in
	  // the whole binary file!!
	  FILE* testfilep = fopen(init_Y_filename,"r");
	  if (testfilep == (FILE *)0)
	    {
	      sprintf(line,
		      "\nCan't open '%s' to read initial wave-functions.\n\n",
		      init_Y_filename);
	      die(line);
	    }
	  fclose(testfilep);

	  fprintf(LOGFILE,
		  "\n-------> Reading Y from '%s'\n\n",init_Y_filename);
	}
      read_column_bundle_array(init_Y_filename,elecinfo.nkpts,elecvars.Y);
    }

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     my_report_level,LOGFILE);

  /* Setup signal handling */
  setup_signals(&elecinfo,&elecvars,LOGFILE);


  /*
   * Start the ion relaxation loop
   *
   */
  ionrelax.init();

  for (int ioniter = 0; ioniter < ntotal; ioniter++)
    {
      /* status */
      if (my_report_level > 0) {
	fprintf(LOGFILE,"\n>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<\n");
	fprintf(LOGFILE,"IonCG: relaxation %d out of %d\n",ioniter,ntotal-1);
	fprintf(LOGFILE,"IonCG: initial stepsize = %lg.\n",K_spring);
	fprintf(LOGFILE,"\n");
	fflush(LOGFILE);
      }

      //
      // Given the current ion position, calculate the ion forces.
      //
      /* Calculate core and Ewald energies */
      calc_core_ewald_pulay_energies(basis,&ionrelax.ioninfo,&elecinfo,&ener,
			       my_report_level, LOGFILE);
      /* Calculate the local pseudopotential */
      Vloc_pseudo(&basis[elecinfo.nkpts],&ionrelax.ioninfo,elecvars.Vlocps.c,my_report_level,LOGFILE);
      /* Calculate the non-local pseudopotential for FHI CPI type psp. */
      Vnl_pseudo(basis,&ionrelax.ioninfo,&elecinfo,my_report_level,LOGFILE);

      /* Do conjugate gradients on electronic variables*/
      stepsize = minimize_elec_pcg_nocosgpg(niter_cg,stepsize,
					    basis,&ionrelax.ioninfo,&elecinfo,&elecvars,
					    &ener,cntrl,my_report_level,LOGFILE);
      /* Calculate the forces on the ions */
      calc_ionic_forces(&elecinfo,&elecvars,&ionrelax.ioninfo,basis,
			my_report_level, LOGFILE);
      
      /* print out the force on the ions */
      if (my_report_level > 0 ) ionrelax.print_forces(LOGFILE);

      ionrelax.impose_sym();
      
      //
      // The ionic forces ARE the negative gradient.
      //
      /* Square length of gradients */
      ionrelax.abs2_grad_old = ionrelax.abs2_grad_now;
      ionrelax.abs2_grad_now = ionrelax.f_dot_f();

      // Some linear minimization statistics from last iteration 
      if (ioniter > 0)
	{
	  ionrelax.linminfact = ionrelax.f_dot_dir()/ionrelax.fold_dot_dir();
	  ionrelax.cosgg = 4.0*ionrelax.f_dot_fold()/
	    sqrt( 16.0 * ionrelax.abs2_grad_now * ionrelax.abs2_grad_old);
	  if (my_report_level > 0)
	    {
	      fprintf(LOGFILE,"\nIonCG: linmin = %8.1le   cosgg = %8.1le\n\n",
		      ionrelax.linminfact, ionrelax.cosgg);
	      fflush(LOGFILE);
	    }
	}

      //
      // If this is the last iteration, don't do linear minimization, but
      // quit the iteration loop. (perhaps report the last position?)
      //
      if (ioniter == ntotal-1) {
	if (my_report_level > 0) ionrelax.print_pos(LOGFILE);
	break;
      }

      // Calculate search direction
      ionrelax.alpha = 0.0;
      // if we have done a "good" linmin, use CG:
      // i.e. calculate gamma
      if (ioniter > 0 
	  && fabs(ionrelax.linminfact) < 0.05 
	  && fabs(ionrelax.cosgg) < 0.1)
	ionrelax.alpha = ionrelax.abs2_grad_now/ionrelax.abs2_grad_old;
      if (my_report_level > 0)
	{
	  fprintf(LOGFILE,"IonGG: |grad| = %le\n",2.0*sqrt(ionrelax.abs2_grad_now));
	  fprintf(LOGFILE,"IonCG: alpha = %8.1le\n",ionrelax.alpha);
	  fflush(LOGFILE);
	}

      // 
      // Calculate current search direction:
      //  dir = 2.0 * force_now + alpha * dir_old 
      if (ioniter == 0)
	ionrelax.dir_zero_out();
      else
	ionrelax.dir_scale(ionrelax.alpha);
      ionrelax.dir_add_f();
// removed by Shanhui      ionrelax.dir_add_f();

      //
      // Do a linear minimization along dir
      //

      K_spring = ionrelax.do_linmin(K_spring, basis, elecinfo, elecvars, 
				    ener, cntrl, niter_cg, stepsize, 
				    my_report_level, LOGFILE);

      // copy force now to old force
      ionrelax.f_to_fold();
      
      /* Print positions out! */
      if (my_report_level > 0) ionrelax.print_pos(LOGFILE);

    } /* ioniter loop */

  /* Write out final electronic variables */
  if (my_report_level > 0) 
    {
      fprintf(LOGFILE,"\nDone!  Dumping final variables:\n\n");
      fflush(LOGFILE);
    }
  dump_and_stamp_elecvars(elecinfo.nkpts,elecvars.C,&(elecvars.n),
			  &(elecvars.d),&(elecvars.Vlocps),
			  &(elecvars.Vscloc),elecvars.Hsub_evecs,
			  my_report_level,
			  LOGFILE);
  if (my_report_level > 0)
    {
      fprintf(LOGFILE,"\n");
      fflush(stdout);
    }

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  ionrelax.freemem();
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  /* End the MPI stuff */
  System::GlobalFinalize();

  return 0;
}

