/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,           May 13 1997
 *
 * This program reads in orthonormalized wave-functions (elecvars->C)
 * and calculates the forces on the ions (in lattice coordinates).
 *
 * There are two invocations of the program from the command-line:
 * (1) with no arguments, the file INPUTFILE (below) is read
 * (2) with one argument, which is the name of the input file to be read.
 *
 * The format of the input file is:
 *
 * #
 * # comments are lines that start with '#'
 * #
 * elecfile    # name of file with electronic state information
 * latfile     # name of file with the lattice vectors
 * ionsfile    # name of file with ionic info:  Z, positions, pseudopots,...
 * ewaldfile   # name of file with ewald setup information
 * C_filename  # name of file to read for orthonormal wave-functions
 * <FFTflag>   # a string describing how FFT box sizes are picked
 * [nx ny nz]  # x,y,z FFT box sizes if FFTflag == 'specified'
 *
 * <FFTflag> must be either 'automatic' or 'specified'.  'automatic'
 * means the program figures out its on FFT box sizes.  'specified'
 * means it tries to use the sizes provided on the next line.
 *
 */

/* $Id: forces.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
/* My header file */
#include "header.h"

#define INPUTFILE    "forces.in"
#define REPORT_LEVEL 1
#define LOGFILE      stdout

#define LN_LEN  150
#define STR_LEN 80


int
main(int argc,char**argv)
{
  /* Initialize MPI through column_bundle class. */
  System::GlobalInit(&argc,&argv);  

  Basis* basis;        /* The basis set for the calculation */
  Ioninfo ioninfo;     /* Ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */
  Energies ener;       /* Holds energies */
  Control cntrl;       /* Holds convergence control data */
  Parameters params;   /* Main program parameters */

  /* various other local vars */
  int k;
  MPI_FILE *filep;
  char line[LN_LEN], elecfilename[STR_LEN], latticefilename[STR_LEN],
    ionsfilename[STR_LEN], C_filename[STR_LEN],
    FFT_action[STR_LEN], ewaldfilename[STR_LEN];
  int nx,ny,nz;
  time_t timenow;
  int my_report_level;

  /* If we're processor responsible for IO, then we get to print stuff to the screen! */
  if (System::Get_ID() == System::Get_IO())
    my_report_level = REPORT_LEVEL;
  else
    my_report_level = 0;
  
  /* Read input file for information */
  if (argc == 1)
    {
      if ( (filep = MPI_fopen(INPUTFILE,"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],INPUTFILE);
	  die(line);
	}
    }
  else if (argc == 2)
    {
      if ( (filep = MPI_fopen(argv[1],"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],argv[1]);
	  die(line);
	}
    }
  else
    {
      sprintf(line,"\nUsage:  %s [inputfile]\n\n",argv[0]);
      die(line);
    }

  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    fprintf(LOGFILE,"******************************************************\n");
  }
  timenow = time(0);

  if (my_report_level > 0) {
    fprintf(LOGFILE,"Current date and time: %s\n",ctime(&timenow));
    if (argc==1)
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],INPUTFILE);
    else
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],argv[1]);
  }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",elecfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"electronic state file = '%s'\n",elecfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",latticefilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"lattice  file = '%s'\n",latticefilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ionsfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ions file = '%s'\n",ionsfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ewaldfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ewald file = '%s'\n",ewaldfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",C_filename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"C_filename = '%s'\n",C_filename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",FFT_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"FFTflag = %s",FFT_action);

  nx = ny = nz = 0;
  if (strcmp(FFT_action,"automatic") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(FFT_action,"specified") == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&nx,&ny,&nz);
      if (my_report_level > 0)
	fprintf(LOGFILE,":  %d by %d by %d\n",nx,ny,nz);
    }
  else
    {
      sprintf(line,
	      "%s:  initial FFTflag must be 'specified' or 'automatic'.\n\n",
	      argv[0]);
      die(line);
    }

  if (my_report_level > 0) {
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"\n");
  }

  fclose(filep);

  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,my_report_level,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,my_report_level,LOGFILE);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ioninfo,ionsfilename,&elecinfo,my_report_level,LOGFILE);

  /* Setup ewald calculation */
  setup_Ewald(ewaldfilename,my_report_level,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     my_report_level,LOGFILE);

  /* Calculate the local pseudopotential */
  Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c,my_report_level,LOGFILE);
  /* Calculate the non-local pseudopotential for FHI CPI type psp. */
  Vnl_pseudo(basis,&ioninfo,&elecinfo,my_report_level,LOGFILE);


  /* Read in orthonormal wave-functions */
  if (my_report_level > 0)
    {
      // cannot use MPI_fopen, since we don't want to read in
      // the whole binary file!!
      FILE* testfilep = fopen(C_filename,"r");
      if (testfilep == (FILE *)0)
	{
	  sprintf(line,
		  "\nCan't open '%s' to read C wave-functions.\n\n",
		  C_filename);
	  die(line);
	}
      fclose(testfilep);

      fprintf(LOGFILE,"\n-------> Reading C from '%s'\n\n",C_filename);
    }
  read_column_bundle_array(C_filename,elecinfo.nkpts,elecvars.C);

  /* Calculate charge density n from the C */
  if (my_report_level > 0)
    fprintf(LOGFILE,"Calculating charge density...\n"); fflush(LOGFILE);
  calc_n(&elecinfo,&elecvars,&ioninfo);

  /* Solve the Poisson eqaution to get Hartree field */
  if (my_report_level > 0)
    fprintf(LOGFILE,"Solving Poisson equation...\n"); fflush(LOGFILE);
  solve_poisson(&elecvars);

  /* Calculate the energy of the configuration */
  if (my_report_level > 0)
    fprintf(LOGFILE,"Calculating the energy of the configuration...\n");
  calc_all_energies(basis,&ioninfo,&elecinfo,&elecvars,&ener,
		    my_report_level,LOGFILE);

  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    print_energies(&elecinfo,&ener,LOGFILE);
  }

  /* Calculate the forces on the ions */
  calc_ionic_forces(&elecinfo,&elecvars,&ioninfo,basis,
		    my_report_level,LOGFILE);

  if (my_report_level > 0)
    {
      /* Print them out! */
      int sp;

      for (sp=0; sp < ioninfo.nspecies; sp++)
	{
	  int i;
	  fprintf(LOGFILE,"\nspecies = %d,  natoms = %d,  forces follow:\n\n",
		  sp,ioninfo.species[sp].natoms);
	  for (i=0; i < ioninfo.species[sp].natoms; i++)
	    fprintf(LOGFILE,"%20.10le%20.10le%20.10le\n",
		    ioninfo.species[sp].forces[i].v[0],
		    ioninfo.species[sp].forces[i].v[1],
		    ioninfo.species[sp].forces[i].v[2]  );
	}
      fprintf(LOGFILE,"\n");
      fflush(LOGFILE);
    }

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  free_ioninfo(&ioninfo);
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  /* End the MPI stuff */
  System::GlobalFinalize();

  return 0;
}
