/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,           May 13 1997
 *
 * Compute empty bands via CG
 *
 */

/* $Id: empties.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
/* My header file */
#include "header.h"

#define INPUTFILE    "empties.in"
#define REPORT_LEVEL 1
#define LOGFILE      stdout

#define LN_LEN  150
#define STR_LEN 80



int
main(int argc,char**argv)
{
  /* Initialize MPI through column_bundle class. */
  System::GlobalInit(&argc,&argv);  

  Basis *basis;        /* The basis set for the calculation */
  Ioninfo ioninfo;     /* Ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */
  Control cntrl;       /* Holds convergence control data */

  /* various other local vars */
  int k;
  dft_text_FILE *filep;
  char line[LN_LEN],elecfilename[STR_LEN],latticefilename[STR_LEN],ionsfilename[STR_LEN],
       C_filled_filename[STR_LEN],FFT_action[STR_LEN],ewaldfilename[STR_LEN],
       init_C_empty_action[STR_LEN],C_empty_filename[STR_LEN];
  int nx,ny,nz;
  int nbands_empty,niter_cg;
  real stepsize;
  int my_report_level;

  /* If we're processor responsible for IO, then we get to print stuff to the screen! */
  if ( System::Get_ID() == System::Get_IO() )
    my_report_level = REPORT_LEVEL;
  else
    my_report_level = 0;

  /* Read input file for information */
  if (argc == 1)
    {
      if ( (filep = MPI_fopen(INPUTFILE,"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],INPUTFILE);
	  die(line);
	}
    }
  else if (argc == 2)
    {
      if ( (filep = MPI_fopen(argv[1],"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],argv[1]);
	  die(line);
	}
    }
  else
    {
      sprintf(line,"\nUsage:  %s [inputfile]\n\n",argv[0]);
      die(line);
    }

  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    fprintf(LOGFILE,"******************************************************\n");
    if (argc==1)
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],INPUTFILE);
    else
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],argv[1]);
  }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%d",&niter_cg);
  if (my_report_level > 0)
    fprintf(LOGFILE,"niter_cg = %d\n",niter_cg);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%lg",&stepsize);
  if (my_report_level > 0)
    fprintf(LOGFILE,"stepsize = %lg\n",stepsize);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",elecfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"electronic state file = '%s'\n",elecfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",latticefilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"lattice  file = '%s'\n",latticefilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ionsfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ions file = '%s'\n",ionsfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ewaldfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ewald file = '%s'\n",ewaldfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",C_filled_filename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"C_filled_filename = '%s'\n",C_filled_filename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",FFT_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"FFTflag = %s",FFT_action);
  nx = ny = nz = 0;
  if (strcmp(FFT_action,"automatic") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(FFT_action,"specified") == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&nx,&ny,&nz);
      if (my_report_level > 0)
	fprintf(LOGFILE,":  %d by %d by %d\n",nx,ny,nz);
    }
  else
    {
      sprintf(line,
	      "%s:  initial FFTflag must be 'specified' or 'automatic'.\n\n",
	      argv[0]);
      die(line);
    }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%d",&nbands_empty);
  if (my_report_level > 0)
    fprintf(LOGFILE,"nbands_empty = %d\n",nbands_empty);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",init_C_empty_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"C_emptyflag = %s",init_C_empty_action);

  if (strcmp(init_C_empty_action,"random") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(init_C_empty_action,"read") == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%s",C_empty_filename);
      if (my_report_level > 0)
	fprintf(LOGFILE," file '%s'\n",C_empty_filename);
    }
  else
    {
      sprintf(line,"%s:  initial C_emptyflag must be 'random' or 'read'.\n\n",
	      argv[0]);
      die(line);
    }

  if (my_report_level > 0) {
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"\n");
  }

  dft_text_fclose(filep);

  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,my_report_level,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,my_report_level,LOGFILE);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ioninfo,ionsfilename,&elecinfo,my_report_level,LOGFILE);

  /* Setup ewald calculation */
  setup_Ewald(ewaldfilename,my_report_level,LOGFILE);

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     my_report_level,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* Calculate the local pseudopotential */
  Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c,my_report_level,LOGFILE);
  /* Calculate the non-local pseudopotential for FHI CPI type psp. */
  Vnl_pseudo(basis,&ioninfo,&elecinfo,my_report_level,LOGFILE);

  /* Read in orthonormal wave-functions */
  if (my_report_level > 0)
    {
      // cannot use MPI_fopen, since we don't want to read in
      // the whole binary file!!
      FILE* testfilep = fopen(C_filled_filename,"r");
      if (testfilep == (FILE *)0)
	{
	  sprintf(line,
		  "\nCan't open '%s' to read orthonormal wave-functions.\n\n",
		  C_filled_filename);
	  die(line);
	}
      fclose(testfilep);
      fprintf(LOGFILE,"\n-------> Reading C_filled from '%s'\n\n",
	      C_filled_filename);
      fflush(LOGFILE);
    }
  read_column_bundle_array(C_filled_filename,elecinfo.nkpts,elecvars.C);

  /* Calculate charge density n from the C */
  if (my_report_level > 0) {
    fprintf(LOGFILE,"Calculating charge density...\n"); fflush(LOGFILE);
  }
  calc_n(&elecinfo,&elecvars,&ioninfo);

  /* Solve the Poisson eqaution to get Hartree field and calculate Vscloc*/
  if (my_report_level > 0) {
    fprintf(LOGFILE,"Solving Poisson equation...\n"); fflush(LOGFILE);
  }
  solve_poisson(&elecvars);
  if (my_report_level > 0) {
    fprintf(LOGFILE,"Calculating Vscloc...\n"); fflush(LOGFILE);
  }
  calc_Vscloc(&elecvars);

  /* Do the empties!!! */
  calc_empties(niter_cg,stepsize,
	       nbands_empty,basis,&elecinfo,&elecvars,&ioninfo,
	       init_C_empty_action,C_empty_filename,
	       my_report_level,LOGFILE);

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  free_ioninfo(&ioninfo);
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  /* End the MPI stuff */
  System::GlobalFinalize();

  return 0;
}
