/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,  Jule 21 1997
 * 
 */

/* $Id: dumpbanddens.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
/* My header file */
#include "header.h"

#define REPORT_LEVEL 1

int
main(int argc,char**argv)
{
  Basis *basis; /* basis set */
  Ioninfo ioninfo; /* ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* Electronic variables */
  Control cntrl;       /* Holds convergence control data */

  /* various other local vars */
  FILE *filep;
  char *latfilename,*elecfilename,*ionsfilename,
       Cin_filename[100],line[100];
  int nx,ny,nz;
  int k,band,j;

  /* Read input file for information */
  if (argc == 7)
    {
      latfilename  = argv[1];
      elecfilename  = argv[2];
      ionsfilename  = argv[3];
      sscanf(argv[4],"%d",&nx);
      sscanf(argv[5],"%d",&ny);
      sscanf(argv[6],"%d",&nz);
    }
  else if (argc == 4)
    {
      latfilename  = argv[1];
      elecfilename  = argv[2];
      ionsfilename = argv[3];
      nx = ny = nz = 0;
    }
  else
    {
      sprintf(line,"\nUsage:  %s latfile elecfile ionsfile [nx ny nz]\n\n",argv[0]);
      die(line);
    }
  printf("latfile = '%s'\n",latfilename);
  printf("elecfile = '%s'\n",elecfilename);
  printf("ionsfile = '%s'\n",ionsfilename);
  printf("nx=%d ny=%d nz=%d\n",nx,ny,nz);

  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,1,stdout);

  /* Read the lattice vectors and set up the bases */
  setup_basis(basis,latfilename,elecinfo,nx,ny,nz,1,stdout);

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     1,stdout);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ioninfo,ionsfilename,&elecinfo,REPORT_LEVEL,stdout);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* Read in wavefunctions or calculate them */
  for(;;)
    {
      printf("\n\n[R]ead diagonalized wavefunctions or [C]alculate them? ");
      scanf("%s",line);
      if (line[0] == 'C' || line[0] == 'R')
	break;
      else
	printf("\n\nInvalid choice:  please enter 'R' or 'C'.\n");
    }

  /* Read in orthonormal wave-functions which are the diagnoal basis of 
   * the subspace Hamiltonian(s) */
  if (line[0]=='R')
    {
      printf("\nFile to read containing diagonal basis wavefunctions: ");
      scanf("%s",Cin_filename);
      if ( (filep=fopen(Cin_filename,"r")) == (FILE *)0)
	{
	  sprintf(line,"\nCan't open '%s' for reading.\n\n",Cin_filename);
	  die(line);
	}
      fclose(filep);
      printf("\n-------> Reading C from '%s'\n\n",Cin_filename);
      fflush(stdout);
      read_column_bundle_array(Cin_filename,elecinfo.nkpts,elecvars.C);
    }
  /* Otherwise, calculate them by diagonalizing the subspace Hamiltonians
   * and write out to disk. */
  else
    {
      /* Calculate the local pseudopotential */
      Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c,REPORT_LEVEL,stdout);
      /* Calculate the non-local pseudopotential for FHI CPI type psp. */
      Vnl_pseudo(basis,&ioninfo,&elecinfo,REPORT_LEVEL,stdout);

      printf("\nFile to read containing orthonormal wavefunctions: ");
      scanf("%s",Cin_filename);
      if ( (filep=fopen(Cin_filename,"r")) == (FILE *)0)
	{
	  sprintf(line,"\nCan't open '%s' for reading.\n\n",Cin_filename);
	  die(line);
	}
      fclose(filep);
      printf("\n-------> Reading C from '%s'\n",Cin_filename);
      fflush(stdout);
      read_column_bundle_array(Cin_filename,elecinfo.nkpts,elecvars.C);
      printf("\n-------> Calculating total charge density\n");
      fflush(stdout);
      calc_n(&elecinfo,&elecvars,&ioninfo);
      printf("\n-------> Solving Poisson's equation\n");
      fflush(stdout);
      solve_poisson(&elecvars);
      printf("\n-------> Calculating Hsub and diagonalizing\n");
      fflush(stdout);
      calc_Hsub(&ioninfo,&elecinfo,&elecvars);
      printf("\n-------> Changing C to diagonal basis\n");
      fflush(stdout);
      for (k=0; k < elecinfo.nkpts; k++)
	{
	  do_column_bundle_matrix_mult(elecvars.C[k],
				       elecvars.Hsub_evecs[k],
				       elecvars.Y[k]);
	  elecvars.C[k] = elecvars.Y[k];
	}
      printf("\n-------> Writing out 'HsubC' to disk\n\n");
      fflush(stdout);
      if ( (filep=fopen("HsubC","w")) == (FILE *)0)
	die("\n\nCan't open 'HsubC' for writing\n\n");
      fclose(filep);
      write_column_bundle_array("HsubC",elecinfo.nkpts,elecvars.C);
    }

  /* Now ask for the bands whose densities we want and spit them out */
  printf("\n\nEnter k and band numbers for density dump (-1 to quit)\n");
  for(;;)
    {
      printf("\nk: ");
      scanf("%d",&k);
      if (k==-1)
	break;
      printf("band: ");
      scanf("%d",&band);
      if (band==-1)
	break;
      printf("Calculating charge density for k=%d  band=%d:  epsilon=%le\n",
	     k,band,elecvars.Hsub_eigs[k][band]);

      /* Use the space for the total charge-density as workspace */
      vector &n = elecvars.n;
      
      /* Space to store the band we're interested in */
      column_bundle Ccol(1,basis[k].nbasis,&basis[k],"local");
      column_bundle ICcol(1,basis[k].NxNyNz,&basis[k],"local");

      /* Copy the band from  C[k] */
      for (j=0; j < basis[k].nbasis; j++)
	Ccol.col[0].c[j] = elecvars.C[k].col[band].c[j];

      /* Fourier transform the band */
      apply_I(Ccol,ICcol);

      /* Calculate the density for the band */
      n.zero_out();
      for (j=0; j < basis[k].NxNyNz; j++)
#if defined SCALAR_IS_COMPLEX
	n.c[j] += ICcol.col[0].c[j].x*ICcol.col[0].c[j].x +
	          ICcol.col[0].c[j].y*ICcol.col[0].c[j].y ;
#elif defined SCALAR_IS_REAL
	n.c[j] += ICCol.col[0].c[j]*ICcol.col[0].c[j];
#else
#error scalar is neither complex nor real
#endif

      /* Write out density to disk */
      sprintf(line,"n.%d.%d",k,band);
      printf("Writing out file '%s'\n",line);
      elecvars.n.write(line);
    }

  printf("\n\nDone!  Quitting.\n\n");
  /* Free up all the used memory */
  free_elecvars(&elecinfo,&elecvars);
  free_basis(basis,elecinfo.nkpts);
  free_elecinfo(&elecinfo);
  free_ioninfo(&ioninfo);
}
