/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

//
// Sohrab Ismail-Beigi     August 2 1999
//
// This is the main DFT++ executable.
// It handles electronic minimizations, calculation of forces, and
// ionic dynamics, and ionic relaxations.
//
// It has been cleaned up a bit and the structure should be clearer.
// The basic idea is to do any MPI setup if needed, then
// to try to read the input file, and then to do any minimizations/force/
// dynamics as needed, and finally to dump the current state and exit.
//

/* $Id: dft.c,v 1.5 1999/12/19 16:31:54 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <time.h>
// My header files
#include "header.h"
#include "parallel.h"

// Default input filename
#define DEFAULT_INPUTFILE    "dft.in"

// This report level reports all the basic messages we might
// be interested to the log file.  More or less anal options
// can be found in header.h
#define REPORT_LEVEL       DFT_BASIC_LOG


/***************************************************************************
 *                                                                         *
 * Main Program:                                                           *
 *                                                                         *
 ***************************************************************************/
int
main(int argc,char**argv)
{
  // Do any system dependent setup if needed (e.g. MPI)
  System::GlobalInit(&argc,&argv);  

#ifdef DFT_PROFILING
  /* Initialize timers */
  timerActivateAll();

  timerOn(0);
#endif // DFT_PROFILING

  // The variables needed to do the work..
  Basis* basis;        // The basis set for different k-points the calc.
  IonDyninfo iondyninfo;     // Dynamic Ionic information
  Ioninfo &ioninfo = iondyninfo.ioninfo;
  Elecinfo elecinfo;   // Electronic state information
  Elecvars elecvars;   // The electronic variables: Y, C, U, n, ... 
  Energies ener;       // Holds energies
  Control cntrl;       // Holds convergence control data
  Parameters params;    // main program parameters.
  

  /**********************************************************
   **                                                      **
   ** Reading command line, setting up logfile.            **
   **                                                      **
   **********************************************************/
  // initialize input/output files to defaults.
  char input_filename[DFT_FILENAME_LEN] = DEFAULT_INPUTFILE;
  char log_filename[DFT_FILENAME_LEN] = "";

  // Parse command line and get input and log filenames.
  parse_command_line(argc,argv,input_filename,log_filename);

  // Open the logfile and make the system logger point to it
  Output logfile(REPORT_LEVEL,log_filename);
  if (System::Get_procID() != System::Get_IOprocID())
    logfile.report_level = DFT_SILENCE-1;
  System::global_log = &logfile;

  

  // Open the input file for information
  dft_text_FILE *inputfile;
  if ( (inputfile=dft_text_fopen(input_filename,"r")) == (dft_text_FILE *)0 )
    die("\n%s:  can't read '%s'.  Aborting.\n\n",argv[0], input_filename);

  // expand all includes in the input file.
  inputfile->expand_include();

  // Print the current time and date
  time_t timenow = time(0);
  dft_log("\n******************************************************\n");
  dft_log("Current date and time: %s\n",ctime(&timenow));
  dft_log("%s:  reading file '%s'\n",argv[0],input_filename);


  /**********************************************************
   **                                                      **
   ** Setup section: read input file and print the current **
   ** variables read from there, then setup electronic     **
   ** variables and prepare wavefunctions.                 **
   **                                                      **
   **********************************************************/
  setup(inputfile, params, elecinfo, &basis, iondyninfo, cntrl);
  dft_text_fclose(inputfile);

  print_template(params, elecinfo, basis, iondyninfo, cntrl);

  // Setup and allocate the electronic variables
  init_elecvars(&elecinfo,basis,&elecvars);

  //    Setup the signal handlers
  // the signal setup is disable right now. only a skeleton is left.
  //   setup_signals(&elecinfo,&elecvars);

  //Prepare the wavefunctions, either random-initialize or read from file.
  prepare_wavefunctions(params, elecinfo, elecvars, iondyninfo);


#ifdef DFT_PROFILING
  timerOff(0);  // turn off initialization timer.
  timerOn(1);  // turn on total computation timer.
#endif // DFT_PROFILING



  /*
   * We could ask the code to do some things specifically for us here
   * without going into the main loop.
   *
   */
  if (params.testing_flag) {
    System::GlobalFinalize();
    exit(0);
  }


  /*********************************************************
   **                                                     **
   ** Ion Movement Loop                                   **
   **                                                     **
   *********************************************************/
  int ion_loop_iter = 0;
  int break_ionloop = DFT_FALSE;

  // Start the ion loop.
  // Elec_minim/forces only jobs go through this only once.
  // We go as long as we're told not to break this loop...
  while(!break_ionloop) {

    // print status
    dft_log("\n>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<\n");
    if ( params.ionic_dynamics_flag || params.ionic_relaxation_flag )
      dft_log("Ionic move %d , %d out of %d\n",
	      ion_loop_iter+iondyninfo.ion_start_iter,
	      ion_loop_iter,
	      cntrl.max_ionic_steps-1);
    
    // Calculate core and Ewald energies and local pseudopotential
    // based on current ionic positions.
    calc_core_ewald_pulay_energies(basis,&ioninfo,&elecinfo,&ener);
    Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c);

    // Also set up non-linear core correction if needed. */
    nlcore_setup(ioninfo, elecvars);

    // Iterative minimization on electronic variables if asked for.
    if ( params.electronic_minimization_flag ) {
      switch (cntrl.algorithm) {
      case DFT_CG : 
	minimize_elec_cg(basis,&ioninfo,&elecinfo,&elecvars,
			 &ener,cntrl);
	break;
      case DFT_CG_nocos :
	minimize_elec_cg_nocosgg(basis,&ioninfo,&elecinfo,&elecvars,
				 &ener,cntrl);
	break;
      case DFT_PCG :
	minimize_elec_pcg(basis,&ioninfo,&elecinfo,&elecvars,
			  &ener,cntrl);
	break;
      case DFT_PCG_nocos :
	minimize_elec_pcg_nocosgpg(basis,&ioninfo,&elecinfo,&elecvars,
				   &ener,cntrl);
	break;
      case DFT_EOM :
	minimize_elec_eom(basis,&ioninfo,&elecinfo,&elecvars,
			  &ener,cntrl);
	break;
      case DFT_PSD :
	minimize_elec_psd(basis,&ioninfo,&elecinfo,&elecvars,
			  &ener,cntrl);
	break;
      }
    }

    // calculate the forces if asked for and print them out.
    if ( params.calculate_forces_flag ) {
      if ( !params.electronic_minimization_flag ) {
	calc_UVCn_d(&elecinfo,&elecvars,&ioninfo);
      }
      calc_ionic_forces(&elecinfo,&elecvars,&ioninfo,basis);
      ioninfo.print_force(basis[elecinfo.nkpts].invR);
    }

    // If doing ionic dynamics...
    if ( params.ionic_dynamics_flag )
      {
	// dump charge density to file for book-keeping.
	// extrapolate wavefunction from current Y and saved old Y.
	iondyninfo.dump_n_extrapolate_Y(elecvars, cntrl,
					elecinfo.nkpts, ion_loop_iter);
	// Use Verlet algorithm to update ion positions.
	iondyninfo.iondyn( ion_loop_iter+iondyninfo.ion_start_iter );

	// Update ionic dyamics counter and decide if we're done
	ion_loop_iter++;
	break_ionloop = (ion_loop_iter >= cntrl.max_ionic_steps);

      }
    // or relaxations...
    else if ( params.ionic_relaxation_flag )
      {
	int force_tolerance_reached = DFT_FALSE;

	// Relax ions.
	force_tolerance_reached = iondyninfo.relax_ion(cntrl);

	ion_loop_iter++;
	break_ionloop = (force_tolerance_reached ||
			 (ion_loop_iter >= cntrl.max_ionic_steps));
	
      }
    // Otherwise, we should quit this ionic loop!
    else
      break_ionloop = DFT_TRUE;

  } // while (!breaking the ion loop)


  // Dump orbital density for chosen k-point and band.
  if (params.number_of_partial_n_lines > 0)
    calc_partial_n(params, &elecinfo, &elecvars);


  // If doing ionic dynamics, write out final ionic config.
  // for current and previous timestep to enable restart.
  // The filanmes are IONLOC and IONLOC.OLD for now.
  // Also dumps the current fillings to FILLINGS. (Tairan  4/14/1999)
  if ( params.ionic_dynamics_flag ) {
    // In the future, binary dump for continuation is also done here (??)
    Output ionloc("IONLOC","w");
    iondyninfo.print_pos(&ionloc, 0, 
			 ion_loop_iter+iondyninfo.ion_start_iter); // current positions
    Output ionloc_old("IONLOC.OLD","w");
    iondyninfo.print_pos(&ionloc_old, -1);  // old positions
    Output fillings("FILLINGS","w"); // fillings
    elecinfo.print_fillings(&fillings);
  }

#ifdef DFT_PROFILING
  timerOff(1); // turn off total computation timer.
#endif // DFT_PROFILING


  // Write out final electronic variables
  dft_log("\nDone!  Dumping final variables:\n\n");
  dft_log_flush();
  dump_and_stamp_variables(elecinfo,elecvars);

#ifdef DFT_PROFILING
  timer_counter_Report();
#endif // DFT_PROFILING

  // Free up all the used memory
  free_basis(basis,elecinfo.nkpts);
  // free_ioninfo(&ioninfo); // <= would be taken care of when compiler calls
  //     iondyninfo distructor.
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  // End any system dependent stuff (e.g. MPI)
  System::GlobalFinalize();

  return 0;
}
