/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,           April 1997
 *
 * Takes an ASCII dump of wavefunction from cgrad (via stdin) and
 * writes of a file 'C' containing the binary data in the format
 * used by my program.
 *
 */

/* $Id: convertwfs.c,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
/* My header file */
#include "header.h"

#define REPORT_LEVEL 1
#define LOGFILE      stdout

int
main(int argc,char**argv)
{
  Basis *basis;         /* The basis set for the calculation */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */

  int nx,ny,nz;
  char line[200];
  char elecfilename[80],latticefilename[80];

  /* Read input file for information */
  if (argc != 6)
    {
      sprintf(line,"\nUsage:  %s nx ny nz <elecfile> <latticefile>\nwhere n_i are FFT box sizes.\n\n",
	      argv[0]);
      die(line);
    }
  sscanf(argv[1],"%d",&nx);
  sscanf(argv[2],"%d",&ny);
  sscanf(argv[3],"%d",&nz);
  sscanf(argv[4],"%s",&elecfilename);
  sscanf(argv[5],"%s",&latticefilename);
  fprintf(LOGFILE,"\nnx = %d   ny = %d  nz = %d\n\n",nx,ny,nz);

  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,REPORT_LEVEL,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,REPORT_LEVEL,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* Now we do the work! */
  int k,band,n,ib,ik,m,index,nbasis;
  int x,y,z;
  real f;
  complex c;

  printf("\n\nStaring conversion:  reading from stdin\n\n");

  for (k=0; k < elecinfo.nkpts; k++)
    nbasis = basis[k].nbasis;
    for (band=0; band < elecinfo.nbands; band++)
      {
	printf("Working on k=%d, band=%d\n",k,band);
	fflush(stdout);
	for (n=0; n < nbasis; n++)
	  {
	    scanf("%d %d %d %d %d %lg %lg %lg",&ik,&ib,&x,&y,&z,&f,
		  &c.x,&c.y);
	    if ( (ik-1)!=k ) {
	      sprintf(line,"\n(k==%d)+1 != ik==%d !!!\n\n",k,ik);
	      die(line);
	    }
	    if ( (ib-1)!=band ) {
	      sprintf(line,"\n(band==%d)+1 != ib==%d !!!\n\n",band,ib);
	      die(line);
	    }
	    index = z+nz*(y+ny*x);
	    for (m=0; m < nbasis; m++)
	      if (basis[k].index[m] == index)
		break;
	    if (m==nbasis) {
	      sprintf(line,"Can't find ik=%d ib=%d (%d,%d,%d) !!!\n\n",
		      ik,ib,x,y,z);
	      die(line);
	    }
	    elecvars.C[k].col[band].c[m] = c;
	  }
      }

  /* Write out C */
  fprintf(LOGFILE,"\nDone converting!  Dumping C.\n\n");
  fflush(stdout);
  write_column_bundle_array("C",elecinfo.nkpts,elecvars.C);

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);
}
