/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* 
 *   Sohrab Ismail-Beigi             May 1997
 *
 * A set of routines to calculate the empty bands via CG minimization
 * of an objective function (sum of empty state eigenvalues) with
 * constraints of orthonormality of the empty states with each other
 * and orthogonality to the fillled states.
 *
 */

/* $Id: calcempties.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <time.h>
#include <string.h>

#include "header.h"

/*
 * Does A_empty*Y where A_empty = Y - C_filled*C_filled^*O*Y,
 * i.e., project out the filled states from Y
 */
static column_bundle
A_empty(const column_bundle &C_filled,const column_bundle &Y)
{
  column_bundle result(Y);

  result -= C_filled*(C_filled^(O(Y)));
  return result;
}

/*
 * U = (A*Y)^O(A*Y)   and   C = A*Y*U^(-1/2),
 * i.e. orthonormalized the Y into C for empty bands.
 */
static void
calc_UC_empty(Elecinfo *empty_info,Elecvars *filled_vars,
	      Elecvars *empty_vars)
{
  int k, nkpts = empty_info->nkpts;
  column_bundle *Y = empty_vars->Y;
  column_bundle *C = empty_vars->C;
  matrix *U = empty_vars->U;
  matrix *W = empty_vars->W;
  real **mu = empty_vars->mu;
  matrix *Umhalf = empty_vars->Umhalf;

  /* temporary workspace */
  int max_col_length = 0;
  for (k=0; k < nkpts; k++)
    if (max_col_length < Y[k].col_length)
      max_col_length = Y[k].col_length;

  column_bundle AY(Y[0].tot_ncols,max_col_length);

  for (k=0; k < nkpts; k++)
    {
      // manually adjust col_length;
      AY.col_length = Y[k].col_length;
      copy_innards_column_bundle(&(Y[k]),&AY);

      AY = A_empty(filled_vars->C[k],Y[k]);
      U[k] = AY^(O(AY));
      Umhalf[k] = Uminusonehalf(U[k],W[k],mu[k]);
      C[k] = AY*Umhalf[k];
    }
}

/*
 * Returns the empty band energy function which is just
 *        E = sum_k { trace(C[k]^Hsp*C[k]) }
 */
static real
empty_energy(Elecinfo *empty_info,Elecvars *filled_vars,
	     Elecvars *empty_vars,
	     Ioninfo *ioninfo)
{
  int k;
  real Ener;
  column_bundle *C = empty_vars->C;

  Ener = 0.0;
  for (k=0; k < empty_info->nkpts; k++)
    Ener += REAL( trace(C[k]^Hsp(C[k],ioninfo)) );

  return Ener;
}

/*
 * Returns the empty band energy function which is just
 *        E = sum_k { trace(C[k]^Hsp*C[k]) }
 * as well as calculating its derivate versus Y[k] (placed into grad[])
 * and computing the subspace Hamiltonians and diagonalizing them.
 */
static real
empty_energy_grad_Hsub(Elecinfo *empty_info,Elecvars *filled_vars,
		       Elecvars *empty_vars,
		       Ioninfo *ioninfo,
		       column_bundle *grad)
{
  int k;
  column_bundle *C = empty_vars->C;
  matrix *Umhalf = empty_vars->Umhalf;
  matrix *Hsub = empty_vars->Hsub;
  matrix *Hsub_evecs = empty_vars->Hsub_evecs;
  real **Hsub_eigs = empty_vars->Hsub_eigs;
  
  real Ener;

  /* temporary workspace */
  int max_col_length = 0;
  for (k=0; k < empty_info->nkpts; k++)
    if (max_col_length < C[k].col_length)
      max_col_length = C[k].col_length;

  column_bundle HspC(C[0].tot_ncols,max_col_length);

  Ener = 0.0;
  for (k=0; k < empty_info->nkpts; k++)
    {
      // manually adjust col_length;
      HspC.col_length = C[k].col_length;
      copy_innards_column_bundle(&(C[k]),&HspC);

      HspC = Hsp(C[k],ioninfo);
      Hsub[k] = C[k]^HspC;
      Hsub[k].hermetian = 1;
      diagonalize_herm(Hsub_eigs[k],Hsub_evecs[k],Hsub[k],Hsub[k].nr);

      Ener += REAL(trace(Hsub[k]));
      
      grad[k] = Pbar(filled_vars->C[k],HspC);
      grad[k] -= O(C[k]*(C[k]^HspC));
      HspC = grad[k]*Umhalf[k];
      grad[k] = HspC;
    }

  return Ener;
}

/*
 * Does a linmin along dir for the empty bands:
 * the configuration we start with in empty_vars has energy E0
 * and gradient grad.  Stepsize is the initial stepsize to use; the final
 * stepsize actually used in returned.
 */
static real
do_linmin_empties(column_bundle *dir,
		  Basis *basis,
		  Ioninfo *ioninfo,
		  Elecinfo *empty_info,Elecvars *filled_vars,
		  Elecvars *empty_vars,
		  real E0,
		  column_bundle *grad,
		  real stepsize)
{
  real gamma,dderiv,curvature,E;
  column_bundle *Y = empty_vars->Y;
  int nkpts = empty_info->nkpts;

  /* Directional derivative */
  dderiv = 2.0*REAL(dot(nkpts,grad,dir));
  for (;;)
    {
      /* Shift Y by stepsize */
      scale_accumulate(nkpts,stepsize,dir,Y);

      /* Calculate energy of shifted position */
      calc_UC_empty(empty_info,filled_vars,empty_vars);
      E = empty_energy(empty_info,filled_vars,empty_vars,ioninfo);

      /* Shift back */
      scale_accumulate(nkpts,-stepsize,dir,Y);

      /* Do a parabolic fit to the E0, E, and dderiv
       * to get curvature for quadratic fit and proposed minimum (gamma) */
      curvature = 2.0*(E-E0-stepsize*dderiv)/(stepsize*stepsize);
      gamma = -dderiv/curvature;
      dft_log("dderiv = %8.1le  curvature = %8.1le\n",
		dderiv,curvature);
      dft_log("stepsize = %8.1le    gamma = %8.1le\n",
		stepsize,gamma);
      dft_log_flush();

      /* If curvature is wrong way, take a bigger step */
      if (curvature < 0.0)
	{ stepsize *= 4.0; continue; }
      /* If the proposed minimum is much larger than the stepsize,
       * increase stepsize */
      else if (fabs(gamma/stepsize) > 10.0)
	{ stepsize *= 10.0; continue; }
      /* If much smaller, decrease */
      else if (fabs(gamma/stepsize) <  0.1)
	{ stepsize *= 0.1; continue; }
      /* Otherwise, it was a good linmin so stop! */
      else
	break;
    }

  /* Move to the bottom of the parabola, given by gamma */
  scale_accumulate(nkpts,gamma,dir,Y);

  /* Return the stepsize */
  return stepsize;
}


/*
 * Do niter CG steps on the empty bands
 */
static real
minimize_empties_cg(int niter,
		    real stepsize,
		    Basis *basis,
		    Ioninfo *ioninfo,
		    Elecinfo *empty_info,Elecvars *filled_vars,
		    Elecvars *empty_vars)
{
  int k,iter;
  time_t timenow;
  real linminfact=0.0,cosgg=0.0,alpha,abs2_grad_now,abs2_grad_old,E;
  int nkpts = empty_info->nkpts;
  int nbands = empty_info->nbands;

  /* gradients and search direction */
  column_bundle *grad_now,*grad_old,*dir;

  /* Allocate space for the gradients and search directions */
  grad_now = alloc_column_bundle_array(nkpts,nbands,basis);
  grad_old = alloc_column_bundle_array(nkpts,nbands,basis);
  dir = alloc_column_bundle_array(nkpts,nbands,basis);

  dft_log("\n----- minimize_empties_cg() -----\n");
  dft_log("Starting %d iterations of conjugate gradients with\n",
	    niter);
  dft_log("initial stepsize = %lg.\n\n",stepsize);
  dft_log_flush();
  
  for (iter=0; iter < niter; iter++)
    {
      /* Calculate energy and gradient of current config */
      calc_UC_empty(empty_info,filled_vars,empty_vars);
      E = empty_energy_grad_Hsub(empty_info,filled_vars,
				 empty_vars,ioninfo,grad_now);

      /* Square length of gradients */
      abs2_grad_now = abs2(nkpts,grad_now);
      abs2_grad_old = abs2(nkpts,grad_old);

      /* Some linimin statistics from last iteration */
      if (iter > 0)
	{
	  linminfact = 
	    REAL(dot(nkpts,grad_now,dir))/REAL(dot(nkpts,grad_old,dir));
	  cosgg = 4.0*REAL(dot(nkpts,grad_old,grad_now))/
	    sqrt( 16.0*abs2_grad_now*abs2_grad_old );
	  dft_log("\nlinmin = %8.1le   cosgg = %8.1le\n\n",
		    linminfact,cosgg);
	  dft_log_flush();
	}

      /* Time stamp and print out current energies and subspace Hamiltonian */
      timenow = time(0);

      dft_log("------------------------------------------------------\n");
      dft_log("Iteration %d   %s\n",iter,ctime(&timenow));
      dft_log("E_empty = %23.13le\n",E);
      dft_log("\n");
      dft_log("nkpts=%d:  Hsub eigenvalues are:\n",nkpts);
      for (int band=0; band < empty_info->nbands; band++)
	{
	  for (k=0; k < nkpts; k++)
	    dft_log("%le  ",empty_vars->Hsub_eigs[k][band]);
	  dft_log("\n");
	}
      dft_log("\n");
      dft_log_flush();


      /* If this is the last iteration, don't linmin, but
       * quit the iteration loop and exit the routine */
      if (iter==niter-1)
	break;

      /* Calculate search direction */
      alpha = 0.0;
      /* If we've done a "good" linmin, use CG: */
      /* i.e. calculate alpha */
      if (iter>0 && fabs(linminfact) < 0.05 && fabs(cosgg) < 0.1)
	alpha = abs2_grad_now/abs2_grad_old;
      dft_log("|grad| = %le\n",2.0*sqrt(abs2_grad_now));
      dft_log("alpha = %8.1le\n",alpha);
      dft_log_flush();

      /* Calculate current search direction:
       * d_now = 2.0*grad_now + alpha*d_old */
      for (k=0; k < nkpts; k++)
	{
	  dir[k] *= alpha;
	  dir[k] += grad_now[k];
	  dir[k] += grad_now[k];
	}

      /* Do a linmin along dir */
      stepsize = do_linmin_empties(dir,basis,
				   ioninfo,empty_info,filled_vars,
				   empty_vars,
				   E,grad_now,stepsize);
      /* copy old gradient */
      for (k=0; k < nkpts; k++)
	{
	  grad_old[k] = grad_now[k];
	}
    }

  /* Free up memory used by all the gradients and diretions */
  free_column_bundle_array(nkpts,grad_now);
  free_column_bundle_array(nkpts,grad_old);
  free_column_bundle_array(nkpts,dir);

  /* return stepsize used */
  return stepsize;

}
		    

/*
 * This function either reads C_empty_filename or uses random numbers
 * for initial empty band C variables.  It allocates space and variables
 * for the empty bands, and then does niter_cg CG cycles on the
 * empty bands.  It then frees up the memory used, writes the final empty C
 * values to the file 'C.empty', returns.
 */
void
calc_empties(int niter_cg,real stepsize,
	     int nbands_empty,
	     Basis *basis,
	     Elecinfo *filled_info,Elecvars *filled_vars,
	     Ioninfo *ioninfo,
	     char *init_C_empty_action,char *C_empty_filename)
{
  Elecinfo empty_info;
  Elecvars empty_vars;
  int k,nkpts;

  /* Copy requisite info from filled states */
  empty_info = *filled_info;
  empty_info.nbands = nbands_empty;
  empty_info.nelectrons = 0;
  nkpts = filled_info->nkpts;

  dft_log("\n----- calc_empties() -----\n");
  dft_log_flush();

  /* Allocate space for the empty variables; make their Vscloc point
   * to the right place */
  init_elecvars(&empty_info,basis,&empty_vars);
  for (k=0; k < nkpts; k++)
    {
      empty_vars.Y[k].Vscloc = empty_vars.C[k].Vscloc = 
	&(filled_vars->Vscloc);
    }

  /* Either read the initial empty states from disk... */
  if (strcmp(init_C_empty_action,"read") == 0)
    {
      dft_log("\n-----> Reading C_empty from '%s'\n\n",
		C_empty_filename);
      dft_log_flush();
      read_column_bundle_array(C_empty_filename,nkpts,empty_vars.Y);
    }
  /* or randomize the empty states. */
  else
    {
      dft_log("\n---->  Randomizing empty bands.\n\n");
      dft_log_flush();
      System::seed_with_time();
      randomize_column_bundle_array(nkpts,empty_vars.Y);
    }
  /* Do CG on empty states */
  minimize_empties_cg(niter_cg,stepsize,basis,ioninfo,
		      &empty_info,filled_vars,&empty_vars);

  /* Change C_empty to be the digonal basis of Hsub */
  dft_log("\nSwitching to diagonal basis of Hsub:  C_empty <- HsubC_emtpy\n\n");
  dft_log_flush();

  for (k=0; k < empty_info.nkpts; k++)
    {
      do_column_bundle_matrix_mult(empty_vars.C[k],
				   empty_vars.Hsub_evecs[k],
				   empty_vars.Y[k],0);
      empty_vars.C[k] = empty_vars.Y[k];
    }
  
  /* Write out empty state variables */
  dft_log("\n\nWriting out C.empty.\n\n");
  dft_log_flush();
  write_column_bundle_array("C.empty",nkpts,empty_vars.C);
  
  /* Make Vscloc point back to the one in empty_vars */
  for (k=0; k < nkpts; k++)
    {
      empty_vars.Y[k].Vscloc = empty_vars.C[k].Vscloc =
	&(empty_vars.Vscloc); 
    }

  /* free up used memory */
  free_elecvars(&empty_info,&empty_vars);
}

