/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,           September 3 1997
 *
 * Compute all bands via CG
 *
 */

/* $Id: allbands.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
/* My header file */
#include "header.h"

#include "parallel.h"

#define INPUTFILE    "allbands.in"
#define REPORT_LEVEL 1
#define LOGFILE      mylogfile

#define LN_LEN  150
#define STR_LEN 80


int
main(int argc,char**argv)
{
  /* Initialize MPI through column_bundle class. */
  System::GlobalInit(&argc,&argv);  

  Basis *basis;        /* The basis set for the calculation */
  Ioninfo ioninfo;     /* Ionic information */
  Elecinfo elecinfo;   /* Electronic state information */
  Elecvars elecvars;   /* The electronic variables: Y, C, U, n, ... */
  Control cntrl;       /* Holds convergence control data */

  /* various other local vars */
  int k;
  MPI_FILE *filep;
  char line[LN_LEN],elecfilename[STR_LEN],latticefilename[STR_LEN],ionsfilename[STR_LEN],
       n_filename[STR_LEN],FFT_action[STR_LEN],ewaldfilename[STR_LEN],
       init_C_action[STR_LEN],C_filename[STR_LEN];
  int nx,ny,nz;
  int niter_cg;
  real stepsize;
  time_t timenow;
  int my_report_level;

  /* If we're processor responsible for IO, then we get to print stuff to the screen! */
  if ( System::Get_ID() == System::Get_IO() )
    my_report_level = REPORT_LEVEL;
  else
    my_report_level = 0;

  FILE * mylogfile;
  int dieflag = 0;
  if (my_report_level > 0) {
    if ( (mylogfile = fopen("allbands.log", "a")) == NULL ) {
      fprintf(stderr, "Cannot open allbands.log for write.\n");
      dieflag = 1;
    }
  } else {
    mylogfile = NULL;
  }
  System::GlobalCheck(dieflag, "Ahh!!");

  /* Read input file for information */
  if (argc == 1)
    {
      if ( (filep = MPI_fopen(INPUTFILE,"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],INPUTFILE);
	  die(line);
	}
    }
  else if (argc == 2)
    {
      if ( (filep = MPI_fopen(argv[1],"r")) == (MPI_FILE *)0 )
	{
	  sprintf(line,"\n%s:  can't read '%s'.  Aborting.\n\n",
		  argv[0],argv[1]);
	  die(line);
	}
    }
  else
    {
      sprintf(line,"\nUsage:  %s [inputfile]\n\n",argv[0]);
      die(line);
    }

  timenow = time(0);
  if (my_report_level > 0) {
    fprintf(LOGFILE,"\n");
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"Current date and time: %s\n",ctime(&timenow));
    if (argc==1)
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],INPUTFILE);
    else
      fprintf(LOGFILE,"%s:  reading file '%s'\n",argv[0],argv[1]);
    fflush(LOGFILE);
  }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%d",&niter_cg);
  if (my_report_level > 0)
    fprintf(LOGFILE,"niter_cg = %d\n",niter_cg);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%lg",&stepsize);
  if (my_report_level > 0)
    fprintf(LOGFILE,"stepsize = %lg\n",stepsize);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",elecfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"electronic state file = '%s'\n",elecfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",latticefilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"lattice  file = '%s'\n",latticefilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ionsfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ions file = '%s'\n",ionsfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",ewaldfilename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"ewald file = '%s'\n",ewaldfilename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",FFT_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"FFTflag = %s",FFT_action);

  nx = ny = nz = 0;
  if (strcmp(FFT_action,"automatic") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(FFT_action,"specified") == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&nx,&ny,&nz);
      if (my_report_level > 0)
        fprintf(LOGFILE,":  %d by %d by %d\n",nx,ny,nz);
    }
  else
    {
      sprintf(line,
	      "%s:  initial FFTflag must be 'specified' or 'automatic'.\n\n",
	      argv[0]);
      die(line);
    }

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",n_filename);
  if (my_report_level > 0)
    fprintf(LOGFILE,"n_filename = '%s'\n",n_filename);

  do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
  sscanf(line,"%s",init_C_action);
  if (my_report_level > 0)
    fprintf(LOGFILE,"init_C_action = %s",init_C_action);

  if (strcmp(init_C_action,"random") == 0)
    {
      if (my_report_level > 0)
	fprintf(LOGFILE,"\n");
    }
  else if (strcmp(init_C_action,"read") == 0)
    {
      do { fgets(line,LN_LEN,filep); } while(line[0] == '#');
      sscanf(line,"%s",C_filename);
      if (my_report_level > 0)
        fprintf(LOGFILE," file '%s'\n",C_filename);
    }
  else
    {
      sprintf(line,"%s:  initial C action must be 'random' or 'read'.\n\n",
	      argv[0]);
      die(line);
    }

  if (my_report_level > 0) {
    fprintf(LOGFILE,"******************************************************\n");
    fprintf(LOGFILE,"\n");
  }

  fclose(filep);

  /* Read the electronic state information: k-points, fillings, weights... */
  setup_elecinfo(&elecinfo,elecfilename,&basis,cntrl,my_report_level,LOGFILE);

  /* Read the lattice vectors and set up the basis */
  setup_basis(basis,latticefilename,elecinfo,
	      nx,ny,nz,my_report_level,LOGFILE);

  /* Read the ioninc positions and pseudopotentials */
  setup_ioninfo(&basis[elecinfo.nkpts],&ioninfo,ionsfilename,&elecinfo,my_report_level,LOGFILE);

  /* Setup ewald calculation */
  setup_Ewald(ewaldfilename,my_report_level,LOGFILE);

  /* Setup the electronic variables */
  init_elecvars(&elecinfo,basis,&elecvars);

  /* setup the FFT3D() routines */
  setupFFT3D(basis[elecinfo.nkpts].Nx,
	     basis[elecinfo.nkpts].Ny,
	     basis[elecinfo.nkpts].Nz,
	     my_report_level,LOGFILE);

  /* Calculate the local pseudopotential */
  Vloc_pseudo(&basis[elecinfo.nkpts],&ioninfo,elecvars.Vlocps.c,my_report_level,LOGFILE);
  /* Calculate the non-local pseudopotential for FHI CPI type psp. */
  Vnl_pseudo(basis,&ioninfo,&elecinfo,my_report_level,LOGFILE);

  /* Setup signals */
  setup_signals(&elecinfo,&elecvars,LOGFILE);

  /* Read in electron density */
  if (my_report_level > 0)
    {
      // cannot use MPI_fopen, since we don't want to read in
      // the whole binary file!!
      FILE* testfilep = fopen(n_filename,"r");
      if (testfilep == (FILE *)0)
	{
	  sprintf(line,"\nCan't open '%s' to electron density.\n\n",n_filename);
	  die(line);
	}
      fclose(testfilep);
      fprintf(LOGFILE,"\n-------> Reading electron density from '%s'\n\n",
	      n_filename);
      fflush(LOGFILE);
    }

  elecvars.n.read(n_filename);

  /* Solve the Poisson eqaution to get Hartree field and calculate Vscloc*/
  if (my_report_level > 0)
    fprintf(LOGFILE,"Solving Poisson equation...\n"); fflush(LOGFILE);
  solve_poisson(&elecvars);

  if (my_report_level > 0)
    fprintf(LOGFILE,"Calculating Vscloc...\n"); fflush(LOGFILE);
  calc_Vscloc(&elecvars);

  /* Minimize the bands!!! */
  calc_all_bands(niter_cg,stepsize,
		 basis,&elecinfo,&elecvars,&ioninfo,
		 init_C_action,C_filename,
		 cntrl,
		 my_report_level,LOGFILE);

  /* Free up all the used memory */
  free_basis(basis,elecinfo.nkpts);
  free_ioninfo(&ioninfo);
  free_elecinfo(&elecinfo);
  free_elecvars(&elecinfo,&elecvars);

  /* End the MPI stuff */
  System::GlobalFinalize();

  return 0;
}
