/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *            Sohrab Ismail-Beigi         June 10th, 1999
 *   
 *            written originally by Tairan Wang
 * 
 * Control.c -- the control structure for convergence; some
 *              member functions needed for it to work
 *
 */

/* $Id: Control.c,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#include "header.h"
#include <stdio.h>
#include <math.h>

/* 
 * Member functions for  class  Control:
 *   the convergence control class.
 */
Control::Control() 
{
  E_tolerance = 0.0;
  force_tolerance = 0.0;
  converged = 0;
  Etot = Etot_old = Etot_old_old = 0.0;
  update_stepsize = DFT_TRUE;
  stepsize_min = 1e-5;
  nel_fermi_tol = 1e-8;
  dump_n = DFT_TRUE;

  algorithm       = DFT_PCG_nocos;
  max_elecs_steps = 20;
  max_ionic_steps = 1;
  elec_stepsize   = -1.0;
}

int
Control::if_e_converged(real E) 
{
  if ( ( fabs(E - Etot) < E_tolerance )
       && converged ) {
    Etot_old_old = Etot_old;
    Etot_old = Etot;
    Etot = E;
    return 1;
  } 
  if ( ( fabs(E - Etot) < E_tolerance ) 
       && ( fabs(Etot - Etot_old) < E_tolerance ) )
    converged = 1;
  else 
    converged = 0;
  Etot_old_old = Etot_old;
  Etot_old = Etot;
  Etot = E;
  return 0;
}
  
void
Control::print(Output *out, int iter) {
  // print out the converged status.
  if (converged == 1) 
    out->printf("Energy converged to %e Hartree after %d iters :\n", 
	    E_tolerance, iter);
  else 
    out->printf("Energy not yet converged to %e Hartree after %d iters : \n", 
	    E_tolerance, iter);
  out->printf("\t%e\n\t%e %e\n\t%e %e\n",
	      Etot,
	      Etot_old, fabs(Etot - Etot_old),
	      Etot_old_old, fabs(Etot_old - Etot_old_old));
}

void
Control::reset_converge() {
  // reset the convergence counter;
  // reset  Etot, Etot_old, and Etot_old_old to 0.
  Etot = Etot_old = Etot_old_old = 0.0; 
}
