/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *     Tairan Wang                   May, 16, 1999
 * 
 * Change the support for FHI psp from the core to the input routine,
 * where the FHI psp is converted to MIT format.
 *
 */
/*
 *     Tairan Wang                   Jan, 12, 1999
 *
 * Add support for reading FHI pseudopotential format from the
 * Berlin package  FHI98MD.
 *
 */
/*
 *     Tairan Wang                   Nov, 23, 1997
 *
 * Add a second input format:
 *  merge the ion position inside  ions.in  file,
 *  together with mask information.
 *
 * The format of the <ionsfilename> file is:
 *
 *    Nspecies  [# comment]
 *    # for i = 1 to Nspecies {
 *     # if (external ion position file) {
 *          Z_i Natoms_i <posfile_i> <potfile_i> <pulay_file_i> [optional ionic_mass]  [# comment]  
 *     # } else {   // inline
 *          inline Z_i Natoms_i <potfile_i>  <pulay_file_i> [optional ionic_mass]  [# comment]
 *        # for j = 1 to Natoms_i {
 *            ion_pos[i][j][x] ion_pos[i][j][y] ion_pos[i][j][z] mask[i][j]
 *        # }
 *    # }
 * 
 * see the original documentation below for comparison.
 * mask[i][j] indicate whether the ion should be moved. 
 *    1 means it should be moved, 0 means it should not be moved.
 * at this point, this information is not yet used.
 */

/*
 *     Sohrab Ismail-Beigi           Mar. 1, 1997
 *
 * Setup up from a file (or frees) the information and data describing the
 * ions and their local/non-local pseudopotentials.
 *
 */

/* $Id: setup_ions.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "header.h"


/*
 * Read the file <ionsfilename> to get all the specifications for the
 * ions: number of species, their charges, positions, and
 * local/non-local pseudopotentials for each species.  The format of
 * the ionsfilename file and all files it depends on are defined in
 * the comments below in a recursive top-down manner.
 *
 ***************************************************************************
 * (1) The format of the <ionsfilename> file:
 *
 *         Nspecies   [# comment] 
 *         # for i = 1 to Nspecies {
 *             Z_i Natoms_i <posfile_i> <potfile_i> <pulay_file_i> [# comment]
 *         # }
 *
 * "# for i = 1 to Nspecies " is my shorthand for saying that there must be
 * Nspecies instances of the lines in the braces.
 *
 * -> Z is a real number giving the atomic charge of the species.
 * -> Natoms gives the number of atoms for that species.
 * -> <posfile> is a filename with the atomic positions for that species.
 * -> <potfile> is a filename with the potentials for that species.
 * -> <pulayfile> is a filename with the pulay correction factors for
 *        that species.  If it is set to "none", then no file is read
 *        and the pulay correction is set to zero.
 *
 * Note:
 * -> any line whose first character is '#' is a comment line and ignored
 * -> [# comment] means any text is ignored until the end of the line
 *
 ***************************************************************************
 * (2) The format of the <posfile> file:
 * 
 *        x1 y1 z1
 *        x2 y2 z2
 *          ...
 *          ...
 *
 * -> x y z are real numbers specifying atomic positions in lattice
 *    coordinates.  There are as many as there are atoms for the species.
 * -> Note: NO COMMENTS OF ANY SORT ARE ALLOWED.
 *
 ***************************************************************************
 * (3) The format of the <potfile> file:
 *
 *        ngrid_loc dq_loc <locvqfile>   [# comment]
 *        nlm   [# comment]
 *        # for i = 1 to nlm {
 *            l_i m_i ngamma_i     [# comment]
 *            M_i
 *            # for j = 1 to ngamma_i {
 *                ngrid_ij dq_ij <flqfile_ij> [# comment]
 *            # }
 *        # }
 *
 * -> ngrid_loc is an integer specifying the number of 
 *    grid points specifying the local pseudopot.
 * -> dq_loc is a real number specifying the spacing in q-space of the grid.
 * -> <locvqfile> contains the actual values of the local pseudopotential.
 * -> nlm is an integer specifying the number of (l,m) states for the 
 *    non-local pseudopotentials.
 * -> l and m are integers and are the values of l and m.
 * -> ngamma is an integer specifying the number of multiple-projector
 *    states for the current (l,m).
 * -> M is a ngamma x ngamma matrix of real numbers specifying the 
 *    details of the multiple-projector states at the current (l,m).
 *    M is read a as a string of real numbers separated by spaces, with
 *    the column-index of M changing fastest.  The last number specifying
 *    M MUST BE FOLLOWED BY A CARRIAGE RETURN (i.e must end the line).
 * -> ngrid (int) is an integer specifying the number of grid-points defining
 *    the non-local pseudopotential for the current (l,m,j) state.
 * -> dq (real) is the regular q-space spacing for the potential in the
 *    (l,m,j) states.
 * -> <flqfile> contains the actual values of the pseudopotential for (l,m,j)
 *
 * Note:
 * -> [# comment] means any text from here to the end-of-line is ignored
 * -> Lines starting with '#' are commenst and are ignored.
 *
 * -> IMPORTANT NOTE REGARDING M:
 *    THERE CAN BE NO COMMENTS AT ALL SEPARATING THE VALUES THAT SPECIFY
 *    THE M-MATRIX.  FURTHREMORE, THE VERY LAST VALUE OF M SPECIFIED MUST
 *    BE FOLLOWED BY A CARRIAGE RETURN (I.E. MUST END THE LINE IT IS ON).
 *
 ***************************************************************************
 * (4) The format of the <locvqfile> and <flqfile> files:
 *
 *            0     v(0)
 *            dq    v(dq)
 *            2*dq  v(2*dq)
 *            3*dq  v(3*dq)
 *               .....
 *
 * where dq is the spacing specified for the locvq/flq files, and v(q) is
 * value of the pseudopotential at q.
 * There must be ngrid pairs of numbers (q,v(q)).
 * 
 * NO COMMENTS ALLOWED.
 *
 ***************************************************************************
 *
 * (5) The format of the <pulayfile> is:
 *
 *         [ # comment ]
 *         NEcuts  [ # comment ]
 *         Ecut1      dEperNatoms_dNGperVol1  [ # comment]
 *         Ecut2      dEperNatoms_dNGperVol2  [ # comment]
 *          ...              ...
 *
 * -> NEcuts: the number of pairs of (Ecut,dE...perVol) provided
 * -> Ecut: planwave cutoff energy in Hartrees
 * -> dEperNatoms_dNGperVol: exactly what is says (NG=# of planewaves)!
 *
 ***************************************************************************
 *
 * ions: the Ioninfo structure to fill up with informatio in ionsfilename.
 * logfile: the file to send the reports to
 *
 * May 16, 1999, Tairan, This subroutine exists for backward compatibility
 *  only.
 */
void
setup_ioninfo(Basis *basis, Ioninfo *ioninfo,char *ionsfilename,
	      Elecinfo *einfo)
{
  dft_text_FILE *ionsfile,*posfile;
  char line[DFT_LINE_LEN], line2[DFT_LINE_LEN],
    posfilename[DFT_FILENAME_LEN], potfilename[DFT_FILENAME_LEN],
    pulayfilename[DFT_FILENAME_LEN];
  
  int nspecies,sp,n,nat;

  if ( (ionsfile = dft_text_fopen(ionsfilename,"r")) == (dft_text_FILE *)0)
    die("Can't read ion information file %s\n",ionsfilename);

  /* Read the number of species on a line and allocate space for them */
  do
    { dft_text_fgets(line,DFT_LINE_LEN,ionsfile); }
  while(line[0] == '#');
  sscanf(line,"%d",&(ioninfo->nspecies));
  nspecies = ioninfo->nspecies;
  ioninfo->species = (Speciesinfo *)mymalloc(sizeof(Speciesinfo)*nspecies,
					  "speciesinfo[]","setup_ioninfo");

  dft_log("\n----- setup_ioninfo() -----\n");
  dft_log("Reading ion information from file %s\n\n",ionsfilename);
  dft_log("nspecies = %d\n",ioninfo->nspecies);

  /* Loop over species and read info */
  for (sp=0; sp < nspecies; sp++)
    {
      int match_items;

      /*
       * Read Z, number of atoms, filename containing their coords,
       * and filename containing pseudo-potential information
       */
      do
	{ dft_text_fgets(line,DFT_LINE_LEN,ionsfile); }
      while(line[0] == '#');
      remove_comment(line);  // remove the comment from end of line.

      sscanf(line, "%s", line2);
      if (strcmp(line2,"inline")==0) {  // inline ion positions
	posfilename[0] = '\0';
	match_items = 
	  sscanf(line,"%*s %lg %d %s %s %lg",
		 &(ioninfo->species[sp].Z),
		 &(ioninfo->species[sp].natoms),
		 potfilename,
		 pulayfilename,
		 &(ioninfo->species[sp].mass));
	switch (match_items) {
	case 5: 
	  // convert mass to natural units
	  ioninfo->species[sp].mass /= Std2AtomicMass; 
	  break; // successful scanning
	case 4: // no ion mass provided, set mass to generic value
	  ioninfo->species[sp].mass = 1.0;
	  break;
	default:
	  dft_log(DFT_SILENCE,
		  "Get parameters:\nZ= %g\nN = %d\npotfile = %s\nmass = %g\n",
		  ioninfo->species[sp].Z, ioninfo->species[sp].natoms,
		  potfilename, ioninfo->species[sp].mass);
	  die("Failed to scan ionsfile(%s). Get %d out of 5 items\n",
	      ionsfilename, match_items);
	  break;
	}
      } else { // external ion position file
	match_items = 
	  sscanf(line,"%lg %d %s %s %s %lg",
		 &(ioninfo->species[sp].Z),
		 &(ioninfo->species[sp].natoms),
		 posfilename,
		 potfilename,
		 pulayfilename,
		 &(ioninfo->species[sp].mass));
	switch (match_items) {
	case 6: 
	  // convert mass to program unit
	  ioninfo->species[sp].mass /= Std2AtomicMass; 
	  break; // successful scanning
	case 5: // no ion mass provided, set mass to generic value
	  ioninfo->species[sp].mass = 1.0;
	  break;
	default:
	  dft_log(DFT_SILENCE,
		  "Get parameters:\nZ= %g\nN = %d\npotfile = %s\nposfile = %s\nmass = %g\n",
		  ioninfo->species[sp].Z, ioninfo->species[sp].natoms,
		  potfilename, posfilename, ioninfo->species[sp].mass);
	  die("Fail to scan ionsfile(%s). Get %d out of 4 items\n",
		  ionsfilename, match_items);
	  break;
	}
      }	

      /*
       * Read Pseudo potential information from file
       *
       * 1. The default file format is as specified by Sohrab.
       *
       * 2. If the file name is "fhi_cpi_file", then we are using 
       *    the pseudopotential file of the berlin format from 
       *    package  FHI98MD.  The next line in the ions file 
       *    should contain the information:
       *        fhi_filename  l_max  l_loc
       *    l_max  indicates which components are present:
       *    l_max = s  :  only s is used
       *            p  :  s and p are used
       *            d  :  s, p and d are used
       *    (these are the only options right now)
       *    l_loc  decides which component is to be used as 
       *           the local component
       *    l_loc = s  :  s is local
       *            p  :  p is local
       *            d  :  d is local
       */
      read_dft_psp(&(ioninfo->species[sp]), potfilename);

      nat = ioninfo->species[sp].natoms;
      dft_log("\nspecies = %d:\n",sp);
      if (posfilename[0] != '\0') {
	dft_log("Z = %g  natoms = %d   mass = %g  ",
		  ioninfo->species[sp].Z,ioninfo->species[sp].natoms,
		  ioninfo->species[sp].mass);
	dft_log("Reading positions from file %s\n", posfilename);
      } else {
	dft_log("Z = %g  natoms = %d   mass = %g  ",
		  ioninfo->species[sp].Z,ioninfo->species[sp].natoms,
		  ioninfo->species[sp].mass);
	dft_log("Continue reading positions from same file\n");
      }
      
      /* Allocate space for atomic positions and forces */
      ioninfo->species[sp].atpos = (vector3 *)mymalloc(sizeof(vector3)*nat,
						    "atpos[]",
						    "setup_ioninfo");
      ioninfo->species[sp].forces = (vector3 *)mymalloc(sizeof(vector3)*nat,
							"forces[]",
							"setup_ioninfo");
      ioninfo->species[sp].move = (int *)mymalloc(sizeof(int)*nat,
						  "move[]",
						  "setup_ioninfo");

      /* Read their coordinates from specified file and close it */
      if (posfilename[0] != '\0') {
	if ( (posfile = dft_text_fopen(posfilename,"r")) == (dft_text_FILE *)0)
	  die("Can't open positions file for reading.\n");
	for (n=0; n < nat; n++) {
	  dft_text_fgets(line,DFT_LINE_LEN,posfile);
	  if (sscanf(line,"%lg %lg %lg %d",
		     &(ioninfo->species[sp].atpos[n].v[0]),
		     &(ioninfo->species[sp].atpos[n].v[1]),
		     &(ioninfo->species[sp].atpos[n].v[2]),
		     &(ioninfo->species[sp].move[n])) < 4)
	    ioninfo->species[sp].move[n] = 0; // default is not moving the ion.
	}
	dft_text_fclose(posfile);
      } else {

	for (n=0; n < nat; n++) {
	  dft_text_fgets(line,DFT_LINE_LEN,ionsfile);
	  if (sscanf(line,"%lg %lg %lg %d",
		     &(ioninfo->species[sp].atpos[n].v[0]),
		     &(ioninfo->species[sp].atpos[n].v[1]),
		     &(ioninfo->species[sp].atpos[n].v[2]),
		     &(ioninfo->species[sp].move[n])) < 4)
	    ioninfo->species[sp].move[n] = 0; // default is not moving the ion.
	}
      }

      /* If requested, print out atomic positions */
      dft_log("Atomic positions:\n");
      for (n=0; n < nat; n++)
	{
	  dft_log("atom = %d ",n);
	  if (ioninfo->species[sp].move[n]==0)
	    dft_log("(move:no ) ");
	  else
	    dft_log("(move:yes) ");
	  ioninfo->species[sp].atpos[n].print(dft_global_log,"%8lg ");
	}
      
      
      /* If Pulay correction filename is "none", then set the
       * Pulay correction to zero and skip reading the file */
      ioninfo->species[sp].setup_pulay(pulayfilename,einfo->Ecut);

    } /* of sp: loop on species */


  /*
   * Check the atomic positions to make sure no two atoms are at the 
   * the same place
   */
  ioninfo->check_pos();


  dft_log_flush();

  /* We're done! (finally) */
  dft_text_fclose(ionsfile);
}


/* Free up all the memory used by the ion information structure */
void
free_ioninfo(Ioninfo *ioninfo)
{
  int sp,lm,gamma;

  /* Loop over species */
  for (sp = 0; sp < ioninfo->nspecies; sp++)
    {
      /* Atomic positions and forces */
      myfree(ioninfo->species[sp].atpos);
      myfree(ioninfo->species[sp].forces);

      /* Local pseudopotential */
      myfree(ioninfo->species[sp].V_loc);
      /* Nonlocal pseudopot:  loom over (l,m) states */
      for (lm = 0; lm < ioninfo->species[sp].nlm; lm++)
	{
	  myfree(ioninfo->species[sp].ngrid_nl[lm]);
	  myfree(ioninfo->species[sp].dq_nl[lm]);
	  for (gamma = 0; gamma < ioninfo->species[sp].ngamma[lm]; gamma++)
	    {
	      myfree(ioninfo->species[sp].flq[lm][gamma]);
	    }
	  myfree(ioninfo->species[sp].flq[lm]);
	  ioninfo->species[sp].M[lm].freemem();
	}
      if (ioninfo->species[sp].nlm > 0) {
	myfree(ioninfo->species[sp].l);
	myfree(ioninfo->species[sp].m);
	myfree(ioninfo->species[sp].ngamma);
	myfree(ioninfo->species[sp].M);
	myfree(ioninfo->species[sp].ngrid_nl);
	myfree(ioninfo->species[sp].dq_nl);
	myfree(ioninfo->species[sp].flq);
      }
    }
  myfree(ioninfo->species);
  if (ioninfo->done != NULL)
    myfree(ioninfo->done);
}


/*
 * Read Pseudo potential information from file
 *
 * 1. The default file format is the MIT format as specified by Sohrab.
 */
void read_dft_psp(Speciesinfo *sp_info, char *potfilename)
{
  dft_text_FILE *potfile,*vqfile,*flqfile;
  char line[DFT_LINE_LEN],
       vqfilename[DFT_FILENAME_LEN],
       flqfilename[DFT_FILENAME_LEN];

  int ngrid, n, nlm, lm, ngamma, gamma, gammap;
  real r;
  
  /* Open the potential file for current species */
  if ( (potfile = dft_text_fopen(potfilename,"r")) == (dft_text_FILE *)0)
    die("Can't open potential file for reading.\n");

  /*
   * Read gridding information for local pseudopotential and file name
   * containing atomic potential in q-space
   */
  do { dft_text_fgets(line,DFT_LINE_LEN,potfile); } while(line[0] == '#');
  sscanf(line,"%d %lg %s",
	 &(sp_info->ngrid_loc),&(sp_info->dq_loc),vqfilename);
  ngrid = sp_info->ngrid_loc;
  dft_log("Reading potential file %s\n",potfilename);
  dft_log("Local pseudopotential: ngrid = %d   dq = %g  file = %s\n",
	    ngrid,sp_info->dq_loc,vqfilename);

  /* Allocate space to store local pseudopotential on grid */
  sp_info->V_loc = (real *)mymalloc(sizeof(real)*ngrid,
				    "V_loc","setup_ioninfo");
      
  /* Read the file containg local pseudopotential in q-space: the
   * format is ngrid triplets of real values (q,V(q),V'(q)) one after the
   * other in ASCII text format. */
  if ( (vqfile = dft_text_fopen(vqfilename,"r")) == (dft_text_FILE *)0)
    die("Can't open vqfile for reading.\n");
  for (n=0; n < ngrid; n++)
    dft_text_fscanf(vqfile,"%lg %lg",&r,&(sp_info->V_loc[n]));
  dft_text_fclose(vqfile);

  /*
   * Read non-local pseudopotential information:
   *
   * first read nlm on a line, the number of (l,m) states described
   */
  do { dft_text_fgets(line,DFT_LINE_LEN,potfile); } while(line[0] == '#');
  sscanf(line,"%d",&(sp_info->nlm));
  nlm = sp_info->nlm;
  dft_log("Non-local pseudopotential(s): nlm = %d\n", sp_info->nlm);


  /*
   * There is no non-linear core correction implemented.
   */
  sp_info->nl_core_flag = DFT_FALSE;  


  /* If there are no non-local pseudopots to read, we're done! */
  if (nlm==0)
    return;
      

  /* Allocate space for l,m,ngamma,M,ngrid_nl,dq_nl,
   * and flq/flqprime at each (l,m) */
  sp_info->l = (int *)mymalloc(sizeof(int)*nlm,"l","setup_ioninfo");
  sp_info->m = (int *)mymalloc(sizeof(int)*nlm,"m","setup_ioninfo");
  sp_info->ngamma = (int *)mymalloc(sizeof(int)*nlm,"ngamma","setup_ioninfo");
  sp_info->M = (matrix *)mymalloc(sizeof(matrix)*nlm,"M","setup_ioninfo");
  sp_info->ngrid_nl = (int **)mymalloc(sizeof(int *)*nlm,
				       "ngrid_nl","setup_ioninfo");
  sp_info->dq_nl = (real **)mymalloc(sizeof(real *)*nlm,
				     "dq_nl","setup_ioninfo");
  sp_info->flq = (real ***)mymalloc(sizeof(real **)*nlm,
				    "flq","setup_ioninfo");

  /* Loop over the (l,m) states */
  for (lm = 0; lm < nlm; lm++)
    {
      /* Read l, m , and ngamma for this (l,m) state */
      do { dft_text_fgets(line,DFT_LINE_LEN,potfile); } while(line[0] == '#');
      sscanf(line,"%d %d %d",&(sp_info->l[lm]),
	     &(sp_info->m[lm]),
	     &(sp_info->ngamma[lm]));
      ngamma = sp_info->ngamma[lm];
      dft_log("lm = %d:\n  l = %d   m = %d   ngamma = %d\n",
		lm,sp_info->l[lm],sp_info->m[lm],ngamma);

      /* Allocate space for ngrid_nl[lm], dq[lm], and flq/flqprime[lm]
       * at each gamma */ 
      sp_info->ngrid_nl[lm] = (int *)mymalloc(sizeof(int)*ngamma,
					      "ngrid_nl[lm]","setup_ioninfo");
      sp_info->dq_nl[lm] = (real *)mymalloc(sizeof(real)*ngamma,
					    "dq_nl[lm]","setup_ioninfo");
      sp_info->flq[lm] = (real **)mymalloc(sizeof(real *)*ngamma,
					   "flq[lm]","setup_ioninfo");

      /* Allocate and read the ngamma x ngamma M[lm] matrix */
      sp_info->M[lm].init(ngamma,ngamma);
      for (gamma = 0; gamma < ngamma; gamma++)
	for (gammap = 0; gammap < ngamma; gammap++)
	  {
	    dft_text_fscanf(potfile,"%lg",&r);
	    sp_info->M[lm](gamma,gammap) = (scalar)r;
	  }
      dft_log("M = \n");
      for (gamma = 0; gamma < ngamma; gamma++)
	{
	  dft_log("[ ");
	  for (gammap = 0; gammap < ngamma; gammap++)
	    dft_log("%g ",REAL(sp_info->M[lm](gamma,gammap)));
	  dft_log(" ]\n");
	}

      /* Skip to next line after last M-entry */
      dft_text_fgets(line,DFT_LINE_LEN,potfile);
	  
      /* For each gamma, read the gridding info and flq filename */
      for (gamma = 0; gamma < ngamma; gamma++)
	{
	  /* Read gridding information and flq file name for 
	   * the current [lm][gamma] pseudopotential */
	  do
	    { dft_text_fgets(line,DFT_LINE_LEN,potfile); }
	  while(line[0] == '#');
	  sscanf(line,"%d %lg %s",
		 &(sp_info->ngrid_nl[lm][gamma]),
		 &(sp_info->dq_nl[lm][gamma]),
		 flqfilename);
	  ngrid = sp_info->ngrid_nl[lm][gamma];
	  dft_log("  gamma = %d:  ngrid = %d  dq = %g  file = %s\n",
		    gamma,ngrid,
		    sp_info->dq_nl[lm][gamma],flqfilename);
	  
	  /* Allocate space to store the gridded flq[lm][gamma] */
	  sp_info->flq[lm][gamma] = (real *)mymalloc(sizeof(real)*ngrid,
						     "flq[lm][gamma]","setup_ioninfo");

	  /* Open flq file for flq[lm][gamma] and read it:
	   * the format is ngrid doublets (q,flq(q)) */
	  if ( (flqfile = dft_text_fopen(flqfilename,"r")) ==
	              (dft_text_FILE *)0)
	    die("Can't open flq file for reading.\n");
	  for (n=0; n < ngrid; n++)
	    dft_text_fscanf(flqfile,"%lg %lg",&r,
		   &(sp_info->flq[lm][gamma][n]));
	  dft_text_fclose(flqfile);
	}
    } /* of lm: loop on (l,m) states */

}


/*
 * Read Pseudo potential information from file
 *
 * 2. The berlin format from package  FHI98MD
 *    l_max indicates which angular momentum components are used
 *          l_max = 0,  s
 *                  1,  s, p
 *                  2,  s, p, d
 *    l_loc indicates which one is being used as the 
 *          local component.
 *
 *    The file format is:
 *     >>>>>>>( Start )>>>>>>>>>>
 *        Zv  L_max
 *          (10 lines of number for fitted psp
 *           not used here.)
 *        ngrid  clog       #  (for each angular momentum)
 *          i  r(i)  Psi(i)  Vion(i)
 *          ...
 *          ...
 *        (ngrid of core electron density for 
 *         nonlinear core correction
 *           --not tested yet)
 *     <<<<<<<(  End  )<<<<<<<<<<
 */
void read_fhi_psp(Speciesinfo *sp_info, char *potfilename,
		  char lmax, char lloc,
		  int ngrid_loc, real dq_loc, int ngrid_nl, real dq_nl)
{
  dft_text_FILE *potfile;
  char line[DFT_LINE_LEN];

  int ngrid, nl, l_max = 2, l_loc = 2;
  real zv, clog; 
  int n, l, m, i, j, nlm, lm, gamma, ngamma, gammap; 

  // Temporary variables.
  // maximum angular momenta is 3.
  real *r[3], *rPsi[3], *Vion[3], fhi_clog[3];
  int fhi_ngrid[3];
  real *r1, *rPsi1, *V1, *V0, *fun, q, q1, arg;
  int ngrid_r = 0;

  const int N_Lines_to_Skip = 10;
  const real fourpi = 4 * M_PI;

  // only support for  -1 < l_loc <= l_max < 3
  switch (lmax) {
  case 's' : l_max = 0; break;
  case 'p' : l_max = 1; break;
  case 'd' : l_max = 2; break;
  default:
    die("Angular momenta other than s, p, and d (0,1,2) are not supported: %d",
	l_max);
  }
  switch (lloc) {
  case 's' : l_loc = 0; break;
  case 'p' : l_loc = 1; break;
  case 'd' : l_loc = 2; break;
  default:
    die("Angular momenta other than s, p, and d (0,1,2) are not supported: %d",
	l_loc);
  }
  if (l_loc > l_max)
    die("Illegal local component: l_max = %d, l_loc = %d",l_max,l_loc);


  /* Open the potential file for current species */
  if ( (potfile = dft_text_fopen(potfilename,"r")) == (dft_text_FILE *)0)
    die("Can't open potential file for reading.\n");

  sp_info->ngrid_loc = ngrid_loc;
  sp_info->dq_loc    = dq_loc;
  ngrid = sp_info->ngrid_loc;
  dft_log("Reading potential file %s (FHI CPI format)\n", potfilename);
  dft_log("Local pseudopotential: ngrid = %d   dq = %g  file = %s",
	    ngrid,sp_info->dq_loc,potfilename);

  switch (l_loc) {
  case 0 : dft_log(" (s) \n"); break;
  case 1 : dft_log(" (p) \n"); break;
  case 2 : dft_log(" (d) \n"); break;
  default: die("Illegal l_loc value!"); break;
  }
  
  /* Allocate space to store local pseudopotential on grid */
  sp_info->V_loc = (real *)mymalloc(sizeof(real)*ngrid,
				    "V_loc","setup_ioninfo");


  /*
   * The first line has two numbers:
   *   Zv   number_of_angular_momentum_components
   */
  dft_text_fgets(line,DFT_LINE_LEN,potfile);
  sscanf(line,"%lg %d", &zv, &nl);
  // verify number of valence charges, and number of angular momentum
  if (zv != sp_info->Z) {
    dft_log("Valence charge unmatched in PSP file %s : %5.1f != %5.1f !!",
	    potfilename,zv,sp_info->Z);
    die("Fatal error in read_fhi_psp()!");
  }
  if (nl != (l_max+1)) {
    /*
     * In the FHI pseudopotentials, nl actually takes values: 1, 2, 3
     * (perhaps due to the fact that FORTRAN index starts from 1 instead of 0)
     * corresponding to our  l_max = 0, 1, 2. 
     */
    dft_log("Number of angular momentum mismatch: %d != %d !!", nl, l_max+1 );
    die("Fatal error in read_fhi_psp()!");
  }


  // there are 10 lines of data for pseudopotentials that 
  // are fitted to expressions. Skip.
  for (n = 0; n < N_Lines_to_Skip; n++) 
    dft_text_fgets(line,DFT_LINE_LEN,potfile);

  // only allow up to 3 angular momentum components.
  // for (i = 0; i < 3; i++)
  //  sp_info->cpi_ngrid[i] = 0;

  /*
   * Read in all wavefunctions and potentials.
   * All are in real space with logrithmic grid spacing.
   */
  for (l = 0; l <= l_max; l++) {
    dft_text_fgets(line,DFT_LINE_LEN,potfile);
    sscanf(line, "%d %lg", &fhi_ngrid[l], &clog);
    // clog[l] == R[l][1]/R[l][0]  is the scale of the log grid spacing.
    fhi_clog[l] = log(clog);
    // now clog[l] = D log(R), used for integration later.
    dft_log(" l = %d:  ngrid = %d, clog = %f\n",
	    l, fhi_ngrid[l], fhi_clog[l]);

    r[l]    = (real*)mymalloc(sizeof(real)*fhi_ngrid[l],"r1","read_fhi_psp");
    rPsi[l] = (real*)mymalloc(sizeof(real)*fhi_ngrid[l],"rPsi","read_fhi_psp");
    Vion[l] = (real*)mymalloc(sizeof(real)*fhi_ngrid[l],"Vion","read_fhi_psp");
    if (fhi_ngrid[l] > ngrid_r) 
      ngrid_r = fhi_ngrid[l];

    for (i = 0; i < fhi_ngrid[l]; i++) {
      dft_text_fgets(line,DFT_LINE_LEN,potfile);
      sscanf(line,"%*d %lg %lg %lg",
	     &(r[l][i]), 
	     &(rPsi[l][i]), 
	     &(Vion[l][i]));
    }
  }

  /*
   * Read core electron density, for nonlinear core correction.
   * (not tested yet)
   */
  sp_info->nl_core_flag = DFT_TRUE;
  // assuming it has the same number of r-points as 1st potential component
  sp_info->core_r = (real*)mymalloc(sizeof(real)*fhi_ngrid[0],"core_r","read_fhi_psp");
  sp_info->n_core = (real*)mymalloc(sizeof(real)*fhi_ngrid[0],"n_core","read_fhi_psp");
  for (j = 0; j < fhi_ngrid[0]; j++) {
    if (dft_text_fgets(line,DFT_LINE_LEN,potfile) == NULL) {
      sp_info->nl_core_flag = DFT_FALSE;
    }
    sscanf(line,"%lg %lg", &(sp_info->core_r[j]), &(sp_info->n_core[j]));
    sp_info->n_core[j] *= (0.25 / M_PI);
  }
  if ( ! sp_info->nl_core_flag  ) { // no complete core electron density is found.
    myfree(sp_info->core_r);
    myfree(sp_info->n_core);
  } else {
    dft_log("Species %s is core corrected.\n",sp_info->name);
    sp_info->n_core_grid = fhi_ngrid[0];
    sp_info->clog_core_grid = fhi_clog[0];
  }


  /*
   * Convert the local potential from log-grid real space to equal-grid k-space
   *
   */
  r1    = r[l_loc]; 
  rPsi1 = rPsi[l_loc];
  V1    = Vion[l_loc];
  fun   = (real*)mymalloc(sizeof(real)* ngrid_r,"fun","read_fhi_psp");
  for (i = 0; i < fhi_ngrid[l_loc]; i++)
    fun[i] = ( V1[i]*r1[i]*r1[i] + sp_info->Z*r1[i] ) * r1[i];
  sp_info->V_loc[0] = simps(fhi_ngrid[l_loc],fun,fhi_clog[l_loc]) * fourpi;
  for (n = 1; n < ngrid; n++) {
    q = n * dq_loc;
    for (i = 0; i < fhi_ngrid[l_loc]; i++) {
      arg = r1[i] * q;
      fun[i] = ( V1[i]*r1[i]*r1[i] + sp_info->Z*r1[i])*r1[i]*sin(arg)/arg;
    }
    // do the logarithmic-grid integral.
    sp_info->V_loc[n] = simps(fhi_ngrid[l_loc],fun,fhi_clog[l_loc]) * fourpi;
  }


  /*
   * Initialize nlm, l, m, ngamma, and M
   */
  sp_info->nlm = 0;
  for (l=0; l<=l_max; l++) {
    if (l != l_loc)
      sp_info->nlm += (2*l + 1);
  }
  nlm = sp_info->nlm;
  dft_log("\nNon-local pseudopotential(s): nlm = %d\n",
	  sp_info->nlm);
  
  /* If there are no non-local pseudopots to read, we're done! */
  if (nlm==0)
    return;

  sp_info->l = (int *)mymalloc(sizeof(int)*nlm,"l","setup_ioninfo");
  sp_info->m = (int *)mymalloc(sizeof(int)*nlm,"m","setup_ioninfo");
  sp_info->ngamma = (int *)mymalloc(sizeof(int)*nlm,"ngamma","setup_ioninfo");
  sp_info->M = (matrix *)mymalloc(sizeof(matrix)*nlm,"M","setup_ioninfo");
  sp_info->ngrid_nl = (int **)mymalloc(sizeof(int *)*nlm,
				       "ngrid_nl","setup_ioninfo");
  sp_info->dq_nl = (real **)mymalloc(sizeof(real *)*nlm,
				     "dq_nl","setup_ioninfo");
  sp_info->flq = (real ***)mymalloc(sizeof(real **)*nlm,
				    "flq","setup_ioninfo");
  lm = 0;  // set lm counter to 0.
  for (l=0; l<= l_max; l++) { // go through all l values.
    if (l != l_loc) { // for nonlocal cases
      // Calculate Mnl = ( <phi_l| dV_l |phi_l> )^(-1)
      ngrid_r = fhi_ngrid[l];
      r1    = &(r[l][0]);
      V1    = &(Vion[l][0]);
      V0    = &(Vion[l_loc][0]);
      rPsi1 = &(rPsi[l][0]);

      for (i = 0; i < ngrid_r; i++)
	fun[i] = (V1[i] - V0[i])*r1[i]*(rPsi1[i]*rPsi1[i]);
      q = simps(ngrid_r,fun,fhi_clog[l]);

      // for all m values of the current l value
      for (m= l; m > (-l-1); m--) {
	sp_info->l[lm] = l;
	sp_info->m[lm] = m;
	ngamma = sp_info->ngamma[lm] = 1;  // single projector for this type of pseudopotential
	sp_info->M[lm].init(ngamma,ngamma);
        for (gamma = 0; gamma < ngamma; gamma++)
          for (gammap = 0; gammap < ngamma; gammap++)
              sp_info->M[lm](gamma,gammap) = 1.0/q;

	dft_log("lm = %d:\n  l = %d   m = %d   ngamma = %d\n",
		lm,l,m,ngamma);
	dft_log("M = \n");
	for (gamma = 0; gamma < ngamma; gamma++) {
	  dft_log("[ ");
	  for (gammap = 0; gammap < ngamma; gammap++)
	    dft_log("%g ", REAL(sp_info->M[lm](gamma,gammap)));
	  dft_log(" ]\n");
	}
	
	/* Allocate space for ngrid_nl[lm], dq[lm], and flq/flqprime[lm]
	 * at each gamma */ 
	sp_info->ngrid_nl[lm] = (int *)mymalloc(sizeof(int)*ngamma,
						"ngrid_nl[lm]","setup_ioninfo");
	sp_info->dq_nl[lm] = (real *)mymalloc(sizeof(real)*ngamma,
					      "dq_nl[lm]","setup_ioninfo");
	sp_info->flq[lm] = (real **)mymalloc(sizeof(real *)*ngamma,
					     "flq[lm]","setup_ioninfo");
	// ngamma must be 1 in this case.
	sp_info->ngrid_nl[lm][0] = ngrid_nl;
	sp_info->dq_nl[lm][0]    = dq_nl;
	sp_info->flq[lm][0] = (real *)mymalloc(sizeof(real)*ngrid_nl,
					       "flq[lm][gamma]","setup_ioninfo");

	lm++; // increment counter.
      }

      for (n = 0; n < ngrid_nl; n++) {
	q = n * dq_nl;
	// construct flq from FHI pseudopotential data.
	// integrate from pseudoatom wavefunctions.
	// int_d(log(r)) r^2 * dV_l(r) * u(r) * j_l(qr)
	for (i = 0; i < ngrid_r; i++) {
	  fun[i] = (V1[i] - V0[i])*(r1[i]*r1[i])*rPsi1[i];
	  arg = r1[i] * q;
	  
	  /* Tairan Wang  1/13/1999
	   * The special cases for evaluating  j_l(x) near x = 0
	   * need to be taken care of. 
	   * What cut off value to use and how exactly to do this 
	   * need further thoughts.
	   */
	  q1 = 0;
	  switch (l) {
	  case 0 : // j_0(qr) = sin(qr)/qr
	    if (arg < 1e-10) // if (qr < 1e-10), j_0(qr) == 1
	      q1 = 1;
	    else
	      q1 = (sin(arg)/arg);
	    break;
	  case 1 : // j_1(qr) = sin(qr)/(qr)^2 - cos(qr)/qr
	    if (arg < 1e-10) // j_1(x) = x/3 - x^3/30 + ...
	      q1 = arg/3;
	    else
	      q1 = (sin(arg)/arg - cos(arg))/arg;
	    break;
	  case 2 : // j_2(qr) = (3/(qr)^3-1/qr)sin(qr) - 3cos(qr)/(qr)^2
	    if (arg < 1e-10) // j_2(x) = x^2/15 - x^4/210 + ...
	      q1 = arg*arg /15;
	    else {
	      q1 = (3/(arg*arg)-1)*sin(arg)-3*cos(arg)/arg;
	      q1 = q1/arg;
	    }
	    break;
	  default :
	    die("Illegal l value!");
	    break;
	  }
	  fun[i] *= q1;
	}

	q1 = simps(ngrid_r,fun,fhi_clog[l]);
	for (m = 0; m < (2*l+1); m++)
	  sp_info->flq[lm-m-1][0][n] = q1;
      }
    }
  }

  // now free up temporary memory.
  myfree(fun);
  for (l = 0; l <= l_max; l++) {
    myfree(r[l]);
    myfree(rPsi[l]);
    myfree(Vion[l]);
  }
}
