/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * setup_basis.c
 *
 */

/* $Id: setup_basis.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include "stdio.h"
#include "math.h"

#include "header.h"



/*
 * Sets up the basis structure: it takes the lattice vectors from the
 * file latticefilename (whose structure is described in the comments
 * below), computes the unit cell volume, the reciprocal primitive
 * lattice vectors (G) and their dot product (GGT), finds all the
 * G-vectors with energy |G|^2 <= Ecut, and determines the FFT box
 * size.
 * 
 * if Nx_spec == Ny_spec == Nz_spec == 0, the program figures
 * out the FFT boxsizes to be used by itself.
 * Otherwise, it uses the sizes specified to it by Ni_spec.
 *
 * report == 0:  silent
 * report == 1:  print latvec, volume, G, GGT, and nbasis
 * report > 1: above, plus a list of all the G-vectors inside cutoff
 *
 ************************************************************************
 * Format of latticefile: a line starting with '#' is a comment line
 *                        and is ignored.
 *
 *         a1x  a2x  a3x
 *         a1y  a2y  a3y
 *         a1z  a2z  a3z
 *
 * Thus the lattice vectors a1, a2, and a3 are in the COLUMNS.  The
 * units are bohr radii.
 * 
 * May 16, 1999, Tairan, This subroutine exists for backward compatibility
 *  only.
 *
 */
void
setup_basis(Basis *basis,char *latticefilename,const Elecinfo &elecinfo,
	    int Nx_spec,int Ny_spec,int Nz_spec)
{
  int i,j,k, nkpts = elecinfo.nkpts;
  real Ecut = elecinfo.Ecut;

  matrix3 invGGT,box;
  vector3 e,f;
  real G2,G2max;
  int ibox[3],fftbox[3],n,index;
  dft_text_FILE *latticefile;
  char line[DFT_LINE_LEN];

  // Temporary holding places for common basis stuffs.
  matrix3 latvec, G, GGT;
  real unit_cell_volume;
  int Nx,Ny,Nz,NxNyNz;
  int Gxmin,Gxmax,Gymin,Gymax,Gzmin,Gzmax;

  /* Read lattice file */
  dft_log("\n----- setup_basis() -----\n");
  dft_log("\nReading file lattice vector file %s\n\n", latticefilename);

  if ( (latticefile = dft_text_fopen(latticefilename,"r")) ==
              (dft_text_FILE *)0)
    die("Can't open lattice file for reading!\n");
  /* Read a row at at time */
  for (i=0; i < 3; i++)
    {
      do
	{ dft_text_fgets(line,DFT_LINE_LEN,latticefile); }
      while (line[0] == '#');
      sscanf(line,"%lg %lg %lg",
	     &(latvec.m[i][0]),
	     &(latvec.m[i][1]),
	     &(latvec.m[i][2]));
    }
  dft_text_fclose(latticefile);

  /* Calculate unit cell volume */
  unit_cell_volume = fabs(det3(latvec));

  /* Calculate recip. lattice vectors and dot products */
  G = (2.0*M_PI)*inv3(latvec);
  GGT = G*(~G);
  dft_log("latvec =\n");
  latvec.print(dft_global_log,"%10lg ");
  dft_log("unit cell volume = %lg\n\n",unit_cell_volume);
  dft_log("G =\n");
  G.print(dft_global_log,"%10lg ");
  dft_log("GGT =\n");
  GGT.print(dft_global_log,"%10lg ");

  /* We want to know for what vectors v lying on the constant energy surface
   * Ecut = 0.5*(v,GGT*v) the x, y, or z component/projection of v is maximal.
   * This is easy:  the gradient at that point v must lie in the x, y, or z
   * direction!  I.e. if e is the unit direction we are interested in,
   * GGT*v = mu*e or v = mu*inv(GGT)*e, where mu is chosen to be
   * mu = sqrt(2*Ecut/(e,inv(GGT)*e) to ensure v is on the const. energy
   * surface. We solve this for e being x,y,z unit vectors and put the
   * resulting vectors into the rows of box. */
  invGGT = inv3(GGT);
  for (i=0; i < 3; i++)
    {
      for (j=0; j < 3; j++)
	e.v[j] = 0.0;
      e.v[i] = 1.0;
      f = invGGT*e;
      f = sqrt(2.0*Ecut/(e*f))*f;
      for (j=0; j < 3; j++)
	box.m[j][i] = f.v[j];
    }


  dft_log("\nEnergy cutoff Ecut = %lg Hartrees\n",Ecut);
  dft_log("On the surface Ecut = 0.5*|G|^2, the vector extremizing\n");
  dft_log("(G,e_x) = "); box[0].print(dft_global_log,"%lg ");
  dft_log("(G,e_y) = "); box[1].print(dft_global_log,"%lg ");
  dft_log("(G,e_z) = "); box[2].print(dft_global_log,"%lg ");

  /* Truncate the values on the diagonal of box to integers.  This will
   * be the size of a box along x/y/z which will contain all G-vectors
   * of energy < Ecut. */
  for (i=0; i < 3; i++)
    ibox[i] = (int)fabs(box.m[i][i]);
  Gxmax = ibox[0]; Gxmin = -ibox[0];
  Gymax = ibox[1]; Gymin = -ibox[1];
  Gzmax = ibox[2]; Gzmin = -ibox[2];

  dft_log("\nSize of box containing G-vectors = ");
  dft_log("[-%d,%d] by [-%d,%d] by [-%d,%d]\n",
	    ibox[0],ibox[0],ibox[1],ibox[1],ibox[2],ibox[2]);
  dft_log("Size of box containing density = ");
  dft_log("[-%d,%d] by [-%d,%d] by [-%d,%d]\n\n",
	    2*ibox[0],2*ibox[0],2*ibox[1],2*ibox[1],2*ibox[2],2*ibox[2]);

  /* Find the minimal FFT box size the factors into the primes (2,3,5,7).
   * The minimum value for the size of the fftbox is 2*2*ibox+1 because
   * we square the wave-functions and so the FFT box G-vectors must
   * at least range over -2*ibox to 2*ibox inclusive.
   * However, various other routines require the FFT box size to be an
   * even integer, so we start the fftbox-size at 4*ibox+2.
   * The loop tries to factorize the fftbox-size into (2,3,5,7)...if that
   * isn't doable, it increases the fftbox-size by 2 and tries again. */

  dft_log("Minimal FFT box size factoring into (2,3,5,7):\n");

  for (i=0; i < 3; i++)
    {
      int b,n2,n3,n5,n7,done_factoring;

      fftbox[i] = 4*ibox[i]+2  -2;
      /* increase fftbox[i] by 2 and try to factor it into (2,3,5,7) */
      do
	{
	  fftbox[i] += 2;
	  b = fftbox[i];
	  n2 = n3 = n5 = n7 = done_factoring = 0;
	  while (!done_factoring)
	    {
	      if (b%2==0) { n2++; b /= 2; continue; }
	      if (b%3==0) { n3++; b /= 3; continue; }
	      if (b%5==0) { n5++; b /= 5; continue; }
	      if (b%7==0) { n7++; b /= 7; continue; }
	      done_factoring = 1;
	    }
	}
      while (b != 1); /*  b==1 means fftbox[i] is (2,3,5,7) factorizable */
      dft_log("fftbox[%d] = %d =(2^%d)*(3^%d)*(5^%d)*(7^%d)\n",
		i,fftbox[i],n2,n3,n5,n7);
    }

  /* If we're given FFT box sizes to use, then use them! */
  if (Nx_spec != 0 || Ny_spec != 0 || Nz_spec !=0)
    {
      fftbox[0] = Nx_spec;
      fftbox[1] = Ny_spec;
      fftbox[2] = Nz_spec;
      dft_log("==============================================\n");
      dft_log("Overiding with specified sizes: %d by %d by %d\n",
		Nx_spec,Ny_spec,Nz_spec);
      dft_log("==============================================\n");
    }

  for (i=0; i < 3; i++)
    if (fftbox[i] < 4*ibox[i]+2)
      {
	dft_log(DFT_SILENCE,
		"\nsetup_basis():  fftbox[%d] is too small.\n",i);
	die("It should be AT LEAST %d = 4*%d+2\n\n",4*ibox[i]+2,ibox[i]);
      }
  Nx = fftbox[0];
  Ny = fftbox[1];
  Nz = fftbox[2];
  NxNyNz = Nx * Ny * Nz;

  dft_log("Using fftbox = %d by %d by %d\n",
	    fftbox[0],fftbox[1],fftbox[2]);


  // Now copy the common info to all basis.
  for (i=0; i <= nkpts; i++) {
    basis[i].latvec           = latvec;
    basis[i].G                = G;
    basis[i].GGT              = GGT;
    basis[i].unit_cell_volume = unit_cell_volume;
    basis[i].Nx               = Nx;
    basis[i].Ny               = Ny;
    basis[i].Nz               = Nz;
    basis[i].NxNyNz           = NxNyNz;
    basis[i].Gxmin            = Gxmin;
    basis[i].Gxmax            = Gxmax;
    basis[i].Gymin            = Gymin;
    basis[i].Gymax            = Gymax;
    basis[i].Gzmin            = Gzmin;
    basis[i].Gzmax            = Gzmax;
  }


  /* Figure out the G-vectors within the cutoff energy.
   * First count up how many there are, then put them in the basis struct.
   * In the continuum limit, there should be (4*pi/3)*(2*Ecut)^(3/2)/det(G)
   * points:  divide the volume of the maximum energy sphere in G-space by
   * the volume of the primitive cell in G-space. */
  dft_log("\nG-vectors with energies <= Ecut:\n");
  G2max = 0.0;
  real avg_nbasis = 0.0;
  for (int kpt = 0; kpt < nkpts; kpt++) {
    basis[kpt].nbasis = 0;
    for (i = -ibox[0]; i <= ibox[0]; i++)
      for (j = -ibox[1]; j <= ibox[1]; j++)
	for (k = -ibox[2]; k <= ibox[2]; k++) {
	  f.v[0] = i;
	  f.v[1] = j;
	  f.v[2] = k;
	  f += elecinfo.kvec[kpt];
	  G2 = f*(basis[kpt].GGT*f);
	  if (0.5*G2 <= Ecut)
	    basis[kpt].nbasis++;
	}
    avg_nbasis += basis[kpt].nbasis * elecinfo.w[kpt];
    dft_log("nbasis = %d for k = [%6.3f %6.3f %6.3f]\n",
	      basis[kpt].nbasis, 
	      elecinfo.kvec[kpt].v[0],
	      elecinfo.kvec[kpt].v[1],
	      elecinfo.kvec[kpt].v[2]);
  
    basis[kpt].Gx = (int *)mymalloc(sizeof(int)*basis[kpt].nbasis,
				    "Gx","setup_basis()");
    basis[kpt].Gy = (int *)mymalloc(sizeof(int)*basis[kpt].nbasis,
				    "Gy","setup_basis()");
    basis[kpt].Gz = (int *)mymalloc(sizeof(int)*basis[kpt].nbasis,
				    "Gz","setup_basis()");
    basis[kpt].index = (int *)mymalloc(sizeof(int)*basis[kpt].nbasis,
				       "index","setup_basis()");
    n = 0;
    for (i = -ibox[0]; i <= ibox[0]; i++)
      for (j = -ibox[1]; j <= ibox[1]; j++)
	for (k = -ibox[2]; k <= ibox[2]; k++)
	  {
	    f.v[0] = i;
	    f.v[1] = j;
	    f.v[2] = k;
	    f += elecinfo.kvec[kpt];
	    G2 = f*(basis[nkpts].GGT*f);
	    if (0.5*G2 <= Ecut)
	      {
		if (G2 > G2max)
		  G2max = G2;
		basis[kpt].Gx[n] = i;
		basis[kpt].Gy[n] = j;
		basis[kpt].Gz[n] = k;
		index = 0;
		if (k >= 0) index += k;
		else        index += k+fftbox[2];
		if (j >= 0) index += fftbox[2]*j;
		else        index += fftbox[2]*(j+fftbox[1]);
		if (i >= 0) index += fftbox[2]*fftbox[1]*i;
		else        index += fftbox[2]*fftbox[1]*(fftbox[0]+i);
		basis[kpt].index[n] = index;
		dft_log(DFT_ANAL_LOG,
			"G = [ %3d %3d %3d ]  index = %7d  G2 = %lg\n",
			basis[kpt].Gx[n],basis[kpt].Gy[n],basis[kpt].Gz[n],
			basis[kpt].index[n],G2);
		n++;
	      }
	  }
  }
  real ideal_nbasis = sqrt(2*Ecut);
  ideal_nbasis *= (ideal_nbasis * ideal_nbasis);
  ideal_nbasis *= (unit_cell_volume / 6 / M_PI / M_PI);
  dft_log("weighted average nbasis = %4.2f , ideal nbasis = %6.3f\n",
	  avg_nbasis, ideal_nbasis);
  dft_log("\n0.5*|G|^2 = %lg for highest energy plane-wave\n\n", 0.5*G2max);
  dft_log_flush();

  // the last basis is for charge density FFT, set index arrays to null.
  basis[nkpts].nbasis = 0;
  basis[nkpts].Gx = basis[nkpts].Gy = basis[nkpts].Gz = NULL;
  basis[nkpts].index = NULL;

}

/*
 * Free all the memory taken up by the basis structure 
 *
 * 5/25/1999 Tairan Wang:  there are nkpts+1 basis structures.
 * 1. the last one is for charge density FFTs and has not allocated 
 *    any index arrays, so there is nothing to do for that one.
 * 2. for the first nkpts ones, there are two scenarios:
 *   2.1. all the basis actually points to the same index arrays,
 *       this is the case when keyword  "basis single" is used.
 *       only need to free once. 
 *   2.2. the basis are kpoint dependent, this is the case when
 *       keyword "basis kpoint-dependent" is used.
 *       need to free for each one of them.
 *   The two scenarios are distinguished by comparing the index
 *   pointer of the structures with that of the first one.
 */
void
free_basis(Basis *basis,int nkpts)
{
  int i;

  for (i=nkpts-1; i>=0; i--) {
    if ( (i > 0) && (basis[i].index == basis[0].index) ) {
      // do nothing.
    } else {
      myfree(basis[i].Gx);
      myfree(basis[i].Gy);
      myfree(basis[i].Gz);
      myfree(basis[i].index);
    }
  }
}
