/*
 *                   Tairan Wang                    April 8, 1999
 * 
 * setup.c -- implement a centralized key-string format input system
 *
 */

/* $Id: setup.c,v 1.3 1999/11/20 04:46:01 tairan Exp $ */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include "header.h"
#include "parallel.h"


/*
 * setup()
 *
 *------------------------------------------------------------------------
 * 
 * Section 1:  Introduction
 *
 * The New Key-Value Free Ordering  Format:
 *
 * Users:  read Section 2 for implemented Keys.
 *
 * Programmers:  read Section 3 on programming details and how to add new Keys.
 *
 *
 *
 *------------------------------------------------------------------------
 * 
 * Section 2:  Format
 *
 * The new format is free ordering:

# starts a line that is a comment
# use \  for line continuation.


# Flag to specify what type of calculations:
job testing
job electronic-minimization
job calculate-forces
job ionic-relaxation
job ionic-dynamics
# 1. testing  : debug flag, will process input file and print startup info and stop.
# 2. ionic-relaxation and ionic-dynamics  cannot be both set.
# 3. for  ionic-relaxation and ionic-dynamics,  electronic-minimization and calculate-forces
#    flags will also be set.

# latt keywords are all done in the first pass
lattice <a1_x> <a2_x> <a3_x>  \
        <a1_y> <a2_y> <a3_y>  \
	<a1_z> <a2_z> <a3_z>  # unit cell length is these numbers x
latt-type  [ parameters ]  {not ready}
latt-scale <numa> [<numb> <numc>]  # 1 number--uniform; 3 numbers--different scalings in x y z
latt-supercell <nx> <ny> <nz>  {not ready}
latt-nosymmetries   # turns off symmetry calculation. Default: autodetect
latt-symmetry <3x3=9 numbers> ...  # till the end. :)



# ion coordinates/ relative in  unit-cell units.
ion-species <identifier> <Z> <mass> <pot-filename> <pulay-filename> <type> [l-component local-component]
ion <identifier> <rx> <ry> <rz> <t/f:1/0>  # <identifier> can be string or number (doesn't matter)



# electronic parameters
elec-cutoff <Ecut>
elec-n-empty-bands <n>
elec-fermi-fillings <niter_recalc> <kT> [eV | K]
elec-ex-corr <lda/gga/...>
elec-initial-fillings automatic | read <file> # auto. calc or read from file



# k-point specifications
kpoint <kx> <ky> <kz> <weight> # in relative coordinates in B.Z.
kpoint-folding <nx> <ny> <nz>



# choose a single basis (at Gamma point) or k-point dependent basis sets.
basis [single/kpoint-dependent]



# controls on electronic minimization and how it is done
cntrl-max-ionic-steps <n>
cntrl-max-elec-steps <n>
cntrl-elec-stepsize <size>
cntrl-energy-convergence <e_tolerance>
cntrl-force-convergence <f_tolerance>
cntrl-update-stepsize [y|n|t|f|1|0]
cntrl-stepsize-min <min_stepsize>
cntrl-nel-fermi-tol <tol>
cntrl-subspace-rotation [<pcond>]
cntrl-dump-n [y|n|t|f|1|0]
cntrl-algorithm  [ PCG | CG | EOM | PSD | ... ] [ method specific parameters ]



# ewald information:
ewald [start_realspace end_realspace start_recip end_recip]




# FFT box override
fftbox <n1> <n2> <n3>




#
# additional initialization files:
#

# default: random wavefunction/atomic based. 
# optional file2 is for ion-dyn projection
wavefunction <filename> [optional <filename2>]


# dynamics part
ion-dynamics velsc
ion-dynamics kT  <kT> [eV | K]
ion-dynamics stepsize <size in femtosecond>
ion-dynamics [new | continue [ iter ]] 
# if  "ion-dynamics continue"  and  "iter" not specified, 
#  "iter" is read from the ion position file


# dumping partial charge density from bands
orbital_density <kpoint-index> <band-index>
...

# ion relaxation.
ion-relax <species name> <damp> <fac>
...

# can specify constraints on atomic motions: need to think more about this.
# 1. single atom constraint,  2. total center-of-mass constraint, 
# 3. 
constraint_line  [direction (relative coord) (x,y,z)] <sp-index> <atom-index>
center-of-mass  F F T  # center of mass can only move in z direction. {not implemented}


# suppress template printing in real runs.
template [0|f|n]





 *
 *
 *------------------------------------------------------------------------
 * 
 * Section 3:  Programmer's Guide
 *
 *
 * 1. What is  setup()?
 *
 * The  setup() function replace the original set of set up routines. 
 *
 *
 * 2. How does it work?
 *
 * It goes through the input file to gather needed information.
 *
 * If it does not find the <job key> anywhere, it assumes this is an old input 
 * file and drops back to the original input routines for program "minim".
 *
 * A valid entry should be a line of text and/or numbers with the first token
 * being the <key>.  This <key> tells the program what to do with the rest of 
 * the line.
 *
 * The order of the entries should not be important. It is up to the programmer
 * to guarantee this freedom.
 *
 * Some of the entries have default values, i.e. if User does not specify those 
 * entries in the input file, the defaults are used.  
 *
 * The rest of the entries MUST have inputs. It is up to the programmer to WARN 
 * the user and/or quit the job when these essential entries are not found.
 *
 *
 * 3. What are the multiple passes?
 * 
 * The setup() routine actually goes through the input file multiple times.
 * This is to guarantee that the data are read in the correct order. Each pass 
 * is documented at its beginning.
 *
 *
 * 4. How to add new Keys?
 *
 * If you would like to add new keywords to initialize some new data, you should
 * most likely add those to the last pass, which is commented to be for new keys.
 * However, if you need to modify some datastructure in earlier passes, you can.
 * Just remember to use caution.
 *
 * If your newly added keyword allow default values, add those in the default
 * section near the beginning of the setup() routine.
 * 
 */

void
setup(dft_text_FILE *filep, Parameters &params,
      Elecinfo &elecinfo, Basis **basis, 
      IonDyninfo &iondyninfo, Control &cntrl)
{
  char key[DFT_MSG_LEN], line[DFT_LINE_LEN];
  int found;

  int sp, kpt, i, j, k;
  char **ion_names, *ch_p;
  int *ion_nats, *sp_found;
  int ion_max_nspecies=10; // automatically increased if not enough.


  /********************************************************************
   ***                                                              ***
   *** Setup defaults:                                              ***
   ***                                                              ***
   ********************************************************************/

  /*-------------------------------------------------*
   * some local variables for temporary operations   *
   *-------------------------------------------------*/
  // for lattice info init, but before we know exactly
  // how many k-points there are.
  Basis tmp_basis;
  vector3 *kvec = NULL;
  real *w = NULL;


  /*-------------------------------------------------*
   * default values for various parameters.          *
   *-------------------------------------------------*/
  int basis_flag = -1; // 0: use single basis;
                       // 1: use kpoint-dependent basis;
                       // there is no default.

  int nfft[3] = { 0, 0, 0 };  // default fft box size: i.e. figure it
                              // automatically

  matrix3 latvec(10.174,10.174,10.174); // some arbitrary lattice.
  real latscale[3] = { 1.0, 1.0, 1.0 }; // no default scaling of lattice
  int calc_symm_flag = DFT_TRUE;        // k-point symmetrization on by default

  int kpt_fold[3] = { 1, 1, 1 };    // no kpoint folding by default

  int Nlat_start_real = 1, Nlat_end_real = 5,    // defaults ewald sum
      Nlat_start_recip = 1, Nlat_end_recip = 6;  //  limits

  Ioninfo &ioninfo = iondyninfo.ioninfo;
  iondyninfo.ionic_kT = 0.0;          // ionic kT = 0. (for dynamics)
  iondyninfo.ionic_stepsize = 2.0;    // ionic stepsize = 2.0 (dynamics)
  iondyninfo.ion_start_iter = 0;      // starting ionic iter. (dyn., restart)
  strcpy(iondyninfo.ion_dyn_action,"new");  // new ion dynamics (for dynamics)
  iondyninfo.velsc_flag  = DFT_FALSE;  // do velocity rescaling (for dynamics)
  ioninfo.nspecies = 0;               // 0 species.
  ioninfo.nrot = 0;                   // 0 symmetries

  elecinfo.Ecut = 0;                  // 0 Hartree cutoff.
  elecinfo.nkpts = 0;                 // 0 kpoint
  elecinfo.nbands = 0;                // 0 number of bands.
  elecinfo.nelectrons = 0;            // 0 electrons
  elecinfo.calc_fillings = DFT_FALSE;  // no filling recalculation
  elecinfo.subspace_rotation = DFT_FALSE; // no subspace rotation
  elecinfo.ex_opt = DFT_EXCORR_LDA;  // use LDA exchange-correlation 
  elecinfo.initial_fillings_flag = 0;  // by default, automatic fillings
  strcpy(elecinfo.initial_fillings_filename,"\0"); // null name by default

  int nemptybands = 0;               // no empty bands by default

  /*
   * Allocate some temporary space for ions to speed up parsing: 
   * The space will be increased later in the code if the need arises.
   * At this point, allocate for 10 species.
   */
  ion_names = (char**)mymalloc(sizeof(char*)*ion_max_nspecies,
			       "ion_names","setup");
  ion_nats = (int*)mymalloc(sizeof(int)*ion_max_nspecies,"ion_nats","setup");
  ch_p = (char*)mymalloc(sizeof(char)*DFT_MSG_LEN*ion_max_nspecies,
			 "ion_names[]","setup");
  for (i = 0; i < ion_max_nspecies; i++, ch_p += DFT_MSG_LEN)
    {
      ion_names[i] = ch_p;
      ion_nats[i] = 0;
    }


  /********************************************************************
   ***                                                              ***
   *** Go through multiple passes of the input file to read         ***
   ***  parameters.                                                 ***
   ***                                                              ***
   ********************************************************************/

  /* 

     Setup order: (0-5,p1-p3)

     Things at the same level can be done in any order.
     p1,p2,p3  are overall-post-processing orders, and must be done
     after all other passes.


     (0) job_type 

     (Basis)
     (1) latvec, latscale
     (p3) complete basis

     (Elecinfo)
     (1) Ecut
     (1) nkpts, kpoint folding
     (1) calc_fillings, niter_recalc_fillings, kT
     (1) subspace_rotation, pcond_for_Bgrad (left in Control section)
     (1) ex_opt
     (1) initial fillings
     (2) kvec, w (initial)
     (p1) nelectrons, nbands
     (p2) kvec, w (final)

     (IonDyninfo)
     (1) velsc_flag
     (1) ion_start_iter, ion_dyn_action
     (1) ionic_kT, ionic_stepsize
     (1) nspecies
     (2) Z, mass, name (for each species)
     (1) natoms (for each species)
     (1) atpos, move
     (1) calc_symm_flag
     (p1) symmetry

     (Ewald)
     (1) ewald limits

     (Control)
     (1) max_ionic_steps, max_elec_steps
     (1) e_tolerance, f_tolerance, nel_fermi_tol
     (1) update_stepsize, stepsize_min, 

     (Other)
     (1) read_Y_flag, init_Y_filename, old_Y_filename
     (1) calc_partial_n_flag
  */

  /****************************************************************
   *                                                              *
   *  Zeroth pass: find "job" keyword                             *
   *                                                              *
   ****************************************************************/

  // filep must have been already opened
  dft_text_rewind(filep);
  found = DFT_FALSE;

  // Keep reading till end of file skipping empty lines and comments
  while (dft_text_fgets(line,DFT_LINE_LEN,'\\',filep) != NULL) {
    if (line[0]=='#' || sscanf(line,"%s",key) <= 0)
      continue;

    // find lines starting with "job" string
    if (MATCH(key,"job") ) {
      
      // Put the string after "job" into key
      if (sscanf(line,"%*s %s",key) > 0) {

	/*
	 * Set the job_type variable and do some job specific initialization
	 */
	/* case 1 */
	if ( MATCH( key,"ionic-dynamics") ) {	  
	  params.ionic_dynamics_flag = DFT_TRUE;
	  found = DFT_TRUE;
	} 
	
	/* case 2 */
	else if ( MATCH( key,"ionic-relaxation") ) {	  
	  params.ionic_relaxation_flag = DFT_TRUE;
	  found = DFT_TRUE;
	} 
	
	/* case 3 */
	else if ( MATCH(key,"electronic-minimization") ) {	  
	  params.electronic_minimization_flag = DFT_TRUE;
	  found = DFT_TRUE;
	} 
	
	/* case 4 */
	else if ( MATCH(key,"calculate-forces") ) {
	  params.calculate_forces_flag = DFT_TRUE;
	  found = DFT_TRUE;
	}

	/* case 5, for debugging purpose */
	else if ( MATCH(key,"testing") ) {
	  params.testing_flag = DFT_TRUE;
	  found = DFT_TRUE;
	}

	/* default */
	else {	  
	  die("Unrecognized job type: %s\n",key);
	}
	
	dft_text_fputc_old('#',filep); // convert this line to comment
      }
    }
  }

  /****************************************************************
   *                                                              *
   *  Zeroth pass post-processing                                 *
   *                                                              *
   ****************************************************************/

  // If a valid job type was not found, then time to die!
  if ( !found )
    die(">> You need to specify a valid job type!\n");

  // check job-request consistency:
  // ionic_dynamics_flag and ionic_relaxation_flag cannot both be set
  if ( params.ionic_dynamics_flag && params.ionic_relaxation_flag ) {
    dft_log(DFT_SILENCE,
	    ">> Can't do both ionic-dynamics and ionic-relaxation\n");
    dft_log(DFT_SILENCE,
	    ">> Please consider doing only one of them.\n");
    die(">> Terminating.\n");
  }
  // if we want ionic dynamics, then we had better also do
  // electronic minimization and force calculations!
  if ( params.ionic_dynamics_flag &&
       ( (! params.electronic_minimization_flag) ||
	 (! params.calculate_forces_flag) ) ) {
    dft_log(DFT_SILENCE,
	    ">> If you want to do ionic-dynamics, please turn on also \n");
    dft_log(DFT_SILENCE,
	    ">> electronic-minimization flag and calculate-forces flag\n");
    dft_log(DFT_SILENCE,
	    ">> I will do it for you this time.\n");
    // die(">> Terminating.\n");
    params.electronic_minimization_flag = DFT_TRUE;
    params.calculate_forces_flag = DFT_TRUE;
  }
  // if we want ionic dynamics, then we had better also do
  // electronic minimization and force calculations!
  if ( params.ionic_relaxation_flag &&
      ( (! params.electronic_minimization_flag) ||
       (! params.calculate_forces_flag) ) ) {
    dft_log(DFT_SILENCE,
	    ">> If you want to do ionic-relaxation, please turn on also \n");
    dft_log(DFT_SILENCE,
	    ">> electronic-minimization flag and calculate-forces flag\n");
    dft_log(DFT_SILENCE,
	    ">> I will do it for you this time.\n");
    params.electronic_minimization_flag = DFT_TRUE;
    params.calculate_forces_flag = DFT_TRUE;
  }


  /****************************************************************
   *                                                              *
   *  First pass:                                                 *
   *                                                              *
   * keywords:                                                    *
   *  latt*, ion, elec ecut, elec extra-charge, elec empty-bands, *
   *  elec initial filllings                                      *
   *  elec kpt, kpt-folding, ewald, fermi-filling,                *
   *  algorithm, init, dynamics, velsc                            *
   ****************************************************************/
  // Prepare the first pass.
  dft_text_rewind(filep);
  // Keep reading till end of file skipping empty lines and comments
  while (dft_text_fgets(line,DFT_LINE_LEN,'\\',filep) != NULL) {
    if (line[0]=='#' || sscanf(line,"%s",key) <= 0)
      continue;
    
    // matching keywords

    /* case 1, lattice stuffs */
    if ( MATCHn(key,"latt",4) ) {      
      
      /* case 1.1, lattice vectors */
      if ( MATCH(key, "lattice") ) {	
	// lattice <num1> ... <num9>  
	sscanf(line,"%*s %lg %lg %lg %lg %lg %lg %lg %lg %lg",
	       &(latvec.m[0][0]), &(latvec.m[0][1]), &(latvec.m[0][2]), 
	       &(latvec.m[1][0]), &(latvec.m[1][1]), &(latvec.m[1][2]), 
	       &(latvec.m[2][0]), &(latvec.m[2][1]), &(latvec.m[2][2]) ); 
	
      }
      /* case 1.2, lattice scalings */
      else if ( MATCH(key,"latt-scale") ) {	
	// latt-scale <numa> [<numb> <numc>]  
	if (sscanf(line,"%*s %lg %lg %lg",
		   &(latscale[0]), &(latscale[1]), &(latscale[2])) < 3) {
	  // assume single lattice scale
	  latscale[2] = latscale[1] = latscale[0];
	}
	
      }
      /* case 1.3, symmetry flag */
      else if ( MATCH(key,"latt-nosymmetries") ) {	
	// latt-nosymmetries 
	calc_symm_flag = DFT_FALSE;
	
      }
      /* case 1.4, symmetries input */
      else if ( MATCHn(key,"latt-symmetry", 15) ) {
	// latt-symmetry <num1> ... <num9>  
	sscanf(line,"%*s %lg %lg %lg %lg %lg %lg %lg %lg %lg",
	       &(ioninfo.sym[ioninfo.nrot].m[0][0]), 
	       &(ioninfo.sym[ioninfo.nrot].m[0][1]), 
	       &(ioninfo.sym[ioninfo.nrot].m[0][2]), 
	       &(ioninfo.sym[ioninfo.nrot].m[1][0]), 
	       &(ioninfo.sym[ioninfo.nrot].m[1][1]), 
	       &(ioninfo.sym[ioninfo.nrot].m[1][2]), 
	       &(ioninfo.sym[ioninfo.nrot].m[2][0]), 
	       &(ioninfo.sym[ioninfo.nrot].m[2][1]), 
	       &(ioninfo.sym[ioninfo.nrot].m[2][2]) );
	ioninfo.nrot++;
      }
      /* unknown option... die! */
      else
	die("%s not yet implemented!\nExiting.\n",key);
    }

    /* case 2, ions: (first pass)
     * 
     * Identify the ion lines and count the number of species
     * and their respective number of atoms.
     */
    else if ( MATCH(key,"ion") ) {
      // ion <identifier> <rx> <ry> <rz> <t/f:1/0>  
      found = DFT_FALSE;
      sp = 0;
      sscanf(line,"%*s %s",key);
      // if species already found, add it to the atoms list
      while ( (!found) && (sp < ioninfo.nspecies) ) {
	if ( MATCH(key, ion_names[sp]) ) {
	  found = DFT_TRUE;
	  ion_nats[sp]++;
	} else {
	  sp++;
	}
      }
      // if not found, increase species list.
      if ( !found ) {
	sp = ioninfo.nspecies;
	strncpy(ion_names[sp],key, DFT_MSG_LEN);
	ion_nats[sp] = 1;
	ioninfo.nspecies++;
      }
      // if exceeds maximum number of species, expand temporary storage.
      if (ioninfo.nspecies >= ion_max_nspecies) {
	ion_max_nspecies += 10;
	
	int * int_p = (int*)mymalloc(sizeof(int)*ion_max_nspecies,
				     "int_p","setup");
	char * ch_p2 = (char*)mymalloc(sizeof(char)*DFT_MSG_LEN*
				       ion_max_nspecies,
				       "ch_p2","setup");
	
	for (sp = 0, ch_p = ion_names[0];
	     sp < ioninfo.nspecies; 
	     sp++, ch_p2 += DFT_MSG_LEN) {
	  int_p[sp] = ion_nats[sp];
	  strncpy(ch_p2, ion_names[sp], DFT_MSG_LEN);
	  ion_names[sp] = ch_p2;
	}
	myfree(ion_nats);
	myfree(ch_p);
	ion_nats = int_p;
      }
    }

    /* case 3,  electrons */
    else if ( MATCHn(key,"elec",4) ) {      
      
      /* case 3.1, cutoff energy */
      if ( MATCH(key,"elec-cutoff") ) {	
	// elec-cutoff <Ecut>
	sscanf(line,"%*s %lg",&elecinfo.Ecut);
      }
      /* case 3.2, number of empty bands */
      else if ( MATCHn(key,"elec-n-empty-bands",13) ) {	
	// elec-n-empty-bands <n>
	sscanf(line,"%*s %d", &nemptybands);
	if (nemptybands < 0)
	  die(">> N empty bands must >= 0 (%d)\n>> Terminating.\n",
	      nemptybands);
      }
      /* case 3.3, fermifilling calculation parameters */
      else if ( MATCH(key,"elec-fermi-fillings") ) {  
	// elec-fermi-fillings <niter_recalc> <kT> [eV | K]
	elecinfo.calc_fillings = DFT_TRUE;
	if ( sscanf(line,"%*s %d %lg %s", 
		    &elecinfo.niter_recalc_fillings,
		    &elecinfo.kT,
		    key) > 2 ) {
	  if ( MATCH(key,"eV") )
	    elecinfo.kT *= eV2Hartree;
	  else if ( MATCH(key,"K") )
	    elecinfo.kT *= K2Hartree;
	}
      }
      /* case 3.4, exchange correlation */
      else if ( MATCH(key,"elec-ex-corr") ) { 
	// elec-ex-corr <lda/gga/...>
	sscanf(line,"%*s %s", key);
	if ( MATCH(key,"gga") )
	  elecinfo.ex_opt = DFT_EXCORR_GGA;
	else if ( MATCH(key,"lda") )
	  elecinfo.ex_opt = DFT_EXCORR_LDA;
	else {
	  dft_log(DFT_SILENCE,">> I don't know of XC: %s\n",key);
	  die(">> Maybe you want to add it to my code?\n");
	}
	
      }
      /* case 3.5, how to setup the initial band fillings */
      else if ( MATCH(key,"elec-initial-fillings") ) {
	// elec-initial-fillings auto/read <file>
	sscanf(line,"%*s %s",key);
	// Do fillings automatically
	if ( MATCH(key,"automatic") )
	  elecinfo.initial_fillings_flag = 0;
	// Read initial fillings from a speicified file
	else if ( MATCH(key,"read") )
	  {
	    elecinfo.initial_fillings_flag = 1;
	    // if the filename was not found, we have a problem!
	    if ( sscanf(line,"%*s %*s %s",
			elecinfo.initial_fillings_filename) <= 0)
	      die(">> Can't get file name containing initial fillings.\n");
	  }
	// Oops!
	else
	  die(">>%s is unkown option for elec-initial-fillings\n",key);
      }
      /* default: unknown elec- option */
      else
	die("%s not yet implemented!\nExiting.\n",key);
    }


    /* case 4,  k-points: (first pass) */
    else if ( MATCHn(key,"kpoint",6) ) {
      
      /* case 4.1, kpoint relative coordinates */
      if ( MATCH(key,"kpoint") )
	elecinfo.nkpts ++;

      /* case 4.2, kpoint folding grids */
      else if ( MATCH(key,"kpoint-folding") ) {
	sscanf(line,"%*s %d %d %d",
	       &kpt_fold[0], &kpt_fold[1], &kpt_fold[2] );
      } 
      /* default */
      else {
	die("%s not yet implemented!\nExiting.\n",key);
      }
    }
    
    
    /* case 5, ewald setup */
    else if ( MATCH(key,"ewald") ) {
      
      sscanf(line,"%*s %d %d %d %d",
	     &Nlat_start_real, &Nlat_end_real,
	     &Nlat_start_recip, &Nlat_end_recip);
    }
    
    
    /* case 6, control object */
    else if ( MATCHn(key,"cntrl",5) ) {
      if ( MATCH(key,"cntrl-max-ionic-steps") ) {
	// cntrl-max-ionic-steps <n>
	sscanf(line,"%*s %d",&(cntrl.max_ionic_steps));
      } else if ( MATCH(key,"cntrl-max-elec-steps") ) {
	// cntrl-max-elec-steps <n>
	sscanf(line,"%*s %d",&(cntrl.max_elecs_steps));
      } else if ( MATCH(key,"cntrl-elec-stepsize") ) {
	// cntrl-elec-stepsize <size>
	sscanf(line,"%*s %lg",&(cntrl.elec_stepsize));
	if (cntrl.elec_stepsize > 0.0) // the step has to be opposite the gradient.
	  cntrl.elec_stepsize = -cntrl.elec_stepsize;
      } else if ( MATCH(key,"cntrl-energy-convergence") ) {
	// cntrl-energy-convergence <e_tolerance>
	sscanf(line,"%*s %lg",&(cntrl.E_tolerance));
      } else if ( MATCH(key,"cntrl-force-convergence") ) {
	// cntrl-force-convergence <f_tolerance>
	sscanf(line,"%*s %lg",&(cntrl.force_tolerance) );
      } else if ( MATCH(key,"cntrl-update-stepsize") ) {
	if (sscanf(line, "%*s %s", key) > 0) {
	  if ( match_no(key) ) 
	    cntrl.update_stepsize = DFT_FALSE;
	}
      } else if ( MATCH(key,"cntrl-stepsize-min") ) {
	// cntrl-stepsize-min <min_stepsize>
	sscanf(line,"%*s %lg",&(cntrl.stepsize_min));
      } else if ( MATCH(key,"cntrl-nel-fermi-tol") ) {
	sscanf(line,"%*s %lg",&(cntrl.nel_fermi_tol));
      } else if ( MATCH(key,"cntrl-subspace-rotation") ) {
	elecinfo.subspace_rotation = DFT_TRUE;
	if (sscanf(line,"%*s %lg",&(elecinfo.pcond_for_Bgrad)) < 1)
	  elecinfo.pcond_for_Bgrad = 30;
      } else if ( MATCH(key,"cntrl-dump-n") ) {
	if (sscanf(line, "%*s %s", key) > 0) {
	  if ( match_no(key) )
	    cntrl.dump_n = DFT_FALSE;
	}
      } else if ( MATCH(key,"cntrl-algorithm") ) {
	sscanf(line,"%*s %s", key);
	
	if ( MATCH(key,"PCG") )
	  cntrl.algorithm = DFT_PCG;
	else if ( MATCH(key,"PCG_nocos") )
	  cntrl.algorithm = DFT_PCG_nocos;
	else if ( MATCH(key,"CG") )
	  cntrl.algorithm = DFT_CG;
	else if ( MATCH(key,"CG_nocos") )
	  cntrl.algorithm = DFT_CG_nocos;
	else if ( MATCH(key,"EOM") )
	  cntrl.algorithm = DFT_EOM;
	else if ( MATCH(key,"PSD") )
	  cntrl.algorithm = DFT_PSD;
	else {
	  dft_log(DFT_SILENCE,
		  "I don't know of algorithm %s\n",key);
	  dft_log(DFT_SILENCE,
		  "Maybe it's a typo. I only know of [ PCG | PCG_nocos | CG | CG_nocos | PSD | EOM ]\n");
	  die("If this one is some new fancy algorithm, maybe you should code it and add to my code.\n");
	}
      } else {
	die("%s not yet implemented!\nExiting.\n",key);
      }
    }
    
    /* case 7, wavefunction initalization */
    else if ( MATCH(key,"wavefunction") ) {
      
      int items = sscanf(line,"%*s %s %s",
			 params.init_Y_filename, params.old_Y_filename);
      if (items < 1) {
	params.read_Y_flag = DFT_FALSE;
      } else {
	params.read_Y_flag = DFT_TRUE;
	if (items < 2)
	  strcpy(params.old_Y_filename,"C.temp");
      }
    }
    
    
    /* case 8, parameters for Ion dynamics */
    else if ( MATCH(key,"ion-dynamics") ) {
      if (sscanf(line,"%*s %s",key) > 0) {
	if ( MATCH(key, "velsc") ) {
	  iondyninfo.velsc_flag = DFT_TRUE;
	} else if ( MATCH(key, "kT") ) {
	  if (sscanf(line,"%*s %*s %lg %s", &iondyninfo.ionic_kT, key)>1) {
	    if ( MATCH(key,"eV") )
	      iondyninfo.ionic_kT *= eV2Hartree;
	    else if ( MATCH(key,"K") )
	      iondyninfo.ionic_kT *= K2Hartree;
	  }
	} else if ( MATCH(key, "stepsize") ) {
	  sscanf(line,"%*s %*s %lg", &iondyninfo.ionic_stepsize);
	} else if ( MATCH(key, "new") ) { // new ion dynamics
	  iondyninfo.ion_start_iter = 0;
	  strcpy(iondyninfo.ion_dyn_action, "new");
	} else if ( MATCH(key, "continue") ) {
	  strcpy(iondyninfo.ion_dyn_action, "continue");
	  if (sscanf(line, "%*s %*s %d", &iondyninfo.ion_start_iter)<1) {
	    iondyninfo.ion_start_iter = -1; // read this from IONLOC file.
	  }
	}
      }
    }
    
    /* case 10, fftbox manual input */
    else if ( MATCH(key,"fftbox") ) {
      // fftbox <n1> <n2> <n3>
      // override for fftbox size.  with out this, FFT box is automatically generated.
      sscanf(line,"%*s %d %d %d",&nfft[0],&nfft[1],&nfft[2]);
    }
    
    /* case 11, basis flag */
    else if ( MATCH(key,"basis") ) {
      // basis [single/kpoint-dependent]
      sscanf(line,"%*s %s",key);
      if ( MATCH(key,"single") )
	basis_flag = 0;
      else if ( MATCH(key,"kpoint-dependent") )
	basis_flag = 1;
      else
	die(">> Unknown basis option: %s.\nUse: basis [single | kpoint-dependent]\n",key);
    }
    
  }

  /****************************************************************
   *                                                              *
   *  First pass post-processing:                                 *
   *                                                              *
   ****************************************************************/
  
  // Do allocations according to information already gathered.
  // nspecies
  // nkpts
  // These two are only temporary, need to be adjusted in the kpoint-folding/reduction part.
  kvec = (vector3 *)mymalloc(sizeof(vector3)*elecinfo.nkpts,"kvec","setup");
  w = (real *)mymalloc(sizeof(real)*elecinfo.nkpts,"w", "setup");

  /*
   * For ions, number of species and their
   * respective number of atoms are known.
   * Next, get the info-line for each species,
   * and get the positions for all atoms.
   */
  ioninfo.species = (Speciesinfo *)mymalloc(sizeof(Speciesinfo)*ioninfo.nspecies,
					    "speciesinfo[]","setup");
  sp_found = (int*)mymalloc(sizeof(int)*ioninfo.nspecies,"sp_found[]","setup");
  for (sp = 0; sp < ioninfo.nspecies; sp++) {
    strcpy(ioninfo.species[sp].name,ion_names[sp]); // copy the name
    ioninfo.species[sp].natoms = 0; // use this as a counter when reading ion positions.
    ioninfo.species[sp].atpos = (vector3*)mymalloc(sizeof(vector3)*ion_nats[sp],"atpos[]","setup");
    ioninfo.species[sp].forces = (vector3*)mymalloc(sizeof(vector3)*ion_nats[sp],"forces[]","setup");
    // The following is for ion relaxations
    ioninfo.species[sp].ion_speed = (vector3*)mymalloc(sizeof(vector3)*ion_nats[sp],"ion_speed[]","setup");
    ioninfo.species[sp].move = (int*)mymalloc(sizeof(int)*ion_nats[sp],"move[]","setup");
    sp_found[sp] = DFT_FALSE; // species-found-flag.
    dft_log(">>(ion) %s %d\n", ioninfo.species[sp].name, ion_nats[sp]);
  }
  // no longer need the temporary storages for names and nats.
  myfree(ion_names[0]);
  myfree(ion_nats);
  myfree(ion_names);


  /****************************************************************
   *                                                              *
   *  Second pass:                                                *
   *                                                              *
   * keywords:                                                    *
   *  ion-species, kpoint                                             *
   *                                                              *
   ****************************************************************/
  // Prepare for a second pass.
  dft_text_rewind(filep);
  sp = 0; kpt = 0;

  // Keep reading till end of file skipping empty lines and comments
  while (dft_text_fgets(line,DFT_LINE_LEN,'\\',filep) != NULL) {
    if (line[0]=='#' || sscanf(line,"%s",key) <= 0)
      continue;

    /* case 1, ions: (second pass) */
    /* case 1.1, Look for an ion-species line for each species needed. */
    if ( MATCH(key,"ion-species") ) {
      /* species <identifier> <Z> <mass> <pot-filename> <pulay-filename> <type> \
	      [l-component local-component ngrid_loc dq_loc ngrid_nl dq_nl]
	 e.g.: 
	    species Si 4 28.0 si.pot none mit 
	    species Si 4 28.0 si.cpi none fhi d d 0.02 
      */
      found = DFT_FALSE;
      sp = 0;
      sscanf(line,"%*s %s",key);
      while ( (!found) && (sp < ioninfo.nspecies) ) {
	if ( MATCH(key, ioninfo.species[sp].name) ) {
	  // found the species this is for
	  found = DFT_TRUE;
	  if ( !sp_found[sp] ) {
	    sp_found[sp] = DFT_TRUE;
	    Speciesinfo *spec = &ioninfo.species[sp];
	    
	    sscanf(line,"%*s %*s %lg %lg %s %s %s",
		   &(spec->Z), &(spec->mass), spec->potfile, spec->pulayfile, key);
	    
	    // read the potential file and pulay file.
	    if ( MATCH(key,"fhi") ) {
	      int ngrid_loc = 2004, ngrid_nl = 501;
	      real dq_loc = 0.02, dq_nl = 0.02;
	      char lmax = 'd', lloc = 'd';
	      /*
	      if (elecinfo.Ecut > 0) {	// set max G according to Ecut.
	        // not yet implemented.
	      }
	      */
	      sscanf(line,"%*s %*s %*g %*g %*s %*s %*s %c %c %d %lg %d %lg",
		     &lmax, &lloc, &ngrid_loc, &dq_loc, &ngrid_nl, &dq_nl);
	      read_fhi_psp(spec, spec->potfile, lmax, lloc,
			   ngrid_loc, dq_loc, ngrid_nl, dq_nl);
	    } else { 
	      // assuming it's mit format by default
	      read_dft_psp(spec, spec->potfile);
	    }
	    
	    // do pulay correction stuffs.
	    spec->setup_pulay(spec->pulayfile,elecinfo.Ecut);
	  } else {
	    // this is not the first species line for this species.
	    // what should we do?  ignore? or die?
	    dft_log(DFT_SILENCE,
		    ">> %s", line);
	    dft_log(DFT_SILENCE,
		    ">>>> This is the second time I read about species %s\n",
		    ioninfo.species[sp].name);
	    dft_log(DFT_SILENCE,
		    ">>>> What are you trying to do? Confuse me?\n");
	    die(">>>> Quitting...\n");
	  }
	} else {
	  sp++;  // if not this species, check if it is for the next one.
	}
      }
      
      // if not found, this species is not needed. 
      
      // if we want to be very stringent, we could die here. 
      // or we can do what we do now: be lenient and just ignore extra species lines.
    }
    
    
    /* case 1.2, get all the ion positions */
    else if ( MATCH(key,"ion") ) {
      vector3 v;
      char move = 'n';
      int move_flag = DFT_FALSE;
      found = DFT_FALSE;
      // ion name [ x1 y1 z1 movable relative-coord-flag ]
      sscanf(line,"%*s %s %lg %lg %lg %c", 
	     key, &v.v[0], &v.v[1], &v.v[2], &move);
      sp = 0;
      while ( (!found) && (sp < ioninfo.nspecies) ) {
	if ( MATCH(key, ioninfo.species[sp].name) ) {
	  found = DFT_TRUE;
	  // shift ion position components to [0.0 1.0]
	  for (i = 0; i < 3; i++) {
	    v.v[i] = fmod(v.v[i], 1.0);
	    if (v.v[i] < 0.0)
	      v.v[i] += 1.0;
	  }
	  if ( match_yes(&move) )
	    move_flag = DFT_TRUE;
	  ioninfo.species[sp].atpos[ioninfo.species[sp].natoms] = v;
	  ioninfo.species[sp].move[ioninfo.species[sp].natoms] = move_flag;
	  ioninfo.species[sp].natoms++;
	} else {
	  sp++;
	}
      }
      if (!found) {
	dft_log(DFT_SILENCE,
		"Something strange happened.\n");
	dft_log(DFT_SILENCE,
		"Species %s was not detected first but here it is...\n",
		key);
	die("Fatal programming error.\n");
      }
    }
    
    
    /* case 2, kpoints: (second pass) */
    else if ( MATCH(key,"kpoint") ) {
      // kpoint <kx> <ky> <kz> <weight>
      if (kpt < elecinfo.nkpts) {
	sscanf(line,"%*s %lg %lg %lg %lg",
	       &(kvec[kpt].v[0]),
	       &(kvec[kpt].v[1]),
	       &(kvec[kpt].v[2]),
	       &(w[kpt]));
	kpt++;
      } else {
	die("Too many kpoints!!");
      }
    }
  }


  /****************************************************************
   *                                                              *
   *  Second pass post-processing:                                *
   *                                                              *
   ****************************************************************/

  // check that all species has pseudopotential read.
  for (sp = 0, found = DFT_TRUE; sp < ioninfo.nspecies; sp++)
    {
      if ( !sp_found[sp] )
	{
	  dft_log(DFT_SILENCE,
		  ">>>> No species info for %s\n",ioninfo.species[sp].name);
	  found = DFT_FALSE;
	}
    }
  if (!found)
    die("Exiting.\n");
  else
    myfree(sp_found);    // free temporary spaces:

  // check ion positions in case atoms are positioned on top of each other.
  ioninfo.check_pos();



  /****************************************************************
   *                                                              *
   *  Third and further passes:                                   *
   *                                                              *
   * If new keywords are introduced, they can be put here:        *
   * (uncomment the block)                                        *
   *                                                              *
   ****************************************************************/
  dft_text_rewind(filep); 
  while (dft_text_fgets(line,DFT_LINE_LEN,'\\',filep) != NULL) {
    if (line[0] == '#')
      continue;

    // matching keywords
    if (sscanf(line,"%s",key) > 0) {  // skip empty lines.
      
      /* case 1, Tairan 9/28/1999, calculate partial charge density */
      if ( MATCH(key,"orbital_density") ) {
        params.number_of_partial_n_lines++;
      }

      /* case 2, Tairan 10/8/1999, reading parameters for relaxation */
      else if (MATCH(key, "ion-relax")) {
	sscanf(line,"%*s %s", key);
	for (sp = 0; sp < ioninfo.nspecies; sp++) {
	  if ( MATCH(key, ioninfo.species[sp].name) ) {
	    sscanf(line,"%*s %*s %lg %lg", &ioninfo.species[sp].ion_damp, 
		   &ioninfo.species[sp].ion_fac); 
	  }
	}
      }

      /* case 3, Maxim 11/1/1999, reading for a line constraint */
      else if (MATCH(key, "constraint_line")) {
	ioninfo.number_of_line_constraints++;
      }

      /* case 4, Tairan 11/6/1999, suppress template printing in runs */
      else if (MATCH(key, "template")) {
	if (sscanf(line,"%*s %s",key) > 0) {
	  if ( match_no(key) )
	    params.template_printing = DFT_FALSE;
	}
      }

      /*
      if ( MATCH(key,"something") ) {
	// do something to something
      }

      else if ( MATCH(key,"something else") ) {
	// do something to something else
      }

      else if ( MATCH(key,"something else") ) {
	// do something to something else
      }*/

    }
  }


  params.partial_n_kpt = (int*)mymalloc(sizeof(int)*params.number_of_partial_n_lines,
					"params.partial_n_kpt","setup");
  params.partial_n_band = (int*)mymalloc(sizeof(int)*params.number_of_partial_n_lines,
					 "params.partial_n_band","setup");

  ioninfo.line_constraint_direction = 
    (vector3*)mymalloc(sizeof(vector3)*ioninfo.number_of_line_constraints,
		       "ioninfo.line_constraint_direction","setup");
  ioninfo.line_constraint_forces = 
    (vector3*)mymalloc(sizeof(vector3)*ioninfo.number_of_line_constraints,
		       "ioninfo.line_constraint_direction","setup");
  ioninfo.line_constraint_n_sp = 
    (int*)mymalloc(sizeof(int)*ioninfo.number_of_line_constraints,
		   "ioninfo.line_constraint_n_sp","setup");
  ioninfo.line_constraint_n_at = 
    (int*)mymalloc(sizeof(int)*ioninfo.number_of_line_constraints,
		   "ioninfo.line_constraint_n_at","setup");

  /* another pass to read in partial n dumping specifications */
  dft_text_rewind(filep); 

  int iorb = 0, ic = 0; 
  real norma;

  while (dft_text_fgets(line,DFT_LINE_LEN,'\\',filep) != NULL) {
    if (line[0] == '#')
      continue;

    // matching keywords
    if (sscanf(line,"%s",key) > 0) {  // skip empty lines.
      
      /* case 1, Tairan 9/28/1999, calculate partial charge density */
      if ( MATCH(key,"orbital_density") ) {
	sscanf(line,"%*s %d %d",
	       params.partial_n_kpt+iorb,
	       params.partial_n_band+iorb);
	dft_log("partial n dump for kpoint %d, band %d\n",
		params.partial_n_kpt[iorb], params.partial_n_band[iorb]);
	iorb++;
      }
      /* case 3, Maxim 11/1/1999, reading for a line constraint in ionic relaxation*/
      else if ( MATCH(key,"constraint_line") ) {
	sscanf(line,"%*s %lg %lg %lg %d %d",
	       &(ioninfo.line_constraint_direction[ic].v[0]),
	       &(ioninfo.line_constraint_direction[ic].v[1]),
	       &(ioninfo.line_constraint_direction[ic].v[2]),
               ioninfo.line_constraint_n_sp+ic,
               ioninfo.line_constraint_n_at+ic);
        norma=abs(ioninfo.line_constraint_direction[ic]);
	ioninfo.line_constraint_direction[ic]=ioninfo.line_constraint_direction[ic]/norma;
	ic++;
      }
    }
  }


  /****************************************************************
   *                                                              *
   *  Overall post-processing:                                    *
   *                                                              *
   ****************************************************************/

  // All the parameters has been read.  So we can do whatever
  // setup remains.




  /*-- Setup the lattice vectors and basis first part --*/

  // if the basis flag is not read, complain and die.
  if ( (basis_flag != 0) && (basis_flag != 1) ) {
    die("You must specify the basis keyword:\nbasis [single | kpoint-dependent]\nQuitting.\n");
  }
  // scale the lattice;
  for (i=0; i<3; i++)
    for (j=0; j<3; j++) {
      latvec.m[j][i] *= latscale[i];
    }
  // make a temporary basis for lattice stuffs. 
  // it's temporary since we don't have the final kpoints yet.
  tmp_basis.latvec = latvec;
  tmp_basis.init(elecinfo.Ecut,nfft[0],nfft[1],nfft[2]);





  /*-- Setup the electronic state information --*/

  // count the number of electrons
  for (sp = 0; sp < ioninfo.nspecies; sp++)
    elecinfo.nelectrons += ioninfo.species[sp].Z * ioninfo.species[sp].natoms;
  // Figure out the the number of bands based on the number of electrons
  // and the number of empty bands.
  elecinfo.nbands = (int)ceil(elecinfo.nelectrons * 0.5);
  elecinfo.nbands += nemptybands;





  /*-- Setup symmetry part --*/

  // if ion-dynamics, disable symmetry calculation
  if ( params.ionic_dynamics_flag ) {
    // disable symmetry calculation.
    calc_symm_flag = DFT_FALSE;
  }
  if (ioninfo.nrot > 0) { 
    // symmetries are put in manually
    // check if they atomic postions agree with symmetries
    check_symmetries(&ioninfo);
    // allocate temporary space for done[].
    ioninfo.done = (int*)mymalloc(sizeof(int)*tmp_basis.NxNyNz,"ioninfo->done",
				  "symmetries");
  }
  else if (calc_symm_flag) {
    // search automatically for symmetries of the system
    symmetries(&ioninfo, &tmp_basis);
    // allocate temporary space for done[].
    ioninfo.done = (int*)mymalloc(sizeof(int)*tmp_basis.NxNyNz,"ioninfo->done",
				  "symmetries");
  }  
  else {
    // no symmetry is used. Only Identity is there.
    ioninfo.nrot = 1;
    ioninfo.done = NULL;
    ioninfo.sym[0].m[0][0] = ioninfo.sym[0].m[1][1] =
      ioninfo.sym[0].m[2][2] = 1.0;
  }
  /*
   * Check that the FFT box is commensurate with the symmetries.
   */
  check_symm_fftbox(ioninfo, tmp_basis);

  /*
   * Fold the kpoints, and reduce depending on symmetries 
   * (time-reversal-symmetry is always assumed in the reduction!
   * Careful with spin dependent cases).
   */
  // check first that number of kpoints is nonzero.
  if (elecinfo.nkpts < 1) {
    die("You must specify at least one kpoint.\n");
  }
  fold_kpoints(&kvec,&w,kpt_fold,elecinfo.nkpts);
  reduce_kpoints(kvec,w,ioninfo,elecinfo,tmp_basis,calc_symm_flag);
  // kpoint reduction finished. kvec and w are no longer needed.
  myfree(kvec);  myfree(w);





  /*-- Setup basis 2nd part (completion) --*/

  /*
   * Now the kpoints are finalized.
   */
  *basis = (Basis *)mymalloc(sizeof(Basis)*(elecinfo.nkpts+1),"basis","setup");
  real G2max = 0.0,
       avg_nbasis = 0.0,
       ideal_nbasis = pow(sqrt(2*elecinfo.Ecut),3)*
                             (tmp_basis.unit_cell_volume/(6*M_PI*M_PI));
  int nkpts = elecinfo.nkpts;
  if (basis_flag == 0) {
    // note that if basis_flag is 0, only one set
    // of index arrays will be created based on Gamma point.
    dft_log("basis: using a single basis centered at Gamma.\n");
    vector3 zero(0.0,0.0,0.0);
    (*basis)[0].init2(tmp_basis,zero,elecinfo.Ecut,G2max);
    avg_nbasis = (*basis)[0].nbasis;
    for (k = 1; k < nkpts; k++) {
      (*basis)[k].init2((*basis)[0]);
    }
  } else {
    dft_log("basis: using kpoint-dependent basis sets.\n");
    for (k = 0; k < nkpts; k++) {
      (*basis)[k].init2(tmp_basis,elecinfo.kvec[k],elecinfo.Ecut,G2max);         
      avg_nbasis += elecinfo.w[k] * (*basis)[k].nbasis;
    }
  }
  // the last basis is for charge density FFT, set index arrays to null.
  (*basis)[nkpts].init2(tmp_basis,elecinfo.kvec[0],0.0,	G2max);
  dft_log("average nbasis = %4.2f , ideal nbasis = %6.3f\n",
	    avg_nbasis, ideal_nbasis);
  dft_log("\n0.5*|G|^2 = %lg for highest energy plane-wave\n\n",
	    0.5*G2max);
  dft_log_flush();




  /*-- Finish setting up elecinfo by setting up the initial fillings --*/
  elecinfo.setup_initial_fillings(*basis);

  // Set non-linear core correction flag in Elecinfo if one or more species have core correction.
  sp = 0; 
  do {
    if ( ioninfo.species[sp].nl_core_flag )
      elecinfo.nl_core_flag = DFT_TRUE;
    sp++;
  } while ( (! elecinfo.nl_core_flag) && (sp < ioninfo.nspecies) );
  
  // Print out all the information inside of elecinfo
  elecinfo.print();





  /*-- Now finish up the dynamics setup --*/
  iondyninfo.basis = &(*basis)[elecinfo.nkpts];
  if ( params.ionic_dynamics_flag ) {

    if ( MATCH(iondyninfo.ion_dyn_action,"new") ) {
      // first time start up dynamics run. Set up ion dynamics.
      iondyninfo.ioninit();

    } else if ( MATCH(iondyninfo.ion_dyn_action,"continue") ) {
      // it's a continuing run. Read the last ion coordinate 
      // in addition to the current ones. 
      iondyninfo.ioncontinue(&elecinfo);

    } else {
      die("ion_dyn_action must be 'new' or 'continue'.\n\n");
    }

    // check positions for overlap.
    iondyninfo.check_pos();

    // Initialize Rotator file pointers for disk-cached wavefunctions for extrapolateion in ion dynamics.
    iondyninfo.oldFileRead  = params.init_Y_filename;
    iondyninfo.oldFileWrite = params.old_Y_filename;

  }

  /*-- Now finish up the dynamics setup --*/
  if ( params.ionic_relaxation_flag ) {
    dft_log("relaxing the ions:\n");
    // ion_start_iter = 0;
    // mean_speed = mean_speed_old = 0.0;
    int sp;
    for (sp=0; sp < ioninfo.nspecies; sp++) {
      int i;
      for (i=0; i < ioninfo.species[sp].natoms; i++) {
	ioninfo.species[sp].ion_speed[i].v[0] = 0.0;
	ioninfo.species[sp].ion_speed[i].v[1] = 0.0;
	ioninfo.species[sp].ion_speed[i].v[2] = 0.0;
      }
      if (ioninfo.species[sp].ion_damp == 0.0)
	ioninfo.species[sp].ion_damp = 0.7;
      if (ioninfo.species[sp].ion_fac == 0.0)
	ioninfo.species[sp].ion_fac = 0.004;
      dft_log("specie %d with ion_damp = %g and ion_fac = %g\n", sp, 
	      ioninfo.species[sp].ion_damp, ioninfo.species[sp].ion_fac);		
    }
  }



  /*-- Setup Ewald calculation --*/
  setup_Ewald(Nlat_start_real, Nlat_end_real,
	      Nlat_start_recip, Nlat_end_recip);






  /*-- Setup the FFT3D() routines --*/
  setupFFT3D(tmp_basis.Nx,
	     tmp_basis.Ny,
	     tmp_basis.Nz);



  // done with setup!
}




//
// Parse command line and exit, print help, and
// setup the input and log filenames appropriately
//
void
parse_command_line(int argc, char **argv, 
		   char *input_filename,
		   char *log_filename)
{
  // Logs to stdout.  We need this to report
  // basic errors and problems on the command line.
  // Make the system logging point to it for now.
  Output helplog;
  System::global_log = &helplog;

  for (int k = 1; k < argc; k++) {
    if ( MATCHn( argv[k],"-h",2 ) ) {
      help();
      System::GlobalFinalize();
      exit(0);
//  }  else if ( MATCHn( argv[k],"-r",2 ) ) {
//       // restart program, next argument is binary-restart-file
//       /* not yet implemented. */
//       die(">> Restart not yet implemented.\n>> Terminating.\n");
    } else if ( MATCHn( argv[k],"-t",2 ) ) {
      // produce a template input file.
      Parameters templateparams;
      Elecinfo templateelecinfo;
      Basis templatebasis;
      IonDyninfo templateiondyninfo;
      Control templatecntrl;
      print_template(templateparams,templateelecinfo,&templatebasis,
		     templateiondyninfo,templatecntrl);
      System::GlobalFinalize();
      exit(0);
    } else if ( MATCHn( argv[k],"-o",2 ) ) {
      // override default output file.
      k++;
      strcpy(log_filename, argv[k]);
    } else if ( MATCHn( argv[k],"-i",2 ) ) {
      // override default input file.
      k++;
      strcpy(input_filename, argv[k]);
    } else {
      // unknown option, skip?
      dft_log(">> unknown option: %s.\n",argv[k]);
      die(">> Terminating.\n");
    }
  }    
}

/*
 * prepare_wavefunctions()
 *
 * Reading the wavefunctions if instructed to do so.
 *
 */
int
prepare_wavefunctions(Parameters &params,
		      Elecinfo &elecinfo, Elecvars &elecvars,
		      IonDyninfo &iondyninfo)
{
  int k;
  FILE *testfilep;

  if ( ! params.read_Y_flag )
    {
      dft_log("\n-------> Setting Y to random values\n\n");
      System::seed_with_time();
      randomize_column_bundle_array(elecinfo.nkpts,elecvars.Y);
      /* Orthonormalize the Y */
      calc_UVC(&elecinfo,&elecvars);
      for (k=0; k < elecinfo.nkpts; k++)
	elecvars.Y[k] = elecvars.C[k];
    }
  else
    {
      // If this is the processor that can do IO, it can
      // check whether the Y-file can be opened
      if (System::Get_procID() == System::Get_IOprocID())
	{
	  // Just see if the file is there...
	  if ((testfilep = fopen(params.init_Y_filename,"r"))==(FILE *)0)
	    die("\nCan't open '%s' to read initial wave-functions.\n\n",
		params.init_Y_filename);
	  fclose(testfilep);
	}

      // We're ok to read it!
      dft_log("\n-------> Reading C from '%s'\n\n", params.init_Y_filename);
      dft_log_flush();
      read_column_bundle_array(params.init_Y_filename,
			       elecinfo.nkpts,
			       elecvars.Y);
    }

  if ( params.ionic_dynamics_flag &&
       ( MATCH(iondyninfo.ion_dyn_action,"continue") ) ) {
    // this is a ionic dynamics continued run,
    // so read the  old wave-function and extrapolate from it
    dft_log("\n-------> Reading old C from '%s'\n\n",params.old_Y_filename);
    dft_log_flush();
    iondyninfo.extrapolate_column_bundle_array(params.old_Y_filename,
					       elecinfo.nkpts,
					       elecvars.Y,
					       elecvars.Y);
  }

  return 0;
}


/*
 * print_template
 *
 * Print out a template list,
 * that could also be used as template for future runs.
 *
 */
void
print_template(Parameters &params,
	       const Elecinfo &elecinfo, Basis *basis,
	       const IonDyninfo &iondyninfo,
	       const Control &cntrl)
{
  int i, sp, nat, k;
  Speciesinfo *spec;
  const Ioninfo &ioninfo = iondyninfo.ioninfo;

  if (!params.template_printing) return;

  dft_log("#>>>>>>>>>>>>>> Template Begins <<<<<<<<<<<<<<<<<<<\n"); 

  dft_log("########################################################\n");
  dft_log("## Input file\n##\n## use '\\' for line continuation\n");
  dft_log("########################################################\n");
  dft_log("\n");
  dft_log("# The unit of length is the Bohr radius.  Energies in Hartrees\n");
  dft_log("\n");

  // print job lines.
  dft_log("# Jobs specify what the program should do.  Most are\n");
  dft_log("# independent from each other and can be specified\n");
  dft_log("# simultaneouly.  Each 'job' line can only list one action.\n");
  dft_log("# The current choices are: 'electronic-minimization',\n");
  dft_log("# 'calculate-forces', 'ionic-relaxtion', and \n");
  dft_log("# 'ionic-dynamics'\n");
  if ( !params.electronic_minimization_flag )
    dft_log("# ");
  dft_log("job electronic-minimization\n");
  if ( !params.calculate_forces_flag ) 
    dft_log("# ");
  dft_log("job calculate-forces\n");
  if ( !params.ionic_dynamics_flag )
    dft_log("# ");
  dft_log("job ionic-dynamics\n");
  if ( !params.ionic_relaxation_flag )
    dft_log("# ");
  dft_log("job ionic-relaxation\n");
  dft_log("\n\n");

  // print lattice lines.
  dft_log("# The lattice vectors are specified below:\n");
  dft_log("# a1 = 1st column, a2 = 2nd column, a3 = 3rd column\n");
  dft_log("lattice\t\\\n");
  for (i=0; i < 3; i++)
    {
      dft_log("\t%20.15le %20.15le %20.15le  ",
	      basis[elecinfo.nkpts].latvec.m[i][0],
	      basis[elecinfo.nkpts].latvec.m[i][1],
	      basis[elecinfo.nkpts].latvec.m[i][2] );
      if (i<2)
	dft_log("\\");
      dft_log("\n");
    }
  dft_log("#\n# Possible scaling of the lattice vectors\n");
  dft_log("latt-scale 1.0 1.0 1.0\n");
  dft_log("#\n# Lattice symmetries:\n");
  dft_log("# either none are done (latt-nosymmetries),\n");
  dft_log("# or symmetries can be specified manually (latt-symmetry) via\n");
  dft_log("# latt-symmetry <x11> <x12> ... <x33>  (symmetry matrix)\n");
  dft_log("# or they are done automatically if nothing is specified.\n");
  if (ioninfo.nrot <= 1) {
    dft_log("latt-nosymmetries\n");
  } else {
    dft_log("# latt-nosymmetries\n");
    dft_log("\n# %d symmetries follow\n", ioninfo.nrot);
    for (int irot = 0; irot < ioninfo.nrot; irot++){
      dft_log("\nlatt-symmetry ");
      for (i=0; i < 3; i++) 
	dft_log("\t\\\n\t%2g %2g %2g", ioninfo.sym[irot].m[i][0],
		  ioninfo.sym[irot].m[i][1], ioninfo.sym[irot].m[i][2] );
    }
    dft_log("\n");
  }
  dft_log("\n\n");

  // print ion lines.
  dft_log("# Each species of ions is specified via:\n");
  dft_log("#\n");
  dft_log("# ion-species <name> <Z> <mass> <potfile> <pulay_file>\n");
  dft_log("#\n");
  dft_log("# where <name> is a string giving a name to the species,\n");
  dft_log("# <Z> is the valence charge (units of proton charges),\n");
  dft_log("# <mass> is the mass in amu (used only for dynamics),\n");
  dft_log("# <potfile> is a DFT++ .pot file describing the species,\n");
  dft_log("# and <pulay_file> is the pulay corrections file ('none' as\n");
  dft_log("# the pulay_file is ok: it just means no pulay corrections).\n");
  dft_log("#\n");
  dft_log("# Ionic positions are specified via\n");
  dft_log("#\n");
  dft_log("# ion <name> <x> <y> <z> <t/f | 1/0>\n");
  dft_log("#\n");
  dft_log("# where <name> is the name specified in the ion-species line,\n");
  dft_log("# x y z are the lattice position of the ion, and t/f or 1/0\n");
  dft_log("# specifies whether the ion may be moved or not (in dynamics or\n");
  dft_log("# relaxation).\n");
  for (sp=0; sp < ioninfo.nspecies; sp++) {
    spec = &ioninfo.species[sp];
    dft_log("ion-species %s %g %g %s %s\n",
	      spec->name, spec->Z, spec->mass,
	      spec->potfile, spec->pulayfile);
    for (nat = 0; nat < spec->natoms; nat++) {
      dft_log("ion %s %15.11lf %15.11lf %15.11lf %d\n",
		spec->name, 
		spec->atpos[nat].v[0],
		spec->atpos[nat].v[1],
		spec->atpos[nat].v[2],
		spec->move[nat]);
    }
  }
  dft_log("\n\n");

  // print electronic lines
  dft_log("# Planewave cutoff energy in Hartrees\n");
  dft_log("elec-cutoff %g\n", elecinfo.Ecut);
  dft_log("#\n# Number of extra 'empty' bands to add at each k-point atop\n");
  dft_log("# whatever is needed to for the valence charge of all ions.\n");
  dft_log("elec-n-empty-bands %d\n",
	  elecinfo.nbands - (int)ceil(elecinfo.nelectrons*0.5) );
  dft_log("#\n# Whether fermi fillings are calculated or not:\n");
  dft_log("# elec-fermi-fillings <niter_recalc> <kT> [eV | K]\n");
  dft_log("# will cause calculation of fermi fillings every <niter_recalc>\n");
  dft_log("# electronic steps at the tempertaure <kT> in Hartree (unless\n");
  dft_log("# the string 'eV' or 'K' is added, in which case those units\n");
  dft_log("# are used with some conversion factors).\n");
  if (elecinfo.calc_fillings)
    dft_log("elec-fermi-fillings %d %g\n",
	      elecinfo.niter_recalc_fillings, elecinfo.kT);
  dft_log("#\n");
  dft_log("# Treatement of electronic correlations:  LDA or GGA for now.\n");
  if (elecinfo.ex_opt != DFT_EXCORR_GGA)
    dft_log("# ");
  dft_log("elec-ex-corr-gga\n");
  if (elecinfo.ex_opt != DFT_EXCORR_LDA)
    dft_log("# ");
  dft_log("elec-ex-corr lda\n");
  dft_log("#\n");
  dft_log("# How the initial fillings are generated:\n");
  dft_log("# either they are calculated automatically (fillings are 2\n");
  dft_log("# and 0 to match number of electrons, or they are read from\n");
  dft_log("# a file.\n");
  if (elecinfo.initial_fillings_flag == DFT_TRUE) {
    dft_log("elec-initial-fillings read %s\n",
	    elecinfo.initial_fillings_filename);
    dft_log("# elec-initial-fillings automatic\n");
  }
  else {
    dft_log("# elec-initial-fillings read <fillings-filename>\n");
    dft_log("elec-initial-fillings automatic\n");
  }
  dft_log("\n\n");

  // print k-point lines
  dft_log("# k-points are specified via:\n");
  dft_log("# k-point <kx> <ky> <kz> <w>\n");
  dft_log("# where kx ky kz are the k-vector component in latt. units\n");
  dft_log("# and w is the weight.\n");
  for (k = 0; k < elecinfo.nkpts; k++)
    dft_log("kpoint %15.11lf %15.11lf %15.11lf %16.11le\n",
	    elecinfo.kvec[k].v[0],
	    elecinfo.kvec[k].v[1],
	    elecinfo.kvec[k].v[2],
	    elecinfo.w[k]);
  dft_log("#\n");
  dft_log("# Possible folding of k-points: >1 increases their number\n");
  dft_log("kpoint-folding 1 1 1\n");
  dft_log("\n\n");

  // print basis line
  dft_log("# Type of basis set used: either all k-points have the same\n");
  dft_log("# set of plane waves ('single'), or each has its own\n");
  dft_log("# ('k-point-dependent').\n");
  
  if ( elecinfo.nkpts > 1 && basis[1].index==basis[0].index ) {
    dft_log("# basis k-point-dependent\n");
    dft_log("basis single\n");
  } else {
    dft_log("# basis single\n");
    dft_log("basis kpoint-dependent\n");
  }
  dft_log("\n\n");

  // print control lines
  dft_log("# Various control parameters during minimization.\n");
  dft_log("#\n");
  dft_log("# Algorithm used for electronic minimization: choices are\n");
  dft_log("# 'CG', 'CG_nocos', 'PCG', 'PCG_nocos', 'EOM', and 'PSD'.\n");
  dft_log("cntrl-algorithm ");
  switch (cntrl.algorithm) {
  case DFT_CG        : dft_log("CG\n"); break;
  case DFT_CG_nocos  : dft_log("CG_nocos\n"); break;
  case DFT_PCG       : dft_log("PCG\n"); break;
  case DFT_PCG_nocos : dft_log("PCG_nocos\n"); break;
  case DFT_EOM       : dft_log("EOM\n"); break;
  case DFT_PSD       : dft_log("PSD\n"); break;
  default : dft_log("unknown-future-technology\n"); break;
  }
  dft_log("#\n");
  dft_log("# Subspace rotations can be turned on:  this makes\n");
  dft_log("# metallic calculations work MUCH MUCH better (i.e. the\n");
  dft_log("# results converge reasonably).  Costs a bit more memory/time\n");
  dft_log("# and is useless for insulating type fillings.\n");
  if (elecinfo.subspace_rotation)
    dft_log("cntrl-subspace-rotation %g\n",elecinfo.pcond_for_Bgrad);
  else
    dft_log("# cntrl-subspace-rotation [<pcond>]\n");
  dft_log("#\n");
  dft_log("# Maximum number of electronic minimization steps.\n");
  dft_log("cntrl-max-elec-steps %d\n", cntrl.max_elecs_steps);
  dft_log("#\n");
  dft_log("# The minimum size of energy changes below which\n");
  dft_log("# the minimization routine quit working (Hartrees).  If zero,\n");
  dft_log("# then it will do the maximum number of iterations.\n");
  dft_log("cntrl-energy-convergence %g\n",cntrl.E_tolerance);
  dft_log("#\n");
  dft_log("# Initial stepsize for electronic line-minimization\n");
  dft_log("cntrl-elec-stepsize %g\n", cntrl.elec_stepsize);
  dft_log("#\n");
  dft_log("# Whether the last best stepsize is used for the next linmin\n");
  if (cntrl.update_stepsize)
    dft_log("cntrl-update-stepsize\n");
  else 
    dft_log("# cntrl-update-stepsize\n");
  dft_log("#\n");
  dft_log("# The minimum size of the stepsize allowed before the\n");
  dft_log("# electronic minimization loop quits.  Zero is allowed.\n");
  dft_log("cntrl-stepsize-min %g\n",cntrl.stepsize_min);
  dft_log("#\n");
  dft_log("# To what tolerance the number of electrons is converged\n");
  dft_log("# when the fillings are being recaculated.\n");
  dft_log("cntrl-nel-fermi-tol %g\n",cntrl.nel_fermi_tol);
  dft_log("\n\n");

  // print wavefunction lines
  dft_log("# Initial wavefunctions:  if no 'wavefunction' line is given\n");
  dft_log("# the initial wavefunctions are random.  Otherwise they\n");
  dft_log("# are read from a file:\n");
  dft_log("# wavefunction <filename> [<filename2>]\n");
  dft_log("# where <filename> has the intial (binary) wavefunctions to be\n");
  dft_log("# read, and the optional <filaname2> for ionic dynamics.\n");
  if ( params.read_Y_flag)
    dft_log("wavefunction %s %s\n",
	      params.init_Y_filename, params.old_Y_filename);
  dft_log("\n\n");

  // ewald lines
  int ew[4];
  dft_log("# Size of Ewald sums:\n");
  dft_log("# ewald <i1> <i2> <i3> <i4>\n");
  dft_log("# Real space sums done with indices going from <i1> to <i2>\n");
  dft_log("# Recip. space sums done with indices going from <i3> to <i4>\n");
  if (get_ewald(ew[0], ew[1], ew[2], ew[3]))
    dft_log("ewald %d %d %d %d\n",
	      ew[0], ew[1], ew[2], ew[3]);
  dft_log("\n\n");

  // print fftbox line
  dft_log("# FFT box could be automatically calculated.\n");
  dft_log("# If you want to manually specify the values, uncomment the following:\n");
  dft_log("# fftbox <n1> <n2> <n3>\n\n");

  // print other lines
  dft_log("# Ionic dynamics parameters: \n");
  dft_log("# velocity rescaling flag, temperature, stepsize in femtosec\n");
  dft_log("# new or continue [iter]\n");
  if (!iondyninfo.velsc_flag)
    dft_log("# ");
  dft_log("ion-dynamics velsc\n");
  dft_log("ion-dynamics kT %lg\n", iondyninfo.ionic_kT);
  dft_log("ion-dynamics kT %lg\n", iondyninfo.ionic_stepsize);
  dft_log("ion-dynamics ");
  if ( MATCH(iondyninfo.ion_dyn_action,"new") ) {
    dft_log("new\n");
  } else {
    dft_log("continue %d\n",iondyninfo.ion_start_iter);
  }

  dft_log("\n# dumping partial charge density from bands\n");
  dft_log("# orbital_density [kpoint index] [band index]\n");
  for (i = 0; i < params.number_of_partial_n_lines; i++)
    dft_log("orbital_density %d %d\n",
	    params.partial_n_kpt[i], params.partial_n_band[i]);
  
  dft_log("\n# Line constraints on atoms in relaxations\n");
  dft_log("# constraint_line [lx] [ly] [lz] [species number] [atom number]\n");
  for (i = 0; i < ioninfo.number_of_line_constraints; i++)
    dft_log("constraint_line %15.11lf %15.11lf %15.11lf %d %d\n",
	    ioninfo.line_constraint_direction[i].v[0],
	    ioninfo.line_constraint_direction[i].v[1],
	    ioninfo.line_constraint_direction[i].v[2],
            ioninfo.line_constraint_n_sp[i], 
	    ioninfo.line_constraint_n_at[i]);

  dft_log("\n#>>>>>>>>>>>>>>> Template Ends <<<<<<<<<<<<<<<<<<<<\n"); 
}
