/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Jan 12, 1999    Tairan Wang
 *
 * Added support for using FHI type pseudopotential;
 * The pseudopotential is  stored as the 
 *    real space pseudo-radial-wavefunction and real space pseudopontential
 *   on a logarithmic spaced radial grid.
 * The interpolation works in each subroutine are replaced with 
 * corresponding fourier transforms.
 *
 * The FHI pseudopotential already has the -4*pi*Z*e^2/k^2 built in.
 */

/*
 * locpseudopot.c:   Sohrab Ismail-Beigi
 *                        Written Aug. 1996, modified Jan. 97, Feb. 97
 *
 * Routines that give the atomic pseudopotential as a function of r or k,
 * and Vpseudogrid() finds the crystal pseudopotential on a specified grid.
 * There are various convergence approximations...  see below
 * for details.
 *
 * complex structfact()
 * void Vpseudogrid()
 */

/* $Id: locpseudopot.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include "header.h"


/*
 * S(G) = sum_[j=1,natoms] { exp(-i*2*pi*G.r_j) }:  the infamous
 * structure factor. Here G is in reciprocal lattice coords and hence
 * given by integers; the atomic positions are in lattice coordinates.
 */
complex
structfact(int natoms,vector3 atpos[],int G[3])
{
  complex S;
  register int i,j;
  register real dot;
  const real twopi = (real)2.0*M_PI;

  S.x = S.y = (real)0.0;
  for (i=0; i < natoms; i++)
    {
      dot = (real)0.0;
      for (j=0; j < 3; j++)
	dot -= atpos[i].v[j]*G[j];
      dot *= twopi;
      S.x += cos(dot);
      S.y += sin(dot);
    }
  return S;
}


/* Calculates the local pseudopotential in G-space on a parallelopiped
 * grid Nx by Ny by Nz in each direction.
 * The result is put in Vp in z then y then x order.
 * The Nx,Ny,Nz values are in the basis structure, as are the lattice
 * vectors.
 *
 * The pseudopotential in G-space, Vp(G), is given in terms of the 
 * local pseudopot. in r-space Vp(r) as given below for a single species:
 *
 *   Vp(G) = integral_{unit cell} { d^3r Vp(r)*exp(-i*G*r)/sqrt(Vol) }
 *         = S(G)*Va(G)/sqrt(Vol)
 *
 * Here Vp(r) = sum_{R,tau} { Va(r-R-tau) }
 * R is a lattice vector and tau is an atomic position in the unit cell.
 * Va(r) is the atomic pseudopotential given above and Va(G) is its
 * continuous Fourier transform.  Vol is the unit cell volume.
 *
 * ---> Vp(G) HAS sqrt(Vol) INSTEAD OF Vol AS WOULD BE EXPECTED <----
 * ---> FOR THE DISCRETE FFT OF Vp(r)                           <----
 *
 * A G-vector G_n = n*G where n is a row vector in Z^3 and G is
 * G = 2*pi*(latvec)^(-1) where latvec is that 3x3 matrix of primitive
 * lattice vectors contained in the columns.
 *
 * Cutoff issues:
 * 
 * We keep n components within a box satisfying -N_i/2 < n_i < N_i/2.
 * The points +/-N_i/2 would cause aliasing problems in the FFT, so we
 * keep them out.
 *
 * G=0 issues:
 *
 * Va(G=0) is infinite, so we must deal with G=0 in some way.
 * Here we choose to set Vp(G=0) = 0...the G=0 energy associated with
 * the difference of the pseudopotential and the coulomb potential is
 * calculated below.
 *
 */
void
Vloc_pseudo(Basis *basis,
	    Ioninfo *ioninfo,
	    complex *Vp)
{
  complex SG;
  real G,G2,G2max,s=0.0,svol,r;
  register real ym1,y0,y1,y2,a,b,c,x;
  int beta[3],i,j,index,n[3],Nx,Ny,Nz,sp,nmax=0;

  /* Constants */
  const real fourpi = (real)4.0*M_PI;
  const real sixth = (real)1.0/(real)6.0;

  /* Copy Nx, Ny, Nz from basis */
  Nx = basis->Nx;
  Ny = basis->Ny;
  Nz = basis->Nz;

  /* Calculate 1/sqrt(unitcellvol) */
  svol = (real)1.0/sqrt(basis->unit_cell_volume);

  dft_log("\n----- Vloc_pseudo() -----\n");
  dft_log("Calculating local pseudopotential with");
  dft_log("grid size Nx=%d, Ny=%d, Nz=%d and\n",Nx,Ny,Nz);
  dft_log("cell_volume=%lg\n",basis->unit_cell_volume);
  dft_log("latvec = \n");
  basis->latvec.print(dft_global_log,"%lg ");
  dft_log("Setting Vpseudo(G=0) = 0\n");
  dft_log("\n");
  dft_log_flush();

  /* Zero out pseudopotential */
  for (i=0; i < basis->NxNyNz; i++)
    Vp[i].x = Vp[i].y = 0.0;
  G2max = (real)0.0;

  /* Calculate Vp in G-space:  the n sums run over the points we keep in
   * the grid (here the FFT G-lattice).  */
  /* First, loop over the different atomic species */
  for (sp = 0; sp < ioninfo->nspecies; sp++)
    {
      Speciesinfo *sp_info = &(ioninfo->species[sp]);

      /* Optional report */
      dft_log("species = %d   Z = %lg   natoms = %d\n",
		sp, sp_info->Z, sp_info->natoms);
      for (i=0; i < sp_info->natoms; i++)
	dft_log(DFT_ANAL_LOG,"atpos[%d]=(%10lg,%10lg,%10lg)\n",i,
		sp_info->atpos[i].v[0],
		sp_info->atpos[i].v[1],
		sp_info->atpos[i].v[2]);
      dft_log("\n");
      dft_log_flush();

      nmax = sp_info->ngrid_loc-3;

      /* Loop over FFT box and add in contribution from current species */
      for (n[0] = -Nx/2+1; n[0] < Nx/2; n[0]++)
	for (n[1] = -Ny/2+1; n[1] < Ny/2; n[1]++)
	  for (n[2] = -Nz/2+1; n[2] < Nz/2; n[2]++)
	    {
	      if (n[0] < 0) beta[0] = n[0]+Nx;
	      else          beta[0] = n[0];
	      if (n[1] < 0) beta[1] = n[1]+Ny;
	      else          beta[1] = n[1];
	      if (n[2] < 0) beta[2] = n[2]+Nz;
	      else          beta[2] = n[2];
	      
	      index = beta[2]+Nz*(beta[1]+Ny*beta[0]);
	      
	      G2 = (real)0.0;
	      for (i=0; i < 3; i++)
		for (j=0; j < 3; j++)
		  G2 += (real)(n[i]*n[j])*basis->GGT.m[i][j];
	      if (G2 > G2max)
		G2max = G2;
	      
	      /* G != 0 */
	      if (n[0] != 0 || n[1] != 0 || n[2] != 0)
		{
		  /* Calculate structure factor */
		  SG = structfact(sp_info->natoms,
				  sp_info->atpos,n);

		  /* Interpolate pseudopotential using internal tables */
		  /* 3rd order fit with j-1, j, j+1, and j+2 points */
		  G = sqrt(G2);
		  r = G/sp_info->dq_loc;
		  j = (int)r;
		  if (j > nmax)
		    die("Local pseudopotential grid is too small!\n");
		  if (j < 1)
		    die("Sampling local pseudopot. too close to 0!\n");
		  ym1 = sp_info->V_loc[j-1];
		  y0  = sp_info->V_loc[j  ];
		  y1  = sp_info->V_loc[j+1];
		  y2  = sp_info->V_loc[j+2];
		  a = sixth*(y2-ym1+3.0*(y0-y1));
		  b = 0.5*(y1+ym1-y0-y0);
		  c = y1-sixth*(y2+3.0*y0+ym1+ym1);
		  x = r-j;
		  s = ((a*x+b)*x+c)*x+y0;

		  /* This line below is quadratic fitting:      */
		  /* s = y0 + 0.5*x*(y1-ym1+x*(y1+ym1-y0-y0));  */

		  /* Add in -4*pi*Z*e^2/G^2 which is needed */
		  s -= fourpi*sp_info->Z/G2;

		  /* Multiply by inverse sqrt(volume) and accumulate */
		  s *= svol;
		  Vp[index].x += SG.x*s;
		  Vp[index].y += SG.y*s;
		}

	      dft_log(DFT_ANAL_LOG,
		      "n=(%3d,%3d,%3d)  G2 = %10.6lf   Vp=%15le%+15lei\n",
		      n[0],n[1],n[2],G2,Vp[index].x,Vp[index].y);
	      dft_log(DFT_ANAL_LOG,
		      "SG = %15le%+15lei\n",SG.x,SG.y);

	    } /* loop on FFT box */

    } /* loop on species */

  dft_log("Maximum 0.5*|G|^2 = %lg\n\n",0.5*G2max);
  dft_log_flush();

}

/*
 * Fills in dVj with the derivative of Vloc_pseudo versus the j'th coordinate
 * of atom 'atom' of species 'species'.  Used for calculating forces
 * on the atoms.
 */
void
dVloc_pseudo_datom_pos(Basis *basis,Ioninfo *ioninfo,
		       int species,int atom,
		       complex *dVx,complex *dVy,complex *dVz)
{
  real G,G2,vatomic=0.0,svol,r;
  register real ym1,y0,y1,y2,a,b,c,x;
  int beta[3],i,j,index,n[3],Nx,Ny,Nz,nmax=0;
  register real phase,taux,tauy,tauz;
  register complex Stau;
  real invdq=0.0;
  Speciesinfo *sp_info = &(ioninfo->species[species]);

  /* Constants */
  const real twopi = (real)2.0*M_PI;
  const real fourpi = (real)4.0*M_PI;
  const real sixth = (real)1.0/(real)6.0;

  /* Copy Nx, Ny, Nz from basis */
  Nx = basis->Nx;
  Ny = basis->Ny;
  Nz = basis->Nz;

  /* Calculate 1/sqrt(unitcellvol) */
  svol = (real)1.0/sqrt(basis->unit_cell_volume);

  /* Zero out the three output arrays */
  for (i=0; i < Nx*Ny*Nz; i++)
    dVx[i].x = dVx[i].y = 
      dVy[i].x = dVy[i].y =
      dVz[i].x = dVz[i].y = 0.0;

  nmax = sp_info->ngrid_loc-3;
  invdq = (real)1.0/sp_info->dq_loc;

  /* Loop over FFT box */
  for (n[0] = -Nx/2+1; n[0] < Nx/2; n[0]++)
    for (n[1] = -Ny/2+1; n[1] < Ny/2; n[1]++)
      for (n[2] = -Nz/2+1; n[2] < Nz/2; n[2]++)
	{
	  if (n[0] < 0) beta[0] = n[0]+Nx;
	  else          beta[0] = n[0];
	  if (n[1] < 0) beta[1] = n[1]+Ny;
	  else          beta[1] = n[1];
	  if (n[2] < 0) beta[2] = n[2]+Nz;
	  else          beta[2] = n[2];
	  
	  index = beta[2]+Nz*(beta[1]+Ny*beta[0]);
	      
	  G2 = (real)0.0;
	  for (i=0; i < 3; i++)
	    for (j=0; j < 3; j++)
	      G2 += (real)(n[i]*n[j])*basis->GGT.m[i][j];
	      
	  /* G != 0 */
	  if (n[0] != 0 || n[1] != 0 || n[2] != 0)
	    {
	      /* Interpolate pseudopotential using internal tables */
	      /* 3rd order fit with j-1, j, j+1, and j+2 points */
	      G = sqrt(G2);
	      r = G*invdq;
	      j = (int)r;
	      if (j > nmax)
		die("Local pseudopotential grid is too small!\n");
	      if (j < 1)
		die("Sampling local pseudopot. too close to 0!\n");
	      ym1 = sp_info->V_loc[j-1];
	      y0  = sp_info->V_loc[j  ];
	      y1  = sp_info->V_loc[j+1];
	      y2  = sp_info->V_loc[j+2];
	      a = sixth*(y2-ym1+3.0*(y0-y1));
	      b = 0.5*(y1+ym1-y0-y0);
	      c = y1-sixth*(y2+3.0*y0+ym1+ym1);
	      x = r-j;
	      vatomic = ((a*x+b)*x+c)*x+y0;
	      
	      /* This line below is quadratic fitting:      */
	      /* s = y0 + 0.5*x*(y1-ym1+x*(y1+ym1-y0-y0));  */

	      /* Add in -4*pi*Z*e^2/k^2 which is needed */
	      vatomic -= fourpi*sp_info->Z/G2;

	      /* Get position of the atom in question and calculate
	       * Stau = exp(-2*pi*i*dot(G,tau)) */
	      taux = sp_info->atpos[atom].v[0];
	      tauy = sp_info->atpos[atom].v[1];
	      tauz = sp_info->atpos[atom].v[2];
	      phase = -twopi*(n[0]*taux + n[1]*tauy + n[2]*tauz);
	      Stau.x = cos(phase);
	      Stau.y = sin(phase);

	      /* dV_j = -2*pi*i*n[j]*vatomic/sqrt(Vol)*Stau */
	      dVx[index].x =  n[0]*twopi*vatomic*svol*Stau.y;
	      dVx[index].y = -n[0]*twopi*vatomic*svol*Stau.x;

	      dVy[index].x =  n[1]*twopi*vatomic*svol*Stau.y;
	      dVy[index].y = -n[1]*twopi*vatomic*svol*Stau.x;

	      dVz[index].x =  n[2]*twopi*vatomic*svol*Stau.y;
	      dVz[index].y = -n[2]*twopi*vatomic*svol*Stau.x;
	    }
	} /* loop on FFT box */

}

/*
 * Same as above, except Va(|G|), the atomic pseudopotential in
 * G-space, is replaced by its derivative versus |G|.  The derivatives
 * are the exact analytical derivatives of the interpolation algorithm
 * in the above code.
 */
void
Vlocprime_pseudo(Basis *basis,
		 Ioninfo *ioninfo,
		 complex *Vpprime)
{
  complex SG;
  real G=0.0,G2,G2max,s=0.0,svol,r;
  register real ym1,y0,y1,y2,a,b,c,x,invdq;
  int beta[3],i,j,index,n[3],Nx,Ny,Nz,sp,nmax=0;

  /* Constants */
  const real eightpi = (real)8.0*M_PI;
  const real sixth = (real)1.0/(real)6.0;

  /* Copy Nx, Ny, Nz from basis */
  Nx = basis->Nx;
  Ny = basis->Ny;
  Nz = basis->Nz;

  /* Calculate 1/sqrt(unitcellvol) */
  svol = (real)1.0/sqrt(basis->unit_cell_volume);

  dft_log("\n----- Vlocprime_pseudo() -----\n");
  dft_log("Calculating deriv of local pseudopotential with");
  dft_log("grid size Nx=%d, Ny=%d, Nz=%d and\n",Nx,Ny,Nz);
  dft_log("cell_volume=%lg\n",basis->unit_cell_volume);
  dft_log("latvec = \n");
  basis->latvec.print(dft_global_log,"%lg ");
  dft_log("Setting Vpseudo'(G=0) = 0\n");
  dft_log("\n");
  dft_log_flush();

  /* Zero out pseudopotential */
  for (i=0; i < basis->NxNyNz; i++)
    Vpprime[i].x = Vpprime[i].y = 0.0;
  G2max = (real)0.0;

  /* Calculate Vp in G-space:  the n sums run over the points we keep in
   * the grid (here the FFT G-lattice).  */
  /* First, loop over the different atomic species */
  for (sp = 0; sp < ioninfo->nspecies; sp++)
    {
      Speciesinfo *sp_info = &(ioninfo->species[sp]);

      /* Optional report */
      dft_log("species = %d   Z = %lg   natoms = %d\n",
		sp,sp_info->Z,sp_info->natoms);
      for (i=0; i < sp_info->natoms; i++)
	dft_log(DFT_ANAL_LOG,
		"atpos[%d]=(%10lg,%10lg,%10lg)\n",i,
		sp_info->atpos[i].v[0],
		sp_info->atpos[i].v[1],
		sp_info->atpos[i].v[2]);
      dft_log("\n");
      dft_log_flush();

      nmax = sp_info->ngrid_loc-3;
      /* Loop over FFT box and add in contribution from current species */
      for (n[0] = -Nx/2+1; n[0] < Nx/2; n[0]++)
	for (n[1] = -Ny/2+1; n[1] < Ny/2; n[1]++)
	  for (n[2] = -Nz/2+1; n[2] < Nz/2; n[2]++)
	    {
	      if (n[0] < 0) beta[0] = n[0]+Nx;
	      else          beta[0] = n[0];
	      if (n[1] < 0) beta[1] = n[1]+Ny;
	      else          beta[1] = n[1];
	      if (n[2] < 0) beta[2] = n[2]+Nz;
	      else          beta[2] = n[2];
	      
	      index = beta[2]+Nz*(beta[1]+Ny*beta[0]);
	      
	      G2 = (real)0.0;
	      for (i=0; i < 3; i++)
		for (j=0; j < 3; j++)
		  G2 += (real)(n[i]*n[j])*basis->GGT.m[i][j];
	      if (G2 > G2max)
		G2max = G2;
	      
	      /* G != 0 */
	      if (n[0] != 0 || n[1] != 0 || n[2] != 0)
		{
		  /* Calculate structure factor */
		  SG = structfact(sp_info->natoms,
				  sp_info->atpos,n);

		  /* Interpolate pseudopotential using internal tables */
		  /* 3rd order fit with j-1, j, j+1, and j+2 points */
		  G = sqrt(G2);
		  invdq = (real)1.0/sp_info->dq_loc;
		  r = G*invdq;
		  j = (int)r;
		  if (j > nmax)
		    die("Local pseudopotential grid is too small!\n");
		  if (j < 1)
		    die("Sampling local pseudopot. too close to 0!\n");
		  ym1 = sp_info->V_loc[j-1];
		  y0  = sp_info->V_loc[j  ];
		  y1  = sp_info->V_loc[j+1];
		  y2  = sp_info->V_loc[j+2];
		  a = sixth*(y2-ym1+3.0*(y0-y1));
		  b = 0.5*(y1+ym1-y0-y0);
		  c = y1-sixth*(y2+3.0*y0+ym1+ym1);
		  x = r-j;
		  s = ((3.0*a*x+2.0*b)*x+c)*invdq;

		  /* Add in 8*pi*Z*e^2/k^3 (deriv of coulomb) */
		  s += eightpi*sp_info->Z/(G2*G);

		  /* Multiply by inverse sqrt(volume) and accumulate */
		  s *= svol;
		  Vpprime[index].x += SG.x*s;
		  Vpprime[index].y += SG.y*s;

		}
	      dft_log(DFT_ANAL_LOG,
		      "n=(%3d,%3d,%3d)  G2 = %10.6lf   Vp'=%15le%+15lei\n",
		      n[0],n[1],n[2],G2,Vpprime[index].x,Vpprime[index].y);
	      dft_log(DFT_ANAL_LOG,
		      "SG = %15le%+15lei\n",SG.x,SG.y);
	    } /* loop on FFT box */

    } /* loop on species */

  dft_log("Maximum 0.5*|G|^2 = %lg\n\n",0.5*G2max);

}

/*
 * The G=0 contribution to the energy for the local pseudopotentials.
 * For each species, this is equal to
 *
 * integral_{unit cell} { d^3r n(r)*(Vp(r)-Vc(r))|G=0 }
 *
 * where n(r) is the number density of the electrons, Vp(r) is the
 * pseudopotential (periodic), and Vc(r) is the potential due to
 * a constant density of positive charge equal to the total ionic charges;
 * i.e., natoms*Z.  The |G=0 denotes that we only need the G=0 part of the
 * discrete fourier expansion of Vp(r)-Vc(r).
 * 
 * Looking at the above routine's comments, the discrete fourier transform
 * of Vp(r) is Vp(G) = S(G)*Va(G)/Vol, where Vol=unit cell volume.
 * Similarly, Vc(G) = -S(G)*4*pi*Z*e^2/(G^2*Vol).
 * Since S(0) = natoms = nions,
 *
 * (Vp(r)-Vc(r))|G=0 = (natoms*lim_{G->0} {Va(G)+4*pi*Z*e^2/G^2})/Vol,
 * which is a constant.  The integral of n(r) then gives nelectrons
 * and the final result is nelectrons*(Vp(r)-Vc(r))|G=0.
 *
 * report <= 0: silent
 * report >  0: says what it is doing
 *
 */
real
Vloc_pseudoGzeroEnergy(real nelectrons,Basis *basis,Ioninfo *ioninfo)
{
  real e,etot;
  int sp;

  dft_log("\n----- VpseudoGzeroEnergy() -----\n");
  dft_log("Nelectrons = %lg    unit cell volume = %lg\n",
	    nelectrons,basis->unit_cell_volume);

  e = etot = 0.0;
  for (sp = 0; sp < ioninfo->nspecies; sp++)
    {
      Speciesinfo *sp_info = &(ioninfo->species[sp]);

      e = (real)(nelectrons* sp_info->natoms)*
	sp_info->V_loc[0]/basis->unit_cell_volume;

      dft_log("species = %d   Vloc[0] = %19.12le   e=%19.12le\n",
	      sp, sp_info->V_loc[0], e);

      etot += e;
    }
  dft_log("\nCore energy = %19.12le\n",etot);
  
  return etot;
}
