/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * header.h
 *
 */

/* $Id: header.h,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#ifndef DFT_HEADER_H
#define DFT_HEADER_H

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <math.h>
#include <stdarg.h>

/*=======================================================================*
 *                                                                       *
 *  Forward declarations of classes in this file                         *
 *                                                                       *
 *=======================================================================*/
class Output;             // Generic class to allow text output to a file
class System;             // take cares of system init./finalize, and 
                          // parallelization information
class Basis;              // both unitcell geometry and G indices.
class vector;             // vector of scalars, also used for FFT-box
class diag_matrix;        // mainly for filling matrix.
class matrix;             // general purpose matrix.
class scalar_column;      // single column of scalars,
                          //  used to make up column_bundle.
class column_bundle;      // to store wavefunctions
class Speciesinfo;        // information for ion species.
class Ioninfo;            // collections of all Speciesinfo,
                          // also houses symmetry info.
class IonDyninfo;         // extension of Ioninfo for dynamics information.
class Elecinfo;           // parameters for electrons.
class Elecvars;           // collection of electronic variables.
class Energies;           // collection of all energies.
class Control;            // contains control parameters for the run.
class Parameters;         // contains parameters from setup().
class Lattice_forces;     // contains forces on the lattice vectors.


/*=========================================================================*
 *                                                                         *
 *  Constant definition section                                            *
 *                                                                         *
 *=========================================================================*/

// Type of algorithms for electronic minimization
#define DFT_CG                 0
#define DFT_CG_nocos           1
#define DFT_PCG                2
#define DFT_PCG_nocos          3
#define DFT_EOM                4
#define DFT_PSD                5


// string constants
#define DFT_LINE_LEN           2024
#define DFT_MSG_LEN            256
#define DFT_FILENAME_LEN       256

// logical constants
#define DFT_TRUE                1
#define DFT_FALSE               0

// types of exchange correlation functionals
#define DFT_EXCORR_LDA        0
#define DFT_EXCORR_GGA        1

// type of data:  for indicating potentially parallel objects
// DFT_LARGE_DATA  will be parallel objects on parallel platform.
#define DFT_SMALL_DATA     0
#define DFT_LARGE_DATA     1


/* Report levels for logging */
#define DFT_SILENCE   0  // don't log anything
#define DFT_BASIC_LOG 1  // print the basics
#define DFT_ANAL_LOG  2  // print lots and lots!
#define DFT_NERD_LOG  3  // only if you're seriously interested...

/* Short-hand for logging:  we don't want to type that ugly
 * long string every time we want to log! */
#define dft_log         System::global_log->printf
#define dft_log_flush   System::global_log->flush
#define dft_global_log  System::global_log


/* program parameters. maybe put them into parameter.h ? */

#define MIN_ION_DISTANCE 1e-10
#define MIN_KPT_DISTANCE 1e-8
#define MIN_SYMM_TOL     1e-4

/*
 * There are several compiler flags that can be chosen:
 *
 * DFT_MPI
 *       : Turn on MPI
 *
 * DFT_PROFILING
 *       : Turn on some profiling timing and allocation counters.
 * 
 * DFT_MEMALIGN
 *       : Forces memory allocation to align on 32-byte-boundaries.
 *       : This option is known to cause some memory-leak problems on Origin 2000
 *
 * DFT_TRACE_MEM
 *       : Turn on memory allocation traces. For memory debugging purpose.
 */


/*=======================================================================*
 *                                                                       *
 *  Definition of the scalar: either real of complex                     *
 *                                                                       *
 *=======================================================================*/

typedef double real;

/* Depending on whether our scalars are complex or real, include
 * one XOR the other of the lines below */

/*
#define SCALAR_IS_REAL
typedef real scalar;
struct complex_template { real x,y; };
typedef struct complex_template complex;
#define REAL(x) (x)
*/

#define SCALAR_IS_COMPLEX
#include "complex.h"
typedef complex scalar;
#define REAL(z) ((z).x)


// the size of scalar variables in multiples of sizeof(double)
#ifdef SCALAR_IS_COMPLEX
#define SCALAR_SIZE 2
#elif SCALAR_IS_REAL
#define SCALAR_SIZE 1
#else
#error Scalar is neither real nor complex!
#endif


// define two shorthand string compare macros:
#define MATCH(a,b) (strcmp(a,b)==0)
#define MATCHn(a,b,c) (strncmp(a,b,c)==0)


/*-----------------------------------------------------------------------*
 *  Output class:  takes care of text output to a file                   *
 *                 (mainly used for logging).                            *
 *-----------------------------------------------------------------------*/
#include "Output.h"

/*=======================================================================*
 *                                                                       *
 *  Include 3D vector and matrix classes and routines                    *
 *                                                                       *
 *=======================================================================*/
#include "lin3.h"

/*-----------------------------------------------------------------------*
 *  Professor Arias file suite for IO class and operations               *
 *  it also defines the dft_text_FILE class                              *
 *-----------------------------------------------------------------------*/
#include "dft_text_FILE.h"

/*-----------------------------------------------------------------------*
 *  System class: takes care of parallelization information              *
 *-----------------------------------------------------------------------*/
#include "System.h"

/*-----------------------------------------------------------------------*
 *  Control class: for convergence of minimization, etc.                 *
 *-----------------------------------------------------------------------*/
#include "Control.h"

/*-----------------------------------------------------------------------*
 *  mem.c:  memory allocation routines and a general die() function      *
 *          that exits with a message.  All memory allocations/deallocs  *
 *          should be done with these routines.                          *
 *-----------------------------------------------------------------------*/
#include "mem.h"

/*-----------------------------------------------------------------------*
 *  utils.c:  various utilities                                          *
 *-----------------------------------------------------------------------*/
#include "utils.h"

/*-----------------------------------------------------------------------*
 *  help.c:   provides help information to command line                  *
 *            and prints a template of what was used to run the program  *
 *-----------------------------------------------------------------------*/
#include "help.h"

/*-----------------------------------------------------------------------*
 *  timer.c:  provides timer functions for timing various operations     *
 *-----------------------------------------------------------------------*/
#include "timer.h"

/*-----------------------------------------------------------------------*
 *  Basisc class: holds information about the basis functions used       *
 *-----------------------------------------------------------------------*/
#include "Basis.h"

/*-----------------------------------------------------------------------*
 *  vector class: a collection of scalars (used also for the FFT box)    *
 *-----------------------------------------------------------------------*/
#include "vector.h"

/*-----------------------------------------------------------------------*
 *  diag_matrix class: a diagonal matrix of scalar elements              *
 *-----------------------------------------------------------------------*/
#include "diag_matrix.h"

/*-----------------------------------------------------------------------*
 *  matrix class: general purpose matrix whose elements are scalars      *
 *-----------------------------------------------------------------------*/
#include "matrix.h"

/*-----------------------------------------------------------------------*
 *  column_bundle class: a collection of vectors composed of scalars     *
 *                       each vector is a scalar_column class.           *
 *                       used to store wave-functions                    *
 *-----------------------------------------------------------------------*/
#include "column_bundle.h"

/*-----------------------------------------------------------------------*
 *  dist_multiply.c: does various matrix multiplies for column_bundle    *
 *                   classes                                             *
 *-----------------------------------------------------------------------*/
#include "dist_multiply.h"

/*-----------------------------------------------------------------------*
 *  Speciesinfo class: contains information describing an ionic species  *
 *                     and its pseudopoentials.  Togther with Ioninfo    *
 *                     defines the ions and pseudopotentials             *
 *-----------------------------------------------------------------------*/
#include "Speciesinfo.h"

/*-----------------------------------------------------------------------*
 *  Ioninfo class: Master structure containing info on all ions          *
 *  (Member functions defined in setup_ions.c)                           *
 *-----------------------------------------------------------------------*/
#include "Ioninfo.h"

/*-----------------------------------------------------------------------*
 *  IonDyninfo class: extended Ioninfo class to include dynamics info    *
 *-----------------------------------------------------------------------*/
#include "IonDyninfo.h"

/*-----------------------------------------------------------------------*
 *  Elecinfo class: contains info for describing electronic states       *
 *-----------------------------------------------------------------------*/
#include "Elecinfo.h"

/*-----------------------------------------------------------------------*
 *  Elecvars class: contains the electronic variables                    *
 *-----------------------------------------------------------------------*/
#include "Elecvars.h"

/*-----------------------------------------------------------------------*
 *  Energies class: contains all the energy terms for our system         *
 *-----------------------------------------------------------------------*/
#include "Energies.h"

/*-----------------------------------------------------------------------*
 *  Parameters class: contains the parameters the main program has read  *
 *                    from the input file (in setup()) that              *
 *                    define the run.                                    *
 *-----------------------------------------------------------------------*/
#include "Parameters.h"

/*-----------------------------------------------------------------------*
 *  Lattice_forces class: holds forces on lattice vectors                *
 *-----------------------------------------------------------------------*/
#include "Lattice_forces.h"

/*-----------------------------------------------------------------------*
 *  PHLO.c: Implement operators:  P, Hsp, L, O, Obar, etc.               *
 *-----------------------------------------------------------------------*/
#include "PHLO.h"

/*-----------------------------------------------------------------------*
 *  IJ.c: I, J, Idag, and Jdag operators                                 *
 *-----------------------------------------------------------------------*/
#include "IJ.h"

/*-----------------------------------------------------------------------*
 *  diaginnerouter.c: find charge density                                *
 *-----------------------------------------------------------------------*/
#include "diaginnerouter.h"

/*-----------------------------------------------------------------------*
 *  ft.c: FFT3D routine (which calls FFTW package)                       *
 *-----------------------------------------------------------------------*/
#include "ft.h"

/*-----------------------------------------------------------------------*
 *  locpseudopot.c: routines to calculate local pseudopotentials         *
 *-----------------------------------------------------------------------*/
#include "locpseudopot.h"

/*-----------------------------------------------------------------------*
 *  nlpseudopot.h:  functions to caluclate nonlocal pseudopotentials     *
 *-----------------------------------------------------------------------*/
#include "nlpseudopot.h"

/*-----------------------------------------------------------------------*
 *  setup.h:  functions to read input files and put the                  *
 *            data into the appropriate data structures                  *
 *-----------------------------------------------------------------------*/
#include "setup.h"

/*-----------------------------------------------------------------------*
 *  setup_basis.h: set up the Basis structure(s)                         *
 *-----------------------------------------------------------------------*/
#include "setup_basis.h"

/*-----------------------------------------------------------------------*
 *  exc.c: exchange-correlation energy densities and deriv. versus n     *
 *-----------------------------------------------------------------------*/
#include "exc.h"

/*-----------------------------------------------------------------------*
 *  ewald.c: Ewald energies and forces                                   *
 *-----------------------------------------------------------------------*/
#include "ewald.h"

/*-----------------------------------------------------------------------*
 *  signals.c: signal handling (dead for MPI versions...???)             *
 *-----------------------------------------------------------------------*/
#include "signals.h"

/*-----------------------------------------------------------------------*
 *  dump.c: Dump the electronic variables with date/time stamps          *
 *-----------------------------------------------------------------------*/
#include "dump.h"

/*-----------------------------------------------------------------------*
 *  setup_ions.c: read ionic info from file and setup the data structs   *
 *                for local and non-local pseudopots                     *
 *-----------------------------------------------------------------------*/
#include "setup_ions.h"

/*-----------------------------------------------------------------------*
 *  setup_elecs.c: read electronic info from files and setup data        *
 *                 structs (fillings, weights, cutoffs, etc) and         *
 *                 allocate/free the space                               *
 *-----------------------------------------------------------------------*/
#include "setup_elecs.h"

/*-----------------------------------------------------------------------*
 *  calcUVCn.c: calc. U, C, and density n (and rot. matrix V)           *
 *-----------------------------------------------------------------------*/
#include "calcUVCn.h"

/*-----------------------------------------------------------------------*
 *  poisson.c: solve Poisson equation                                    *
 *-----------------------------------------------------------------------*/
#include "poisson.h"

/*-----------------------------------------------------------------------*
 *  calcener.c: calculates all the various energy terms in various       *
 *              combinations.                                            *
 *-----------------------------------------------------------------------*/
#include "calcener.h"

/*-----------------------------------------------------------------------*
 *  calcelecgrad.c: calc. energy gradient versus elec. variables         *
 *-----------------------------------------------------------------------*/
#include "calcelecgrad.h"

/*-----------------------------------------------------------------------*
 *  minimize.c: minimizes energy versus elec. vars (CG, PCG, PSD, etc.)  *
 *-----------------------------------------------------------------------*/
#include "minimize.h"

/*-----------------------------------------------------------------------*
 *  rnd.c: defines a random number generator using the system rand()     *
 *         function.  Also uses this rnd() to create gauss()             *
 *-----------------------------------------------------------------------*/
#include "rnd.h"

/*-----------------------------------------------------------------------*
 *  calcionicforces.c: self explanatory!                                 *
 *-----------------------------------------------------------------------*/
#include "calcionicforces.h"

/*-----------------------------------------------------------------------*
 *  calclatforces.c: calculate forces on lattice vectors                 *
 *-----------------------------------------------------------------------*/
#include "calclatforces.h"

/*-----------------------------------------------------------------------*
 *  calcempties.c: calculates eigenenergies (and states) for empty bands *
 *                 via CG minimization                                   *
 *-----------------------------------------------------------------------*/
#include "calcempties.h"

/*-----------------------------------------------------------------------*
 *  fermifill.c: calc. Fermi-Dirac fillings (via FD distribution)        *
 *-----------------------------------------------------------------------*/
#include "fermifill.h"

/*-----------------------------------------------------------------------*
 *  calcallbands.c: calc. band structure for fixed Hamiltonian           *
 *-----------------------------------------------------------------------*/
#include "calcallbands.h"

/*-----------------------------------------------------------------------*
 *  symm.c: calculates symmetries, fold and reduce k-points, and         *
 *          symmetrize charge density                                    *
 *-----------------------------------------------------------------------*/
#include "symm.h"

/*-----------------------------------------------------------------------*
 *  dft_io.c: centralized i/o routines for our matrix, vector, and       *
 *            column_bundle classes that does serial and/or parallel     *
 *            i/o depending on the case.                                 *
 *-----------------------------------------------------------------------*/
#include "dft_io.h"

/*-----------------------------------------------------------------------*
 *  matrix_mult.c: does all the actual matrix multiplications and the    *
 *                 blockings to get good performance.  This is the place *
 *                 to go for optimizing matrix multiplies.               *
 *-----------------------------------------------------------------------*/
#include "matrix_mult.h"


/*-----------------------------------------------------------------------*
 *  nlcore.c: Implements the nonlinear core correction to Exc, and Vxc   *
 *-----------------------------------------------------------------------*/
#include "nlcore.h"


/*-----------------------------------------------------------------------*
 *  other.c: temporary place for new subroutines.                        *
 *-----------------------------------------------------------------------*/
#include "other.h"

/*-----------------------------------------------------------------------*
 *  Some physical constants.                                             *
 *  The code is done fully in atomic units. Some of the inputs have,     *
 *  however, option to use some other everyday units.                    *
 *  USE WITH CATION!                                                     *
 *-----------------------------------------------------------------------*/
#include "constants.h"

#endif // DFT_HEADER_H
