/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * ft.c:             Sohrab Ismail-Beigi               7/18/96, 4/7/97
 *
 * A set of routines to do discrete Fourier transforms on 1D and 3D boxes.
 *
 * 
 * The routines now use the FFTW package from the MIT LCS written by
 * Steven G. Johnson and Matteo Frigo.
 *
 * void FFT3D() -- Uses FFTW
 */

/* $Id: ft.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>
#include "header.h"
#include <time.h>

/*
 * Include the FFTW header and also define two routines which allow us
 * to make usable duplicates of plans for in-place transforms.
 * Courtesy of Steven G. Johnson.
 */
#include "fftw.h"

/*
 * This is a 3D FFT which uses the FFTW package.
 *
 * The transform performed is:
 *
 * if (sign == -1)
 * data(kx,ky,kz) <- sum_(x,y,z) { exp(-i*k.r))*data(x,y,z) } / (nx*ny*nz)
 *
 * if (sign == 1)
 * data(x,y,z) <- sum_(kx,ky,kz) { exp(i*k.r))*data(kx,ky,kz) }
 *
 * nx, ny, nz:  size of FFT box
 * 
 * FFTW routines do not divide by (nx*ny*nz) in the sign==-1 case, so
 * I do it by hand.
 *
 * The plus_- and minus_plan's are static so that we do the setup only once
 * and on all future calls on FFT boxes of the same size we use those
 * plans.
 *
 * NOTE:  setupFFT3D() must be called before using FFT3D().
 *
 */

/* Plans used by FFT3D...they are created by setupFFT3D() */
static int plan_nx,plan_ny,plan_nz;
static fftwnd_plan plus_plan,minus_plan;
static int FFTWsetupwasdone = 0;

void
setupFFT3D(int nx,int ny,int nz)
{
  plan_nx = nx;
  plan_ny = ny;
  plan_nz = nz;
  dft_log("\nSetting up FFTW... clock() = %lf secs\n",
	  clock()/(real)CLOCKS_PER_SEC);
  dft_log_flush();

  plus_plan  = fftw3d_create_plan(nx,ny,nz, FFTW_BACKWARD, 
				  FFTW_MEASURE|FFTW_IN_PLACE|FFTW_USE_WISDOM);
  minus_plan = fftw3d_create_plan(nx,ny,nz, FFTW_FORWARD,  
				  FFTW_MEASURE|FFTW_IN_PLACE|FFTW_USE_WISDOM);
  /*
  plus_plan  = fftw3d_create_plan(nx,ny,nz, FFTW_BACKWARD, 
				  FFTW_MEASURE|FFTW_IN_PLACE|
				  FFTW_THREADSAFE|FFTW_USE_WISDOM);
  minus_plan = fftw3d_create_plan(nx,ny,nz, FFTW_FORWARD,  
				  FFTW_MEASURE|FFTW_IN_PLACE|
				  FFTW_THREADSAFE|FFTW_USE_WISDOM);
				  */
  dft_log("Done clock() = %lf secs\n\n",
	    clock()/(real)CLOCKS_PER_SEC);
  dft_log_flush();

  if (!plus_plan || !minus_plan)
    die("\nCan't create plans for FFTW in setupFFT3D()\n\n");
  FFTWsetupwasdone = 1;
}

void
FFT3D(int nx,int ny,int nz,complex *data,int sign)
{
  if (!FFTWsetupwasdone)
    die("\n\nFFTW setup was not called!  Call setupFFT3D()!!\n\n");

  if (nx != plan_nx || ny != plan_ny || nz != plan_nz)
    die("\n\nFFT3D() called with sizes different from the static plans!\n\n");

  if (sign == 1)
    {
      fftwnd(plus_plan,1,(FFTW_COMPLEX *)data,1,0,NULL,0,0);
    }
  else if (sign == -1)
    {
      fftwnd(minus_plan,1,(FFTW_COMPLEX *)data,1,0,NULL,0,0);

      const int nxyz = nx*ny*nz;
      const real scale = (real)1.0/(real)(nx*ny*nz);
      int i;

      for (i=0; i < nxyz; i++)
	{
	  data[i].x *= scale;
	  data[i].y *= scale;
	}
    }
}
