/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/* 
 *    Sohrab Ismail-Beigi                    June 12, 1997
 *
 * A set of routines to calculate Fermi-Dirac fillings for the
 * electronic bands.
 *
 */

/* $Id: fermifill.c,v 1.3 1999/11/23 02:26:25 tairan Exp $ */

#include <stdio.h>
#include <math.h>

#include "header.h"
// #include "parallel.h"

/* The Fermi-Dirac distribution */
real
Fermi_Dirac_filling(const real epsilon,const real kT,const real mu)
{
  real ex;
  ex = (epsilon-mu)/kT;
  if (ex < 700)   // prevent floating point number overflow
    return 2.0/(1.0+exp(ex));
  else
    return 0.0;
}     

/*
 * Calculates the number of electrons given kT and mu and the
 * the eigenenergies epsilon[k][band] using the Fermi-Dirac distribution
 */
real Nelecs_fermi(const int nkpts,const int nbands,real *w,
	     real **epsilon,const real kT,const real mu)
{
  int k;
  int band;
  real N;

  N = (real)0.0;
  for (k=0; k < nkpts; k++)
    for (band=0; band < nbands; band++) {
      N += w[k]*Fermi_Dirac_filling(epsilon[k][band],kT,mu);
    }
  return N;
}

/*
 * The function calculates the fillings of the bands at all kpts by adjusting
 * the chemical potential mu (via binary sections) so as to make
 * N(mu) = elecinfo->nelectrons.  Then it calculates the fillings based on 
 * that mu.
 * 
 * kT is the temperature in Hartrees.
 *
 * The eigenenergies must be sorted in ascending order and be placed
 * in elecvars->Hsub_eigs[k][band].
 *
 */
void
calc_fermi_fillings(Elecinfo *elecinfo,Elecvars *elecvars,
		    real kT, Control& cntrl)
{
  int nkpts = elecinfo->nkpts;
  int nbands = elecinfo->nbands;
  int k,band;
  real mu1,mu2,mu = 0.0;
  real N = 0.0;
  const real Nideal = elecinfo->nelectrons;
  real **epsilon = elecvars->Hsub_eigs;
  real *w = elecinfo->w;
  int section;

#define NSECTIONS 50
  
  /* Set mu1 = minimal eigenenergy, mu2=maximal eigenenergy */
  mu1 = epsilon[0][0];
  mu2 = epsilon[0][nbands-1];
  /* 
    for (k=0; k < nkpts; k++)
      for (band=0; band < nbands; band++)
        {
	  if (epsilon[k][band] < mu1) mu1 = epsilon[k][band];
	  if (epsilon[k][band] > mu2) mu2 = epsilon[k][band];
	}
  */
  // assume that bands are ordered.
  for (k=0; k < nkpts; k++) {
    if (epsilon[k][nbands-1] < mu1) 
      mu1 = epsilon[k][nbands-1];
    if (epsilon[k][nbands-1] > mu2) 
      mu2 = epsilon[k][nbands-1];
  }

  dft_log("\n--- calc_fermi_fillings() ---\n");
  dft_log("nkpts = %d    nbands=%d   kT = %le   Nideal=%le\n",
	  nkpts,nbands,kT,Nideal);
  dft_log("min(epsilon)=%le   max(epsilon)=%le\n",mu1,mu2);
  dft_log("k-points and eigenenergies follow:\n\n");
  for (k=0; k < nkpts; k++) {
    dft_log("k[%d] = [ %lg %lg %lg ]\n",k,
	    elecinfo->kvec[k].v[0],
	    elecinfo->kvec[k].v[1],
	    elecinfo->kvec[k].v[2]);
    for (band=0; band < nbands; band++)
      dft_log("%le\t",epsilon[k][band]);
    dft_log("\n\n");
  }
  dft_log_flush();

  /* Do binary sections NSECTIONS times to locate mu so N(mu) = Nideal */
  for (section = 0; section < NSECTIONS; section++) {
    mu = (mu1 + mu2)/2.0;
    N = Nelecs_fermi(nkpts,nbands,w,epsilon,kT,mu);

    dft_log("section=%3d  mu=[%le,%le,%le]  N=%le\n",
	    section,mu1,mu,mu2,N);
    dft_log_flush();
    // if converged, break out
    if (fabs(N - Nideal) < cntrl.nel_fermi_tol) break;
    
    // bisection search:
    if ( N > Nideal ) 
      mu2 = mu;
    else
      mu1 = mu;
  }

  // If we couldn't converge, die (this should never happen
  // as 2^NSECTIONS should reprsent a ridiculously small tolerance...
  if (section == NSECTIONS)
    die("\nCould not find appropriate mu in %d bisections!\nQuitting\n",
	NSECTIONS);
  
  /* Store mu into elecinfo */
  elecinfo->mu = mu;

  /* Print final mu and N */
  dft_log("\nFinal mu=%le  N=%le  N-Nideal=%le\n\n",mu,N,N-Nideal);
  
  /* Calculate the fillings for the mu we've found and print out fillings */
  dft_log("Calculated k-points, weights, and fillings follow:\n\n");
  for (k=0; k < nkpts; k++) {
    dft_log("%15.10lf %15.10lf %15.10lf %15.9le\n",
	    elecinfo->kvec[k].v[0],
	    elecinfo->kvec[k].v[1],
	    elecinfo->kvec[k].v[2],
	    elecinfo->w[k]);
    for (band=0; band < nbands; band++) {
      elecinfo->F[k].c[band] = Fermi_Dirac_filling(epsilon[k][band],kT,mu);
      dft_log("%15.9le ",REAL(elecinfo->F[k].c[band]));
    }
    dft_log("\n#\n");
  }

  dft_log("\n");
  dft_log_flush();
}
