/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 * Sohrab Ismail-Beigi,  May 22
 * 
 * This program converts C files between different basis sets.  Both
 * sets must have the same electronic state information (except
 * the energy cutoff which may be different).  Each set has its own
 * lattice vectors.  It reads in a C-file from set 1 and sends out
 * a C-file for set 2.  Basis elements in one set but not the other
 * are either ignored or have their coefficients set to zero.
 *
 */

// Sohrab Ismail-Beigi Aug 5 1999
// I just rewrote the code to work with the new DFT++ formalism.
// Getting all the input files correct is really crazy... there
// has to be a better way.

/* $Id: changebasis.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>

#include <time.h>
/* My header file */
#include "header.h"
#include "parallel.h"

#define REPORT_LEVEL       DFT_BASIC_LOG


int
main(int argc,char**argv)
{
  System::GlobalInit(&argc,&argv);

#ifdef DFT_PROFILING
  /* Initialize timers */
  timerActivateAll();
  timerOn(0);
#endif // DFT_PROFILING

  Basis *basis1,*basis2;
  IonDyninfo iondyninfo1,iondyninfo2;
  Ioninfo &ioninfo1 = iondyninfo1.ioninfo;
  Ioninfo &ioninfo2 = iondyninfo2.ioninfo;
  Elecinfo elecinfo1,elecinfo2;
  Elecvars elecvars1,elecvars2;
  Control cntrl1,cntrl2;
  Parameters params1,params2;

  // initialize input filenames to defaults.
  char input_filename1[DFT_FILENAME_LEN];
  char input_filename2[DFT_FILENAME_LEN];
  char log_filename[DFT_FILENAME_LEN] = "";

  // Get filenames from comand line
  if (argc==3)
    {
      sscanf(argv[1],"%s",input_filename1);
      sscanf(argv[2],"%s",input_filename2);
    }
  else if (argc==4)
    {
      sscanf(argv[1],"%s",input_filename1);
      sscanf(argv[2],"%s",input_filename2);
      sscanf(argv[3],"%s",log_filename);
    }
  else
    die("\nUsage: %s <inputfile1> <inputfile2> [<logfile>]\n\n");

  // Open the logfile and make the system logger point to it
  Output logfile(REPORT_LEVEL,log_filename);
  System::global_log = &logfile;

  // Open the first input file for information
  dft_text_FILE *inputfile1;
  if ( (inputfile1 = dft_text_fopen(input_filename1,"r")) ==
              (dft_text_FILE *)0 )
      die("\n%s:  can't read '%s'.  Aborting.\n\n",argv[0], input_filename1);
  // expand all includes in the input file.
  inputfile1->expand_include();
  // Print the current time and date
  time_t timenow = time(0);
  dft_log("\n******************************************************\n");
  dft_log("Current date and time: %s\n",ctime(&timenow));
  dft_log("%s:  reading file '%s'\n",argv[0],input_filename1);
  // Do setup with that inpufile
  setup(inputfile1, params1, elecinfo1, &basis1, iondyninfo1, cntrl1);
  dft_text_fclose(inputfile1);
  // Print parameters from first inputfile
  print_params(params1, elecinfo1, basis1, iondyninfo1, cntrl1);
  // Setup and allocate the electronic variables for first input
  init_elecvars(&elecinfo1,basis1,&elecvars1);
  // Prepare wavefunctions for first input
  prepare_wavefunctions(params1, elecinfo1, elecvars1, iondyninfo1);

  // Open the second input file for information
  dft_text_FILE *inputfile2;
  if ( (inputfile2 = dft_text_fopen(input_filename2,"r")) ==
              (dft_text_FILE *)0 )
      die("\n%s:  can't read '%s'.  Aborting.\n\n",argv[0], input_filename2);
  // expand all includes in the input file.
  inputfile2->expand_include();
  // Print the current time and date
  dft_log("\n******************************************************\n");
  dft_log("******************************************************\n");
  dft_log("******************************************************\n");
  dft_log("%s:  reading file '%s'\n",argv[0],input_filename2);
  // Do setup with that inpufile
  setup(inputfile2, params2, elecinfo2, &basis2, iondyninfo2, cntrl2);
  dft_text_fclose(inputfile2);
  // Print parameters from second inputfile
  print_params(params2, elecinfo2, basis2, iondyninfo2, cntrl2);
  // Setup and allocate the electronic variables for second input
  init_elecvars(&elecinfo2,basis2,&elecvars2);
  // Prepare wavefunctions for second input
  prepare_wavefunctions(params2, elecinfo2, elecvars2, iondyninfo2);


#ifdef DFT_PROFILING
  timerOff(0);  // turn off initialization timer.
  timerOn(1);  // turn on total computation timer.
#endif // DFT_PROFILING


  /////////////////////////////////////////////////
  // Do the conversion!
  /////////////////////////////////////////////////

  // First get a table for workspaces
  dft_log("\nAllocating conversion tables\n");
  dft_log_flush();
  if (elecinfo1.nkpts != elecinfo2.nkpts)
    die("\nInputfiles 1 and 2 don't have the same # of kpts!\n\n");
  int **table = (int **)mymalloc(sizeof(int*)*elecinfo1.nkpts,"table","main");
  int k;
  for (k=0; k < elecinfo1.nkpts; k++)
    {
      int bigger_nbasis = (basis1[k].nbasis > basis2[k].nbasis)
	? basis1[k].nbasis : basis2[k].nbasis; 
      table[k] = (int *)mymalloc(sizeof(int)*bigger_nbasis,"table[]","main");
    }

  // table[k][i] will tell us which basis function in basis2 corresponds
  // to function i in basis1.  table[k][i]==-1 means there is no
  // corresponding function.
  dft_log("Filling in the conversion tables\n");
  dft_log_flush();
  int i,j,gx,gy,gz;
  for (k=0; k < elecinfo1.nkpts; k++)
    for (i=0; i < basis1[k].nbasis; i++)
      {
	table[k][i] = -1;
	gx = basis1[k].Gx[i];
	gy = basis1[k].Gy[i];
	gz = basis1[k].Gz[i];
	for (j=0; j < basis2[k].nbasis; j++)
	  if (gx==basis2[k].Gx[j] && gy==basis2[k].Gy[j] && gz==basis2[k].Gz[j])
	    {
	      table[k][i] = j;
	      break;
	    }
      }

  // Go over the k-points and bands and convert the data
  int b;
  dft_log("Please make sure that Y2 (second set) is randomized by now!\n");
  dft_log("Performing conversion from set 1 to set 2\n");
  dft_log_flush();
  for (k=0; k < elecinfo1.nkpts; k++)
    {
      int n1 = elecvars1.Y[k].my_ncols;
      int n2 = elecvars2.Y[k].my_ncols;
      int nmin = ( (n1 < n2) ? n1 : n2 );
      dft_log("Y1 has %d bands and Y2 has %d bands on proc 0, minimum=%d\n",n1,n2,nmin);
      dft_log_flush();
      
      for (b=0; b < nmin; b++)
	{
	  // dft_log("Working on k=%d/%d b=%d/%d\n",
	  //          k,elecinfo.nkpts-1,b,elecinfo.nbands-1);
	  
	  // We've decided NOT to zero out the output band
	  // because it should already be randomized with
	  // small numbers!!! Make sure that when this program
	  // runs, the set 2 of wavefunctions is actually randomized.
	  
	  // 	// zero out the output band
	  // 	for (i=0; i < basis2[k].nbasis; i++)
	  // 	  elecvars2.Y[k].col[b].c[i] = 0.0;
	  
	  // fill in the output band
	  for (i=0; i < basis1[k].nbasis; i++)
	  if (table[k][i]!=-1)
	    elecvars2.Y[k].col[b].c[table[k][i]] = elecvars1.Y[k].col[b].c[i];
	}
    }

  // Now write out the second set of wavefunctions.
  dft_log("Writing out Y of set 2 to file 'Y.out'\n");
  dft_log_flush();
  write_column_bundle_array("Y.out",elecinfo2.nkpts,elecvars2.Y);

  // It's over!
  dft_log("\nDone!\n\n");
  dft_log_flush();

  // Free up all the used memory
  for (k=0; k < elecinfo1.nkpts; k++)
    myfree(table[k]);
  myfree(table);
  free_elecvars(&elecinfo1,&elecvars1);
  free_elecvars(&elecinfo2,&elecvars2);
  free_basis(basis1,elecinfo1.nkpts);
  free_basis(basis2,elecinfo2.nkpts);
  free_elecinfo(&elecinfo1);
  free_elecinfo(&elecinfo2);

  // End the MPI stuff
  System::GlobalFinalize();

  return 0;
}
