/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/
/*
 *             Tairan Wang                      April 7, 1999
 *
 * Output.c -- define the operation of member functions of Output class
 *
 */

/* $Id: Output.c,v 1.2 1999/11/11 02:40:31 tairan Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "header.h"

//
// This defines a class that does output to a file based on a report level.
// This can be used mainly for logging and printing misc. text data
// to files.
//
// One creates such an object with a report_level and filename.  The
// object then opens the file and sets the report level.  The default
// values for these are file=stdout and report_level=DFT_BASIC_LOG.
//
// The file is opened in append mode.
// 
// If the filename=="" then stdout is opened.
//
// The printf(...) member will send data to the file if the report_level is
// above DFT_SILENCE.
//
// The printf(level,...) member function prints when
//                         level <= Output::report_level.
//

/* Constructor/Destructor */
Output::Output()
{
  fp = stdout;
  report_level = DFT_BASIC_LOG;
  strcpy(filename, "stdout");
}

Output::Output(int level)
{
  report_level = level;
  fp = stdout;
  strcpy(filename, "stdout");
}

Output::Output(char *fname)
{
  report_level = DFT_BASIC_LOG;
  // if given a null name
  if (strcmp(fname,"") == 0){
    fp = stdout;
    strcpy(filename, "stdout");
  }
  else{
    fp = dft_fopen(fname,"a");
    strcpy(filename, fname);
  }
}

Output::Output(int level,char *fname)
{
  report_level = level;
  // if given a null name
  if (strcmp(fname,"") == 0){
    fp = stdout;
    strcpy(filename, "stdout");
  }
  else{
    fp = dft_fopen(fname,"a");
    strcpy(filename, fname);
  } 
}

Output::~Output()
{
  if (fp != stdout)
    dft_fclose(fp);
}

// Flush the file
int
Output::flush()
{
  if (report_level > DFT_SILENCE)
    return dft_fflush(fp);
  else
    return 0;
}


/*
 * A customized fprintf,
 * such that it only prints if report is no smaller than DFT_SILENCE
 */
int
Output::printf(const char *format, ...) const
{
  // exit if report level not hight enough
  if (report_level <= DFT_SILENCE)
    return 0;

  /* Variable argument list pointer */
  va_list ap;

  /* Initialize variable argument list to proper point in the stack */
  va_start(ap, format); 

  return dft_vfprintf(fp, (char *)format, ap);
}

int
Output::printf(int level, const char *format, ...) const
{
  if (level > report_level)
    return 0;

  /* Variable argument list pointer */
  va_list ap;

  /* Initialize variable argument list to proper point in the stack */
  va_start(ap, format); 

  return dft_vfprintf(fp, (char *)format, ap);
}


int
Output::vprintf(int level, const char *format, va_list ap) const
{
  if (level > report_level)
    return 0;

  return dft_vfprintf(fp, (char *)format, ap);
}


