/****************************************************************************
 *
 * DFT++:  density functional package developed by
 *         the research group of Prof. Tomas Arias, MIT.
 *
 * Principal author: Sohrab Ismail-Beigi
 *
 * Modifications for MPI version: Kenneth P Esler,
 *                                Sohrab Ismail-Beigi, and
 *                                Tairan Wang.
 *
 * Modifications for LSD version: Jason A Cline
 *
 * Modifications for lattice/Pulay forces: Gabor Csanyi and
 *                                         Sohrab Ismail-Beigi
 *
 * Copyright (C) 1996-1998 The Massachusetts Institute of Technology (MIT).
 *
 ****************************************************************************/

/*
 *             Sohrab Ismail-Beigi               Jan. 1997
 *
 * Routines to do the I, J, Idag, and Jdag operators for a plane-wave basis.
 * 
 * I(p,G) = exp(i*G*r_p)/sqrt(Vol)     in our case, where Vol is the unit
 *                                     cell volume.
 *
 * Looking at the FFT routines in ft.c, we see that if we define
 * FFT(+/-) to be the FFT3D routine called with a +/- sign (and they
 * are clearly inverses), then
 *
 * I = FFT(+)/sqrt(Vol)
 * J = I^(-1) = sqrt(Vol)*FFT(-)
 * Idag = Nx*Ny*Nz*FFT(-)/sqrt(Vol) = (Nx*Ny*Nz)*J/Vol
 * Jdag = sqrt(Vol)*FFT(+)/(Nx*Ny*Nz) = Vol/(Nx*Ny*Nz)*I
 *
 * To make matters even more complicated, note that I/Jdag go from
 * G space to r space while J/Idag go from r to G space.  Since our
 * wavefunction coeficients comprise only a subset of the FFT box,
 * the I,J,Idag,Jdag operators on column_bundles return vectors with
 * different sizes than their inputs!  So, in some cases, we have to
 * copy to bigger FFT boxes before doing FFTs and in some cases we have to
 * have to do an FFT on a temporay space and copy a subset of it.
 *
 */

/* $Id: IJ.c,v 1.1.1.1 1999/11/10 01:30:17 tairan Exp $ */

#include <stdio.h>
#include <math.h>

#include "header.h"

/* The I operator on vectors: for vectors, the r-space and G-space size
 * are the same, namely the size of the FFT box. */
vector
I(const vector &v)
{
#ifdef DFT_PROFILING
  timerOn(34);   // Turn on I timer
#endif // DFT_PROFILING

  vector Iv(v);
  register int i;
  register real s;
  Basis *basis;

  basis = v.basis;
  if (basis == (Basis *)0)
    die("I(vector) called with vector.basis == 0!\n");
  if (v.n != basis->NxNyNz)
    die("I(vector) called with vector.n != NxNyNz == size of FFT box\n");
#if defined SCALAR_IS_COMPLEX
  /* FFT with exp(+iG*r) */
  FFT3D(basis->Nx,basis->Ny,basis->Nz,Iv.c,1);
  /* Get rid of FFT scaling and divide by sqrt(Vol) */
  s = (real)1.0/sqrt(basis->unit_cell_volume);
  for (i=0; i < Iv.n; i++)
    Iv.c[i] *= s;
  
#elif defined SCALAR_IS_REAL
  die("I don't know how to do I(vector) for scalar being real!\n");
#else
#error scalar is neither real nor complex!
#endif

#ifdef DFT_PROFILING
  timerOff(34);   // Turn off I timer
#endif // DFT_PROFILING
  return Iv;
}

/* The I operator on column_bundles:  this takes from G to r space,
 * so the sizes of the basis sets are different! It actually calls
 * apply_I() below for the work. */
column_bundle
I(const column_bundle &Y)
{
  if (Y.basis == (Basis *)0)
    die("I(column_undle) called with column_bundle->basis == 0!\n");
  if (Y.col_length != Y.basis->nbasis)
    die("I(Y) called with Y.col_length != Y->basis->nbasis\n");

  /* Allocate the correct size for IY and copy everything else from Y */
  column_bundle IY(Y.tot_ncols,Y.basis->NxNyNz);
  copy_innards_column_bundle(&Y,&IY);

  /* do the work */
  apply_I(Y,IY);

  return IY;
}

/* Apply I() to Y and puts the result in IY; uses no extra memory. */
/* The I operator on column_bundles:  this takes from G to r space,
 * so the sizes of the basis sets are different! */
void
apply_I(const column_bundle &Y,column_bundle &IY)
{
#ifdef DFT_PROFILING
  timerOn(34);   // Turn on I timer
#endif // DFT_PROFILING

  register int i,j,Nx,Ny,Nz;
  register real s;
  Basis *basis;
  register int *index;

  basis = Y.basis;
  if (basis == (Basis *)0)
    die("apply_I(Y,IY) called with Y->basis == 0!\n");
  if (Y.col_length != basis->nbasis)
    die("apply_I(Y,IY) called with Y.col_length != Y->basis->nbasis\n");
  if (Y.tot_ncols != IY.tot_ncols)
    die("apply_I(Y,IY) called with Y.tot_ncols != IY.tot_ncols\n");
  if (IY.col_length != basis->NxNyNz)
    die("apply_I(Y,IY) called with IY.col_length != basis->NxNyNz\n");

#if defined SCALAR_IS_COMPLEX

  /* Now, do I to each column of Y */
  Nx = basis->Nx; Ny = basis->Ny; Nz = basis->Nz;
  s = (real)1.0/sqrt(basis->unit_cell_volume);
  index = basis->index;
  for (i=0; i < IY.my_ncols; i++)
    {
      /* Fill in part of G-space taken up by the basis with correct values */
      for (j=0; j < basis->NxNyNz; j++)
	IY.col[i].c[j] = (real)0.0;
      for (j=0; j < basis->nbasis; j++)
	IY.col[i].c[index[j]] = Y.col[i].c[j];

      /* FFT column with exp(+iG*r) */
      FFT3D(Nx,Ny,Nz,IY.col[i].c,1);
      /* Get rid of FFT scaling and divide by sqrt(Vol) */
      for (j=0; j < IY.col[i].n; j++)
	IY.col[i].c[j] *= s;
    }

#elif defined SCALAR_IS_REAL
  die("I don't know how to do I(Y) for scalar being real!\n");
#else
#error scalar is neither real nor complex!
#endif

#ifdef DFT_PROFILING
  timerOff(34);   // Turn off I timer
#endif // DFT_PROFILING

}

/* The J operator on vectors:  J = inv(I) for our case of plane waves.
 * Both v and J*v are the size of the FFT box.  It is actually a wrapper
 * for the routine below it which does all the work. */
vector
J(const vector &v)
{
  vector Jv(v);

  if (v.basis == (Basis *)0)
    die("J(vector) called with vector.basis == 0!\n");
  if (v.n != v.basis->NxNyNz)
    die("J(vector) called with vector.n != NxNyNz == size of FFT box\n");

  apply_J(v,Jv);

  return Jv;
}

/* Does J on a vector, but uses no exra memory except what it is given. */
void
apply_J(const vector &v,vector &Jv)
{
#ifdef DFT_PROFILING
  timerOn(35);   // Turn on J timer
#endif // DFT_PROFILING

  register int i;
  register real s;
  Basis *basis;

  basis = v.basis;
  if (basis == (Basis *)0)
    die("apply_J(vector) called with vector.basis == 0!\n");
  if (v.n != basis->NxNyNz)
    die("apply_J(vector) called with vector.n != NxNyNz == size of FFT box\n");
  if (v.n != Jv.n)
    die("apply_J called with v.n != Jv.n\n");
  /* Copy v onto Jv since the FFT is done in place! */
  Jv = v;
#if defined SCALAR_IS_COMPLEX
  /* FFT with exp(+iG*r) */
  FFT3D(basis->Nx,basis->Ny,basis->Nz,Jv.c,-1);
  /* Scale by sqrt(Vol)/(Nx*Ny*Nz) */
  s = sqrt(basis->unit_cell_volume);
  for (i=0; i < Jv.n; i++)
    Jv.c[i] *= s;
  
#elif defined SCALAR_IS_REAL
  die("I don't know how to do apply_J(vector) for scalar being real!\n");
#else
#error scalar is neither real nor complex!
#endif

#ifdef DFT_PROFILING
  timerOff(35);   // Turn off J timer
#endif // DFT_PROFILING

}


/* The J operator on column_bundles:  this takes from r to G space,
 * so the sizes of the basis sets are different! */
column_bundle
J(const column_bundle &Y)
{
  if (Y.basis == (Basis *)0)
    die("J(Y) called with Y.basis == 0!\n");
  if (Y.col_length != Y.basis->NxNyNz)
    die("J(Y) called with Y.col_length != NxNyNz == size of FFT box\n");

  /* Allocate the correct size for JY and copy everything else from Y */
  column_bundle JY(Y.tot_ncols,Y.basis->nbasis);
  copy_innards_column_bundle(&Y,&JY);  

  /* do the work */
  apply_J(Y,JY);

  return JY;
}

/* The J operator on column_bundles:  this takes from r to G space,
 * so the sizes of the basis sets are different! */
void
apply_J(const column_bundle &Y,column_bundle &JY)
{
#ifdef DFT_PROFILING
  timerOn(35);   // Turn on J timer
#endif // DFT_PROFILING

  register int i,j,Nx,Ny,Nz;
  real s;
  Basis *basis;

  basis = Y.basis;
  if (basis == (Basis *)0)
    die("apply_J(Y,JY) called with Y.basis == 0!\n");
  if (Y.col_length != basis->NxNyNz)
    die("apply_J(Y) called with Y.col_length != NxNyNz == FFT box size\n");
  if (Y.tot_ncols != JY.tot_ncols)
    die("apply_J(Y,JY) Y.tot_ncols != JY.tot_ncols\n");
  if (JY.col_length != basis->nbasis)
    die("apply_J(Y,JY) called with JY.col_length != basis->nbasis\n");

#if defined SCALAR_IS_COMPLEX

  /* Now, do J to each column of Y: this is temporary workspace */
  register complex *fft_col;
  register int *index = basis->index;

  Nx = basis->Nx; Ny = basis->Ny; Nz = basis->Nz;
  s = sqrt(basis->unit_cell_volume);
  fft_col = (complex *)mymalloc(sizeof(complex)*basis->NxNyNz,
				"fft_col","J(column_bundle)");
  for (i=0; i < JY.my_ncols; i++)
    {
      /* Copy Y column to temporary fft workspace */
      for (j=0; j < basis->NxNyNz; j++)
	fft_col[j] = Y.col[i].c[j];
      /* Do FFT with exp(-iG*r) */
      FFT3D(Nx,Ny,Nz,fft_col,-1);
      /* Copy the G-space vector coefs. we need and scale them */
      for (j=0; j < basis->nbasis; j++)
	JY.col[i].c[j] = s*fft_col[index[j]];
    }
  myfree(fft_col);

#elif defined SCALAR_IS_REAL
  die("I don't know how to do I(vector) for scalar being real!\n");
#else
#error scalar is neither real nor complex!
#endif

#ifdef DFT_PROFILING
  timerOff(35);   // Turn off J timer
#endif // DFT_PROFILING

}

/* We use Idag = (Nx*Ny*Nz)/Vol*J */
vector
Idag(const vector &v)
{
#ifdef DFT_PROFILING
  timerOn(36);   // Turn on Idag timer
#endif // DFT_PROFILING
  vector Idagv(v);
  register real s;
  register int i;

  if (v.n != v.basis->NxNyNz)
    die("Idag(vector) called with vector.n != NxNyNz == size of FFT box\n");
  Idagv = J(v);
  s = (real)v.basis->NxNyNz/v.basis->unit_cell_volume;
  for (i=0; i < Idagv.n; i++)
    Idagv.c[i] *= s;

#ifdef DFT_PROFILING
  timerOff(36);   // Turn off Idag timer
#endif // DFT_PROFILING
  return Idagv;
}

/* We use Idag = (Nx*Ny*Nz)/Vol*J:  calls apply_Idag() for the work */
column_bundle
Idag(const column_bundle &Y)
{
  if (Y.basis == 0)
    die("Idag(Y) called with Y.basis == 0\n");
  if (Y.col_length != Y.basis->NxNyNz)
    die("Idag(Y) called with Y.col_length != NxNyNz == size of FFT box\n");

  /* space for the result */
  column_bundle IdagY(Y.tot_ncols,Y.basis->nbasis);

  /* do the work */
  apply_Idag(Y,IdagY);

  return IdagY;
}

/* applies Idag=(Nx*Ny*Nz)/Vol*J to Y and puts it into IdagY */
void
apply_Idag(const column_bundle &Y,column_bundle &IdagY)
{
#ifdef DFT_PROFILING
  timerOn(36);   // Turn on Idag timer
#endif // DFT_PROFILING
  register real s;
  register int i,j;

  if (Y.basis == 0)
    die("apply_Idag(Y) called with Y.basis == 0\n");
  if (Y.col_length != Y.basis->NxNyNz)
    die("apply_Idag(Y) called with Y.col_length != NxNyNz\n");
  if (IdagY.tot_ncols != Y.tot_ncols)
    die("apply_Idag(Y,IdagY) called with Y.tot_ncols != IdagY.tot_ncols\n");
  if (IdagY.col_length != Y.basis->nbasis)
    die("apply_Idag(Y,IdagY) called with IdagY.col_length != nbasis\n");

  apply_J(Y,IdagY); /* does:  IdagY = J(Y); */

  s = (real)Y.basis->NxNyNz/Y.basis->unit_cell_volume;
  for (i=0; i < IdagY.my_ncols; i++)
    for (j=0; j < IdagY.col_length; j++)
      IdagY.col[i].c[j] *= s;

#ifdef DFT_PROFILING
  timerOff(36);   // Turn off Idag timer
#endif // DFT_PROFILING
}

/* Idag = Nx*Ny*Nz*FFT(-)/sqrt(Vol) = (Nx*Ny*Nz)*J/Vol.
 * This routine applies Idag to Y and places the result into IdagY, but
 * in the process destroys the contents of Y (i.e. it does FFT in place
 * on Y). */
void
apply_Idag_inplace(column_bundle &Y,column_bundle &IdagY)
{
#ifdef DFT_PROFILING
  timerOn(36);   // Turn on Idag timer
#endif // DFT_PROFILING
  register int i,j,Nx,Ny,Nz;
  real s;
  Basis *basis;
  register int *index;

  basis = Y.basis;
  if (basis == (Basis *)0)
    die("apply_Idag_inplace(Y,IdagY) called with Y.basis == 0!\n");
  if (Y.col_length != basis->NxNyNz)
    die("apply_Idag_inplace(Y) called with Y.col_length != FFT box size\n");
  if (Y.tot_ncols != IdagY.tot_ncols)
    die("apply_Idag_inplaceJ(Y,IdagY) Y.tot_ncols != IdagY.tot_ncols\n");
  if (IdagY.col_length != basis->nbasis)
    die("apply_Idag_inplace(Y,IdagY) called with IdagY.col_length != basis->nbasis\n");

#if defined SCALAR_IS_COMPLEX
  Nx = basis->Nx; Ny = basis->Ny; Nz = basis->Nz;
  s = (real)(Nx*Ny*Nz)/sqrt(basis->unit_cell_volume);
  index = basis->index;
  for (i=0; i < IdagY.my_ncols; i++)
    {
      /* Do FFT with exp(-iG*r) IN PLACE on Y */
      FFT3D(Nx,Ny,Nz,&(Y.col[i].c[0]),-1);
      /* Copy the G-space vector coefs. we need and scale them */
      for (j=0; j < basis->nbasis; j++)
	IdagY.col[i].c[j] = s*Y.col[i].c[index[j]];
    }
#elif defined SCALAR_IS_REAL
  die("I don't know how to do Idag for scalar being real!\n");
#else
#error scalar is neither real nor complex!
#endif

#ifdef DFT_PROFILING
  timerOff(36);   // Turn off Idag timer
#endif // DFT_PROFILING
}

/* We use Jdag = Vol/(Nx*Ny*Nz)*I */
vector
Jdag(const vector &v)
{
#ifdef DFT_PROFILING
  timerOn(37);   // Turn on Jdag timer
#endif // DFT_PROFILING
  vector Jdagv(v);
  register real s;
  register int i;

  if (v.n != v.basis->NxNyNz)
    die("Jdag(vector) called with vector.n != NxNyNz == size of FFT box\n");
  Jdagv = I(v);
  s = v.basis->unit_cell_volume/(real)v.basis->NxNyNz;
  for (i=0; i < Jdagv.n; i++)
    Jdagv.c[i] *= s;

#ifdef DFT_PROFILING
  timerOff(37);   // Turn off Jdag timer
#endif // DFT_PROFILING
  return Jdagv;
}

/* We use Jdag = Vol/(Nx*Ny*Nz)*I */
column_bundle
Jdag(const column_bundle &Y)
{
  if (Y.basis == 0)
    die("Jdag(Y) called with Y.basis == 0\n");
  if (Y.col_length != Y.basis->nbasis)
    die("Jdag(Y) called with Y.col_length != Y.basis->nbasis\n");

  /* where to stick the result */
  column_bundle JdagY(Y.tot_ncols,Y.basis->NxNyNz);

  /* do the work */
  apply_Jdag(Y,JdagY);

  return JdagY;
}

/* We use Jdag = Vol/(Nx*Ny*Nz)*I */
void
apply_Jdag(const column_bundle &Y,column_bundle &JdagY)
{
#ifdef DFT_PROFILING
  timerOn(37);   // Turn on Jdag timer
#endif // DFT_PROFILING

  register real s;
  register int i,j;

  if (Y.basis == 0)
    die("apply_Jdag(Y) called with Y.basis == 0\n");
  if (Y.col_length != Y.basis->nbasis)
    die("apply_Jdag(Y) called with Y.col_length != Y.basis->nbasis\n");
  if (Y.tot_ncols != JdagY.tot_ncols)
    die("apply_Jdag(Y,JdagY) called with Y.tot_ncols != JdagY.tot_ncols\n");
  if (JdagY.col_length != Y.basis->NxNyNz)
    die("apply_Jdag(Y,JdagY) called with JdagY.col_length != basis->NxNyNz\n");

  apply_I(Y,JdagY); /* does  JdagY = I(Y); */

  s = Y.basis->unit_cell_volume/(real)Y.basis->NxNyNz;
  for (i=0; i < JdagY.my_ncols; i++)
    for (j=0; j < JdagY.col_length; j++)
      JdagY.col[i].c[j] *= s;

#ifdef DFT_PROFILING
  timerOff(37);   // Turn off Jdag timer
#endif // DFT_PROFILING
}


/*================================================================*
 *                                                                *
 * Some short-hand routines to take advantage of the              *
 * specifics of the planewave basis-set                           *
 *                                                                *
 *================================================================*/

vector
Jdag_O_J(const vector &v)
{
  // in plane wave basis, Jdag_O_J = w * Identity
  // where   w = 
  vector Jdag_O_Jv(v);
  real s = v.basis->unit_cell_volume/(real)v.basis->NxNyNz;
  for (int i=0; i < v.n; i++) 
    Jdag_O_Jv.c[i] *= s;
  
  return Jdag_O_Jv;
}
